/*
 *  Copyright 2009 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.odf.orgunit;

import java.util.HashSet;
import java.util.List;
import java.util.Set;

import org.apache.avalon.framework.component.Component;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;

import org.ametys.core.ui.Callable;
import org.ametys.odf.workflow.CreateOrgUnitFunction;
import org.ametys.plugins.repository.AmetysObjectIterable;
import org.ametys.plugins.repository.AmetysObjectIterator;
import org.ametys.plugins.repository.AmetysObjectResolver;
import org.ametys.plugins.repository.UnknownAmetysObjectException;
import org.ametys.runtime.plugin.component.AbstractLogEnabled;

/**
 * Component to provide the root OrgUnit
 */
public class RootOrgUnitProvider extends AbstractLogEnabled implements Component, Serviceable
{
    /** Avalon role. */
    public static final String ROLE = RootOrgUnitProvider.class.getName();
    
    private AmetysObjectResolver _resolver;
    
    @Override
    public void service(ServiceManager smanager) throws ServiceException
    {
        _resolver = (AmetysObjectResolver) smanager.lookup(AmetysObjectResolver.ROLE);
    }
    
    /**
     * Returns the root {@link OrgUnit}.
     * @return the root {@link OrgUnit}.
     */
    public OrgUnit getRoot()
    {
        AmetysObjectIterable<OrgUnit> ouIterable = _resolver.query("//element(" + CreateOrgUnitFunction.CONTENT_NAME_PREFIX + OrgUnitFactory.ODF_ORGUNIT_ROOT_NODE + ", " + OrgUnitFactory.ORGUNIT_NODETYPE + ")");
        AmetysObjectIterator<OrgUnit> ouIterator = ouIterable.iterator();
        
        if (ouIterator.hasNext())
        {
            return ouIterator.next();
        }

        return null;
    }
    
    /**
     * Get the root id
     * @return the root id
     */
    @Callable
    public String getRootId()
    {
        OrgUnit root = getRoot();
        return root != null ? getRoot().getId() : null;
    }
    
    /**
     * Determines the given id is the root orgunit id
     * @param id the id to test
     * @return true if is root
     */
    public boolean isRoot (String id)
    {
        return id.equals(getRootId());
    }
    
    /**
     * Determines the given orgunit is the root orgunit
     * @param orgUnit the orgunit to test
     * @return true if is root
     */
    public boolean isRoot (OrgUnit orgUnit)
    {
        return orgUnit.getId().equals(getRootId());
    }
    
    /**
     * Get the child orgunits of an orgunit
     * @param orgUnitId The identifier of the main orgunit
     * @param recursive true to children recursively
     * @return A set containing all the child orgunit identifiers.
     */
    public Set<String> getChildOrgUnitIds(String orgUnitId, boolean recursive)
    {
        Set<String> childIds = new HashSet<>();
        
        try
        {
            OrgUnit orgUnit = _resolver.resolveById(orgUnitId);
            List<String> childOrgUnitIds = orgUnit.getSubOrgUnits();
            childIds.addAll(childOrgUnitIds);
            
            if (recursive)
            {
                for (String childOrgUnitD : orgUnit.getSubOrgUnits())
                {
                    childIds.addAll(getChildOrgUnitIds(childOrgUnitD, recursive));
                }
            }
        }
        catch (UnknownAmetysObjectException e)
        {
            // log and ignore
            getLogger().warn("Encountered a reference to an unknown orgunit : {}", orgUnitId, e);
        }
        
        return childIds;
    }
}
