/*
 *  Copyright 2010 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.odf.person;

import javax.jcr.Node;

import org.apache.commons.lang3.ArrayUtils;
import org.apache.commons.lang3.StringUtils;

import org.ametys.cms.data.RichText;
import org.ametys.cms.repository.ModifiableDefaultContent;
import org.ametys.odf.cdmfr.CDMEntity;

/**
 * Class representing a {@link Person}
 */
public class Person extends ModifiableDefaultContent<PersonFactory> implements CDMEntity
{
    /** ldap uid */
    public static final String LDAP_UID = "ldapUid";
    
    /** Mandatory Identifier to generate the CDM-fr id */
    public static final String LOGIN = "login";

    /** Name attribute. */
    public static final String NAME = "lastName";

    /** Given name attribute. */
    public static final String GIVEN_NAME = "givenName";
    
    /** Academic title attribute. */
    public static final String PERSON_TITLE = "personTitle";

    /** ROLE LABEL attribute. */
    public static final String ROLE = "role";

    /** Additionnal informations attribute. */
    public static final String ADDITIONAL_INFORMATIONS = "additionalInformations";

    /**
     * Constructor
     * @param node The JCR node
     * @param parentPath The parent path
     * @param factory the factory
     */
    public Person(Node node, String parentPath, PersonFactory factory)
    {
        super(node, parentPath, factory);
    }

    // --------------------------------------------------------------------------------------//
    //
    // GETTERS AND SETTERS
    //
    // --------------------------------------------------------------------------------------//

    /**
     * Get the login of the person
     * This identifier will be the CDM-fr id of the person
     * @return the login or null
     */
    public String getLogin()
    {
        return getValue(Person.LOGIN, false, StringUtils.EMPTY);
    }
    
    /**
     * Get the lastname of the person
     * @return the Name or null
     */
    public String getLastName()
    {
        return getValue(NAME, false, StringUtils.EMPTY);
    }
    
    /**
     * Get the given name of the person
     * @return the given name or null
     */
    public String getGivenName()
    {
        return getValue(GIVEN_NAME, false, StringUtils.EMPTY);
    }

    /**
     * Get the contact data informations
     * @return ContactData
     */
    public ContactData getContactData()
    {
        ContactData contact = new ContactData();
        contact.setAddress(getValue(ContactData.ADDRESS, false, StringUtils.EMPTY));
        contact.setAdditionalAddress(getValue(ContactData.ADDITIONAL_ADDRESS, false, StringUtils.EMPTY));
        contact.setZipCode(getValue(ContactData.ZIP_CODE, false, StringUtils.EMPTY));
        contact.setTown(getValue(ContactData.TOWN, false, StringUtils.EMPTY));
        contact.setPhone(getValue(ContactData.PHONE, false, StringUtils.EMPTY));
        contact.setFax(getValue(ContactData.FAX, false, StringUtils.EMPTY));
        contact.setMail(getValue(ContactData.MAIL, false, StringUtils.EMPTY));
        contact.setWebLinkLabel(getValue(ContactData.WEB_LINK_LABEL, false, StringUtils.EMPTY));
        contact.setWebLinkUrl(getValue(ContactData.WEB_LINK_URL, false, StringUtils.EMPTY));
        return contact;
    }

    /**
     * Get the title posessed by the person
     * @return the person title
     */
    public String getPersonTitle()
    {
        return getValue(PERSON_TITLE, false, StringUtils.EMPTY);
    }

    /**
     * Get the role informations, a person may have different roles in different contexts, e.g. lecturer.
     * @return role
     */
    public String[] getRole()
    {
        return getValue(ROLE, false, ArrayUtils.EMPTY_STRING_ARRAY);
    }
    
    
    /**
     * Return the additionnals informations
     * @return the description
     */
    public RichText getAdditionalInformations()
    {
        return getValue(Person.ADDITIONAL_INFORMATIONS);
    }
    
    // --------------------------------------------------------------------------------------//
    // CDM-fr
    // --------------------------------------------------------------------------------------//
    @Override
    public String getCDMId() 
    {
        String cdmCode = getCdmCode();
        if (StringUtils.isEmpty(cdmCode))
        {
            String login = getLogin();
            return "FRUAI" + _getFactory()._getRootOrgUnitRNE() + "PE" + (StringUtils.isNotEmpty(login) ? login.toUpperCase() : getName().toUpperCase());
        }
        return cdmCode;
    }
    
    @Override
    public String getCdmCode()
    {
        return getValue(CDM_CODE, false, StringUtils.EMPTY);
    }
    
    @Override
    public void setCdmCode(String cdmCode) 
    {
        setValue(CDM_CODE, cdmCode);
    }
}
