/*
 *  Copyright 2010 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.odf.program;

import java.util.Arrays;
import java.util.List;
import java.util.Optional;
import java.util.stream.Collectors;

import javax.jcr.Node;

import org.apache.commons.lang3.ArrayUtils;

import org.ametys.cms.data.ContentDataHelper;
import org.ametys.cms.data.ContentValue;
import org.ametys.cms.data.RichText;
import org.ametys.odf.courselist.CourseList;
import org.ametys.odf.courselist.CourseListContainer;
import org.ametys.plugins.repository.AmetysRepositoryException;

/**
 * Container java object
 */
public class Container extends AbstractTraversableProgramPart<ContainerFactory> implements CourseListContainer
{

    /** Constants for ects attribute */
    public static final String ECTS = "ects";

    /** Constants for description attribute */
    public static final String DESCRIPTION = "description";

    /** Constants for period attribute */
    public static final String PERIOD = "period";

    /** Constants for nature attribute* */
    public static final String NATURE = "nature";
    
    /**
     * Constructor
     * @param node The JCR node
     * @param parentPath The parent path
     * @param factory the ametys object factory
     */
    public Container(Node node, String parentPath, ContainerFactory factory)
    {
        super(node, parentPath, factory);
    }
    
    // --------------------------------------------------------------------------------------//
    // GETTERS AND SETTERS
    // --------------------------------------------------------------------------------------//
    /**
     * Get the ECTS credits
     * @return the ECTS credits
     * @throws AmetysRepositoryException if an error occurred
     */
    public double getEcts() throws AmetysRepositoryException
    {
        return getValue(ECTS, false, 0D);
    }

    /**
     * Get the description
     * @return the description or null if not set
     */
    public RichText getDescription()
    {
        return getValue(DESCRIPTION);
    }
    
    /**
     * Get the period
     * @return the period
     * @throws AmetysRepositoryException if an error occurs
     */
    public String getPeriod() throws AmetysRepositoryException
    {
        return ContentDataHelper.getContentIdFromContentData(this, PERIOD);
    }

    /**
     * Get the nature
     * @return the nature
     * @throws AmetysRepositoryException if an error occurs
     */
    public String getNature() throws AmetysRepositoryException
    {
        return ContentDataHelper.getContentIdFromContentData(this, NATURE);
    }
    
    // --------------------------------------------------------------------------------------//
    // Methods of CourseListContainer
    // --------------------------------------------------------------------------------------//
    @Override
    public List<CourseList> getCourseLists()
    {
        return Arrays.stream(getValue(CHILD_PROGRAM_PARTS, false, new ContentValue[0]))
                .map(ContentValue::getContentIfExists)
                .flatMap(Optional::stream)
                // This cast is not checked because an exception must be thrown if the retrieved content is not a program part
                // TODO: change this behavior to throw our own exception and not a CassCastException
                .map(ProgramPart.class::cast)
                // Program parts that are not course lists are simply ignored
                .filter(CourseList.class::isInstance)
                .map(CourseList.class::cast)
                .collect(Collectors.toList());
    }

    @Override
    public boolean containsCourseList(String clId)
    {
        return ArrayUtils.contains(ContentDataHelper.getContentIdsArrayFromMultipleContentData(this, CHILD_PROGRAM_PARTS), clId);
    }

    @Override
    public boolean hasCourseLists()
    {
        return !getCourseLists().isEmpty();
    }
    
    // --------------------------------------------------------------------------------------//
    // CDMfr
    // --------------------------------------------------------------------------------------//
    @Override
    protected String getCDMType()
    {
        return "PR";
    }
}
