/*
 *  Copyright 2022 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.odf.schedulable;

import java.io.IOException;
import java.util.HashMap;
import java.util.Map;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.exception.ExceptionUtils;
import org.quartz.JobDataMap;
import org.quartz.JobExecutionContext;
import org.quartz.JobExecutionException;

import org.ametys.cms.schedule.AbstractSendingMailSchedulable;
import org.ametys.core.schedule.progression.ContainerProgressionTracker;
import org.ametys.core.ui.mail.StandardMailBodyHelper;
import org.ametys.odf.catalog.Catalog;
import org.ametys.odf.catalog.CatalogsManager;
import org.ametys.plugins.core.schedule.Scheduler;
import org.ametys.runtime.i18n.I18nizableText;
import org.ametys.runtime.i18n.I18nizableTextParameter;

/**
 * Schedulable to delete a catalog
 */
public class DeleteCatalogSchedulable extends AbstractSendingMailSchedulable
{
    /** Schedulable ID */
    public static final String SCHEDULABLE_ID = DeleteCatalogSchedulable.class.getName();
    
    /** The job data map key for the catalog Id to delete*/
    public static final String JOBDATAMAP_CATALOG_NAME_KEY = "catalog";
    
    private CatalogsManager _catalogsManager;

    @Override
    public void service(ServiceManager manager) throws ServiceException
    {
        super.service(manager);
        _catalogsManager = (CatalogsManager) manager.lookup(CatalogsManager.ROLE);
    }
    
    @Override
    protected void _doExecute(JobExecutionContext context, ContainerProgressionTracker progressionTracker) throws Exception
    {
        JobDataMap jobDataMap = context.getJobDetail().getJobDataMap();
        
        String catalogName = jobDataMap.getString(Scheduler.PARAM_VALUES_PREFIX + JOBDATAMAP_CATALOG_NAME_KEY);
        if (StringUtils.isEmpty(catalogName))
        {
            throw new JobExecutionException("The catalog name cannot be empty.");
        }
        Catalog catalog = _catalogsManager.getCatalog(catalogName);
        if (catalog == null)
        {
            throw new JobExecutionException("The catalog " + catalogName + " can't be retrieved.");
        }
        
        Map<String, Object> deleteCatalogResult = _catalogsManager.deleteCatalog(catalog);
        if (deleteCatalogResult.containsKey("error"))
        {
            switch ((String) deleteCatalogResult.get("error"))
            {
                case "unknown-catalog":
                    throw new JobExecutionException("The catalog with id " + catalogName + " can not be resolved.");
                case "referencing-contents":
                    throw new JobExecutionException("Impossible to delete catalog with id " + catalogName + ". There are contents outside of the catalog referencing the contents in the catalog to delete. See previous log for more info.");
                default:
                    throw new JobExecutionException("Something bad happened while trying to delete the catalog " + catalogName);
            }
        }
    }

    @Override
    protected I18nizableText _getSuccessMailSubject(JobExecutionContext context) throws Exception
    {
        return new I18nizableText("plugin.odf", "PLUGINS_ODF_SCHEDULABLE_DELETE_CATALOG_SUCCESS_MAIL_SUBJECT", _getI18nParams(context));
    }
    
    @Override
    protected boolean _isMailBodyInHTML(JobExecutionContext context) throws Exception
    {
        return true;
    }

    @Override
    protected String _getSuccessMailBody(JobExecutionContext context) throws Exception
    {
        try
        {
            return StandardMailBodyHelper.newHTMLBody()
                    .withTitle(_getSuccessMailSubject(context))
                    .withMessage(new I18nizableText("plugin.odf", "PLUGINS_ODF_SCHEDULABLE_DELETE_CATALOG_SUCCESS_MAIL_BODY", _getI18nParams(context)))
                    .build();
        }
        catch (IOException e)
        {
            getLogger().warn("Failed to build HTML email body for catalog deletion result. Fallback to no wrapped email", e);
            return _i18nUtils.translate(new I18nizableText("plugin.odf", "PLUGINS_ODF_SCHEDULABLE_DELETE_CATALOG_SUCCESS_MAIL_BODY", _getI18nParams(context)));
        }
    }

    @Override
    protected I18nizableText _getErrorMailSubject(JobExecutionContext context) throws Exception
    {
        return new I18nizableText("plugin.odf", "PLUGINS_ODF_SCHEDULABLE_DELETE_CATALOG_ERROR_MAIL_SUBJECT", _getI18nParams(context));
    }

    @Override
    protected String _getErrorMailBody(JobExecutionContext context, Throwable throwable) throws Exception
    {
        String error = ExceptionUtils.getStackTrace(throwable);
        
        try
        {
            return StandardMailBodyHelper.newHTMLBody()
                    .withTitle(_getErrorMailSubject(context))
                    .withMessage(new I18nizableText("plugin.odf", "PLUGINS_ODF_SCHEDULABLE_DELETE_CATALOG_ERROR_MAIL_BODY", _getI18nParams(context)))
                    .withDetails(null, error, true)
                    .build();
        }
        catch (IOException e)
        {
            getLogger().warn("Failed to build HTML email body for catalog deletion result. Fallback to no wrapped email", e);
            return _i18nUtils.translate(new I18nizableText("plugin.odf", "PLUGINS_ODF_SCHEDULABLE_DELETE_CATALOG_ERROR_MAIL_BODY", _getI18nParams(context)));
        }
    }

    private Map<String, I18nizableTextParameter> _getI18nParams(JobExecutionContext context)
    {
        Map<String, I18nizableTextParameter> i18nParams = new HashMap<>();
        
        JobDataMap jobDataMap = context.getJobDetail().getJobDataMap();
        String catalogName = jobDataMap.getString(Scheduler.PARAM_VALUES_PREFIX + JOBDATAMAP_CATALOG_NAME_KEY);
        i18nParams.put("catalogName", new I18nizableText(catalogName));
        return i18nParams;
    }
}

