/*
 *  Copyright 2019 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.odf.schedulable;

import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.apache.avalon.framework.component.Component;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;
import org.apache.excalibur.xml.dom.DOMParser;
import org.apache.excalibur.xml.xpath.XPathProcessor;
import org.quartz.JobKey;
import org.quartz.SchedulerException;
import org.w3c.dom.Document;
import org.w3c.dom.NamedNodeMap;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;
import org.xml.sax.InputSource;
import org.xml.sax.SAXException;

import org.ametys.core.schedule.Runnable;
import org.ametys.core.schedule.Runnable.FireProcess;
import org.ametys.core.schedule.Runnable.MisfirePolicy;
import org.ametys.core.ui.Callable;
import org.ametys.core.user.CurrentUserProvider;
import org.ametys.plugins.core.impl.schedule.DefaultRunnable;
import org.ametys.plugins.core.schedule.Scheduler;
import org.ametys.runtime.i18n.I18nizableText;
import org.ametys.runtime.plugin.component.AbstractLogEnabled;
import org.ametys.runtime.util.AmetysHomeHelper;

/**
 * The component for scheduling a global validation report on programs
 */
public class GlobalValidationReport extends AbstractLogEnabled implements Component, Serviceable
{
    /** The Avalon role */
    public static final String ROLE = GlobalValidationReport.class.getName();
    
    private static final String __SCHEDULABLE_ID = "org.ametys.odf.schedulable.GlobalValidationSchedulable";
    private static final String __JOB_KEY = __SCHEDULABLE_ID + "$task";
    
    private Scheduler _scheduler;
    private CurrentUserProvider _userProvider;

    private XPathProcessor _xPathProcessor;

    private DOMParser _domParser;
    
    public void service(ServiceManager smanager) throws ServiceException
    {
        _scheduler = (Scheduler) smanager.lookup(Scheduler.ROLE);
        _userProvider = (CurrentUserProvider) smanager.lookup(CurrentUserProvider.ROLE);
        _domParser = (DOMParser) smanager.lookup(DOMParser.ROLE);
        _xPathProcessor = (XPathProcessor) smanager.lookup(XPathProcessor.ROLE);
    }
    
    /**
     * Determines if the scheduler is currently running
     * @return true if a check is already running
     */
    @Callable
    public boolean isRunning()
    {
        return Boolean.TRUE.equals(_scheduler.isRunning(__JOB_KEY).get("running"));
    }
    
    /**
     * Start the global validation report on programs
     * @return the result
     */
    @Callable
    public Map<String, Object> startReport()
    {
        Map<String, Object> result = new HashMap<>();

        try
        {
            Runnable runnable = new DefaultRunnable(__JOB_KEY, 
                    new I18nizableText("plugin.odf", "PLUGINS_ODF_SCHEDULABLE_GLOBAL_VALIDATION_REPORT_LABEL"), 
                    new I18nizableText("plugin.odf", "PLUGINS_ODF_SCHEDULABLE_GLOBAL_VALIDATION_REPORT_DESCRIPTION"), 
                    FireProcess.NOW, 
                    null /* cron*/, 
                    __SCHEDULABLE_ID, 
                    false /* removable */, 
                    false /* modifiable */, 
                    false /* deactivatable */, 
                    MisfirePolicy.FIRE_ONCE, 
                    false /* isVolatile */, 
                    _userProvider.getUser(), 
                    Collections.EMPTY_MAP);
            
            if (isRunning())
            {
                getLogger().warn("A global validation check is already running");
                result.put("error", "already-running");
                result.put("success", false);
            }
            else
            {
                JobKey jobKey = new JobKey(runnable.getId(), Scheduler.JOB_GROUP);
                
                if (_scheduler.getScheduler().checkExists(jobKey))
                {
                    _scheduler.getScheduler().deleteJob(jobKey);
                }
                _scheduler.scheduleJob(runnable);
                result.put("success", true);
            }
            
            return result;
        }
        catch (SchedulerException e)
        {
            getLogger().error("An error occured when trying to schedule the global validation status on programs", e);
            result.put("success", false);
            return result;
        }
    }
    
    /**
     * Get the list of invalidated contents computed by the last report for a given content
     * @param contentId the content id
     * @return the list of invalidated contents
     */
    @Callable
    public List<Map<String, Object>> getInvalidatedContents(String contentId)
    {
        List<Map<String, Object>> invalidatedContents = new ArrayList<>();
        
        File reportFile = new File(AmetysHomeHelper.getAmetysHomeData(), "odf/report/global-validation.xml");
        
        if (reportFile.exists())
        {
            try (InputStream is = new FileInputStream(reportFile))
            {
                Document doc = _domParser.parseDocument(new InputSource(is));
                
                NodeList nodeList = _xPathProcessor.selectNodeList(doc, "/programs/program[@id='" + contentId + "']/invalidatedContents/content");
                
                for (int i = 0; i < nodeList.getLength(); i++)
                {
                    Node node = nodeList.item(i);
                    invalidatedContents.add(_getAttributes(node));
                }
            }
            catch (IOException | SAXException e)
            {
                getLogger().error("Unable to get the list of invalidated contents computed by the last report for program '{}'", contentId, e);
            }
        }
        
        return invalidatedContents;
    }
    
    private Map<String, Object> _getAttributes(Node node)
    {
        Map<String, Object> map = new HashMap<>();
        
        NamedNodeMap attributes = node.getAttributes();
        for (int i = 0; i < attributes.getLength(); i++)
        {
            Node attribute = attributes.item(i);
            
            String name = attribute.getNodeName();
            String value = attribute.getNodeValue();
            
            map.put(name, value);
        }
        
        return map;
    }
}
