/*
 *  Copyright 2019 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.odf.schedulable;

import java.io.IOException;
import java.util.Date;
import java.util.Set;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.cocoon.ProcessingException;
import org.apache.cocoon.generation.ServiceableGenerator;
import org.apache.cocoon.xml.AttributesImpl;
import org.apache.cocoon.xml.XMLUtils;
import org.apache.commons.lang3.StringUtils;
import org.xml.sax.SAXException;

import org.ametys.cms.contenttype.ContentTypesHelper;
import org.ametys.cms.repository.Content;
import org.ametys.cms.repository.ContentQueryHelper;
import org.ametys.cms.repository.ContentTypeExpression;
import org.ametys.cms.repository.WorkflowAwareContent;
import org.ametys.core.user.UserIdentity;
import org.ametys.core.util.DateUtils;
import org.ametys.odf.ProgramItem;
import org.ametys.odf.catalog.Catalog;
import org.ametys.odf.catalog.CatalogsManager;
import org.ametys.odf.program.ProgramFactory;
import org.ametys.odf.workflow.ODFWorkflowHelper;
import org.ametys.plugins.core.user.UserHelper;
import org.ametys.plugins.repository.AmetysObjectIterable;
import org.ametys.plugins.repository.AmetysObjectResolver;
import org.ametys.plugins.repository.query.expression.Expression;
import org.ametys.plugins.repository.query.expression.Expression.Operator;

/**
 * Generate XML report for global validation on all programs.
 */
public class GlobalValidationReportGenerator extends ServiceableGenerator
{
    private ODFWorkflowHelper _odfWorkflowHelper;
    private AmetysObjectResolver _resolver;
    private ContentTypesHelper _cTypesHelper;
    private CatalogsManager _catalogManager;
    private UserHelper _userHelper;

    @Override
    public void service(ServiceManager smanager) throws ServiceException
    {
        super.service(smanager);
        _odfWorkflowHelper = (ODFWorkflowHelper) smanager.lookup(ODFWorkflowHelper.ROLE);
        _resolver = (AmetysObjectResolver) smanager.lookup(AmetysObjectResolver.ROLE);
        _cTypesHelper = (ContentTypesHelper) smanager.lookup(ContentTypesHelper.ROLE);
        _catalogManager = (CatalogsManager) smanager.lookup(CatalogsManager.ROLE);
        _userHelper = (UserHelper) smanager.lookup(UserHelper.ROLE);
    }
    
    public void generate() throws IOException, SAXException, ProcessingException
    {
        contentHandler.startDocument();
        
        AttributesImpl atts = new AttributesImpl();
        
        atts.addCDATAAttribute("date", DateUtils.dateToString(new Date()));
        XMLUtils.startElement(contentHandler, "programs", atts);
        
        try (AmetysObjectIterable<Content> programs = _getPrograms())
        {
            for (Content program : programs)
            {
                Set<Content> invalidatedContents = _odfWorkflowHelper.getInvalidatedContents((WorkflowAwareContent) program);
                if (!invalidatedContents.isEmpty())
                {
                    _saxGlobalValidationStatus(program, invalidatedContents);
                }
            }
        }
        
        XMLUtils.endElement(contentHandler, "programs");
        
        contentHandler.endDocument();

    }
    
    private AmetysObjectIterable<Content> _getPrograms()
    {
        Expression expression = new ContentTypeExpression(Operator.EQ, ProgramFactory.PROGRAM_CONTENT_TYPE);
        
        String query = ContentQueryHelper.getContentXPathQuery(expression);
        
        return _resolver.query(query);
    }
    
    private void _saxGlobalValidationStatus(Content content, Set<Content> invalidatedContents) throws SAXException
    {
        AttributesImpl atts = new AttributesImpl();
        atts.addCDATAAttribute("id", content.getId());
        atts.addCDATAAttribute("title", content.getTitle(null));
        
        String smallIcon = _cTypesHelper.getSmallIcon(content);
        if (StringUtils.isNotEmpty(smallIcon))
        {
            atts.addCDATAAttribute("iconSmall", smallIcon);
        }
        String iconGlyph = _cTypesHelper.getIconGlyph(content);
        if (StringUtils.isNotEmpty(iconGlyph))
        {
            atts.addCDATAAttribute("iconGlyph", iconGlyph);
        }
        
        atts.addCDATAAttribute("invalidated-contents-count", Integer.toString(invalidatedContents.size()));
        
        XMLUtils.startElement(contentHandler, "program", atts);
        
        if (content instanceof ProgramItem)
        {
            XMLUtils.createElement(contentHandler, "code", ((ProgramItem) content).getCode());
            
            String codeCatalog = ((ProgramItem) content).getCatalog();
            Catalog catalog = _catalogManager.getCatalog(codeCatalog);
            if (catalog != null)
            {
                AttributesImpl catalogAttrs = new AttributesImpl();
                catalogAttrs.addCDATAAttribute("code", codeCatalog);
                XMLUtils.createElement(contentHandler, "catalog", catalogAttrs, catalog.getTitle());
            }
        }
        
        XMLUtils.createElement(contentHandler, "lastModified", DateUtils.zonedDateTimeToString(content.getLastModified()));
        
        UserIdentity user = content.getLastContributor();
        _userHelper.saxUserIdentity(user, contentHandler, "contributor");
        
        XMLUtils.startElement(contentHandler, "invalidatedContents");
        for (Content invalidatedContent : invalidatedContents)
        {
            _saxInvalidatedContent(invalidatedContent);
        }
        XMLUtils.endElement(contentHandler, "invalidatedContents");
        
        XMLUtils.endElement(contentHandler, "program");
    }
    
    private void _saxInvalidatedContent(Content content) throws SAXException
    {
        AttributesImpl atts = new AttributesImpl();
        atts.addCDATAAttribute("id", content.getId());
        atts.addCDATAAttribute("title", content.getTitle(null));
        if (content instanceof ProgramItem)
        {
            atts.addCDATAAttribute("code", ((ProgramItem) content).getCode());
        }
        
        XMLUtils.createElement(contentHandler, "content", atts);
    }

}
