/*
 *  Copyright 2020 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.odf.skill;

import java.util.Optional;
import java.util.Set;
import java.util.stream.Collectors;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;

import org.ametys.cms.data.ContentValue;
import org.ametys.cms.model.properties.AbstractContentProperty;
import org.ametys.cms.model.properties.Property;
import org.ametys.cms.repository.Content;
import org.ametys.cms.repository.ModifiableContent;
import org.ametys.odf.enumeration.OdfReferenceTableHelper;
import org.ametys.odf.program.AbstractProgram;

/**
 * {@link Property} for acquired skills of child courses on a {@link AbstractProgram}
 */
public class ProgramSkillsProperty extends AbstractContentProperty<Content>
{
    /** The name of indexing field holding the computed acquired skills */
    public static final String PROGRAM_SKILLS_PROPERTY_NAME = "acquiredSkills";
    
    private ODFSkillsHelper _skillsHelper;
    
    @Override
    public void service(ServiceManager manager) throws ServiceException
    {
        super.service(manager);
        _skillsHelper = (ODFSkillsHelper) manager.lookup(ODFSkillsHelper.ROLE);
    }
    
    @Override
    protected Set< ? extends ModifiableContent> _getLinkedContents(Content content)
    {
        if (content instanceof AbstractProgram abstractProgram)
        {
            Set<ModifiableContent> values = _skillsHelper.getComputedSkills(abstractProgram, 2)
                                           .stream()
                                           .map(ContentValue::getContentIfExists)
                                           .flatMap(Optional::stream)
                                           .collect(Collectors.toSet());
            return values;
        }
        
        return null;
    }
    
    @Override
    public boolean isMultiple()
    {
        return true;
    }
    
    public String getContentTypeId()
    {
        return OdfReferenceTableHelper.SKILL;
    }
}
