/*
 *  Copyright 2018 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.odf.validator;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import org.apache.avalon.framework.configuration.Configurable;
import org.apache.avalon.framework.configuration.Configuration;
import org.apache.avalon.framework.configuration.ConfigurationException;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;

import org.ametys.cms.contenttype.validation.AbstractContentValidator;
import org.ametys.cms.data.ContentValue;
import org.ametys.cms.repository.Content;
import org.ametys.odf.ODFHelper;
import org.ametys.odf.ProgramItem;
import org.ametys.odf.course.Course;
import org.ametys.odf.courselist.CourseList;
import org.ametys.odf.program.ProgramPart;
import org.ametys.plugins.repository.data.holder.impl.DataHolderHelper;
import org.ametys.plugins.repository.data.holder.values.SynchronizableValue;
import org.ametys.plugins.repository.data.holder.values.SynchronizableValue.Mode;
import org.ametys.plugins.repository.data.holder.values.SynchronizationContext;
import org.ametys.runtime.i18n.I18nizableText;
import org.ametys.runtime.i18n.I18nizableTextParameter;
import org.ametys.runtime.model.ElementDefinition;
import org.ametys.runtime.model.View;
import org.ametys.runtime.parameter.ValidationResult;

/**
 * Global validator for {@link ProgramItem} content.
 * Check that structure to not create an infinite loop
 */
public class ProgramItemHierarchyValidator extends AbstractContentValidator implements Serviceable, Configurable
{
    /** The ODF helper */
    protected ODFHelper _odfHelper;
    
    private Set<String> _childAttributeNames;
    private Set<String> _parentAttributeNames;
    
    @Override
    public void service(ServiceManager smanager) throws ServiceException
    {
        _odfHelper = (ODFHelper) smanager.lookup(ODFHelper.ROLE);
    }
    
    @Override
    public ValidationResult validate(Content content)
    {
        // Nothing
        return ValidationResult.empty();
    }
    
    public void configure(Configuration configuration) throws ConfigurationException
    {
        _childAttributeNames = _configureAttributeNames(configuration, "childAttribute");
        _parentAttributeNames = _configureAttributeNames(configuration, "parentAttribute");
    }
    
    private Set<String> _configureAttributeNames(Configuration configuration, String childName) throws ConfigurationException
    {
        Set<String> attributeNames = new HashSet<>();
        
        Configuration[] children = configuration.getChildren(childName);
        for (Configuration childConf : children)
        {
            attributeNames.add(childConf.getAttribute("name"));
        }
        
        return attributeNames;
    }
    
    @Override
    public ValidationResult validate(Content content, Map<String, Object> values, View view)
    {
        ValidationResult result = new ValidationResult();
        
        List<ProgramItem> newParentProgramItems = new ArrayList<>();
        
        // Get the new parents from values
        Set<String> parentAttributeNames = getParentAttributeNames();
        for (String parentAttributeName : parentAttributeNames)
        {
            if (view.hasModelViewItem(parentAttributeName) && content instanceof ProgramItem)
            {
                ElementDefinition parentDefinition = (ElementDefinition) content.getDefinition(parentAttributeName);
                Object value = values.get(parentAttributeName);
                Object[] valueToValidate = (Object[]) DataHolderHelper.getValueFromSynchronizableValue(value, content, parentDefinition, Optional.of(parentAttributeName), SynchronizationContext.newInstance());
                
                newParentProgramItems = Optional.ofNullable(valueToValidate)
                        .map(Stream::of)
                        .orElseGet(Stream::of)
                        .map(parentDefinition.getType()::castValue)
                        .map(ContentValue.class::cast)
                        .map(ContentValue::getContent)
                        .filter(ProgramItem.class::isInstance)
                        .map(ProgramItem.class::cast)
                        .collect(Collectors.toList());
                
            }
        }
        
        Set<String> names = getChildAttributeNames();
        for (String name : names)
        {
            if (view.hasModelViewItem(name) && content instanceof ProgramItem)
            {
                List<String> childContentIds = _odfHelper.getChildProgramItems((ProgramItem) content).stream()
                        .map(ProgramItem::getId)
                        .collect(Collectors.toList());
                
                ElementDefinition childDefinition = (ElementDefinition) content.getDefinition(name);
                
                Object value = values.get(name);
                Object[] valueToValidate = (Object[]) DataHolderHelper.getValueFromSynchronizableValue(value, content, childDefinition, Optional.of(name), SynchronizationContext.newInstance());
                Mode mode = _getMode(value);
                
                // Check duplicate child and infinite loop is not necessary when value is removed
                if (Mode.REMOVE != mode)
                {
                    ContentValue[] contentValues = Optional.ofNullable(valueToValidate)
                            .map(Stream::of)
                            .orElseGet(Stream::of)
                            .map(childDefinition.getType()::castValue)
                            .toArray(ContentValue[]::new);
                    for (ContentValue contentValue : contentValues)
                    {
                        Content childContent = contentValue.getContent();
                        
                        // Check duplicate child only on APPEND mode
                        if (Mode.APPEND == mode)
                        {
                            if (childContentIds.contains(childContent.getId()))
                            {
                                I18nizableText errorMessage = _getError(childDefinition, content, childContent, "plugin.odf", "PLUGINS_ODF_CONTENT_VALIDATOR_DUPLICATE_CHILD_ERROR");
                                result.addError(errorMessage);
                            }
                        }
                        
                        if (childContent instanceof ProgramItem)
                        {
                            if (!checkAncestors((ProgramItem) childContent, newParentProgramItems) || !checkAncestors((ProgramItem) content, (ProgramItem) childContent))
                            {
                                I18nizableText errorMessage = _getError(childDefinition, content, childContent, "plugin.odf", "PLUGINS_ODF_CONTENT_VALIDATOR_HIERARCHY_ERROR");
                                result.addError(errorMessage);
                            }
                        }
                    }
                }
            }
        }
        
        return result;
    }
    
    private Mode _getMode(Object value)
    {
        return value instanceof SynchronizableValue ? ((SynchronizableValue) value).getMode() : Mode.REPLACE;
    }
    
    /**
     * Get the names of child attribute to be checked
     * @return the child attribute's names
     */
    protected Set<String> getChildAttributeNames()
    {
        return _childAttributeNames;
    }
    
    /**
     * Get the names of parent attribute to be checked
     * @return the parent attribute's names
     */
    protected Set<String> getParentAttributeNames()
    {
        return _parentAttributeNames;
    }
    
    /**
     * Check if the hierarchy of a program item will be still valid if adding the given program item as child.
     * Return false if the {@link ProgramItem} to add is in the hierarchy of the given {@link ProgramItem}
     * @param programItem The content to start search
     * @param childProgramItem The child program item to search in ancestors
     * @return true if child program item is already part of the hierarchy (ascendant search)
     */
    protected boolean checkAncestors(ProgramItem programItem, ProgramItem childProgramItem)
    {
        if (programItem.equals(childProgramItem))
        {
            return false;
        }

        List<? extends ProgramItem> parents = new ArrayList<>();
        if (programItem instanceof Course)
        {
            parents = ((Course) programItem).getParentCourseLists();
        }
        else if (programItem instanceof CourseList)
        {
            parents = ((CourseList) programItem).getParentCourses();
        }
        else if (programItem instanceof ProgramPart)
        {
            parents = ((ProgramPart) programItem).getProgramPartParents();
        }
        
        return checkAncestors(childProgramItem, parents);
    }
    
    /**
     * Check if the hierarchy of a program item will be still valid if adding the given program item as child.
     * Return false if the {@link ProgramItem} to add is in the hierarchy of the given {@link ProgramItem}
     * @param childProgramItem The child program item to add
     * @param parentProgramItems The parent program items of the target
     * @return true if child program item is already part of the hierarchy (ascendant search)
     */
    protected boolean checkAncestors(ProgramItem childProgramItem, List<? extends ProgramItem> parentProgramItems)
    {
        for (ProgramItem parent : parentProgramItems)
        {
            if (parent.equals(childProgramItem))
            {
                return false;
            }
            else if (!checkAncestors(parent, childProgramItem))
            {
                return false;
            }
        }
        
        return true;
    }
    
    /**
     * Retrieves an error message
     * @param childDefinition The child attribute definition
     * @param content The content being edited
     * @param childContent The content to add as child content
     * @param catalog the i18n catalog 
     * @param i18nKey the i18n key for the error message
     * @return the error message
     */
    protected I18nizableText _getError(ElementDefinition childDefinition, Content content, Content childContent, String catalog, String i18nKey)
    {
        Map<String, I18nizableTextParameter> i18nParams = new HashMap<>();
        i18nParams.put("title", new I18nizableText(content.getTitle()));
        i18nParams.put("childTitle", new I18nizableText(childContent.getTitle()));
        i18nParams.put("fieldLabel", childDefinition.getLabel());
        return new I18nizableText(catalog, i18nKey, i18nParams);
    }
}
