/*
 *  Copyright 2022 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.odf.workflow.task;

import java.io.IOException;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import org.apache.avalon.framework.configuration.Configuration;
import org.apache.avalon.framework.configuration.ConfigurationException;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.cocoon.ProcessingException;
import org.apache.cocoon.xml.XMLUtils;
import org.xml.sax.ContentHandler;
import org.xml.sax.SAXException;

import org.ametys.cms.CmsConstants;
import org.ametys.cms.content.compare.ContentComparatorChange;
import org.ametys.cms.content.version.CompareVersionHelper;
import org.ametys.cms.repository.Content;
import org.ametys.core.util.JSONUtils;
import org.ametys.plugins.repository.AmetysRepositoryException;
import org.ametys.plugins.repository.version.VersionAwareAmetysObject;
import org.ametys.runtime.model.DefinitionContext;
import org.ametys.runtime.model.ModelItem;

/**
 * Override {@link AbstractOdfWorkflowTasksComponent} to add the change informations
 */
public abstract class AbstractOdfWorkflowTasksWithChangesComponent extends AbstractOdfWorkflowTasksComponent
{

    /** The JSON util */
    protected JSONUtils _jsonUtils;
    /** The helper for comparing versions */
    protected CompareVersionHelper _compareVersionHelper;
    /** Map of show changes indexed by task id */
    protected Map<String, Boolean> _showChanges = new HashMap<>();
    /** Map of show important changes indexed by task id */
    protected Map<String, Boolean> _showImportantChanges = new HashMap<>();
    /** The (defined in configuration) important attributes by task */
    protected Map<String, Collection<String>> _importantAttributes = new HashMap<>();

    @Override
    public void service(ServiceManager manager) throws ServiceException
    {
        super.service(manager);
        _jsonUtils = (JSONUtils) manager.lookup(JSONUtils.ROLE);
        _compareVersionHelper = (CompareVersionHelper) manager.lookup(CompareVersionHelper.ROLE);
    }

    @Override
    protected void _configureAdditional(Task task, Configuration taskConf) throws ConfigurationException
    {
        _configureChanges(taskConf);
    }

    /**
     * Determines if changes have to be shown for the given task
     * @param task the task
     * @return <code>true</code> if changes are configured to be SAXed
     */
    public boolean showChanges(Task task)
    {
        return _showChanges.containsKey(task.getId()) && _showChanges.get(task.getId());
    }

    /**
     * Determines if important changes have to be shown for the given task
     * @param task the task
     * @return <code>true</code> if changes are configured to be SAXed
     */
    public boolean showImportantChanges(Task task)
    {
        return _showImportantChanges.containsKey(task.getId()) && _showImportantChanges.get(task.getId());
    }

    /**
     * Configures the changes to SAX
     * @param taskConfiguration The task configuration
     * @throws ConfigurationException  If the configuration is invalid.
     */
    protected void _configureChanges(Configuration taskConfiguration) throws ConfigurationException
    {
        String taskId = taskConfiguration.getAttribute("id", "").trim();
        
        _showChanges.put(taskId, taskConfiguration.getChild("show-changes", false) != null);
        
        Configuration importantChangesConf = taskConfiguration.getChild("show-important-changes", false);
        _showImportantChanges.put(taskId, importantChangesConf != null);
        
        if (importantChangesConf != null)
        {
            Configuration[] attsConf = importantChangesConf.getChildren("attribute");
            Collection<String> importantAttributes = Stream.of(attsConf)
                    .map(attr -> attr.getAttribute("ref", null))
                    .filter(Objects::nonNull)
                    .collect(Collectors.toList());
            
            _importantAttributes.put(taskId, importantAttributes);
        }
    }

    @Override
    protected void _saxAdditionalData(ContentHandler ch, Content content, Task task) throws SAXException
    {
        super._saxAdditionalData(ch, content, task);
        
        if (content instanceof VersionAwareAmetysObject && (showChanges(task) || showImportantChanges(task)))
        {
            try
            {
                List<ContentComparatorChange> changes = _getChanges((Content & VersionAwareAmetysObject) content);
                _saxAttributeChanges(ch, content, task, changes);
            }
            catch (AmetysRepositoryException | IOException e)
            {
                throw new SAXException("Cannot determine the change of attributes for Content '" + content + "'", e);
            }
        }
    }

    /**
     * Gets the changes
     * @param <C> The type of the {@link VersionAwareAmetysObject} {@link Content}
     * @param content The content
     * @return The changes
     * @throws AmetysRepositoryException repository exception
     * @throws IOException IO exception
     */
    protected <C extends Content & VersionAwareAmetysObject> List<ContentComparatorChange> _getChanges(C content) throws AmetysRepositoryException, IOException
    {
        String baseVersion = _getBaseVersion(content);
        if (baseVersion == null)
        {
            // No target version => no change
            return Collections.emptyList();
        }
        return _compareVersionHelper.compareVersions(content.getId(), _getTargetVersion(content), baseVersion).getChanges();
    }

    /**
     * Gets the source version of comparison. By default, it is the current one.
     * @param <C> The type of the {@link VersionAwareAmetysObject} {@link Content}
     * @param versionable The {@link Content}
     * @return the source version
     */
    protected <C extends Content & VersionAwareAmetysObject> String _getTargetVersion(C versionable)
    {
        return _compareVersionHelper.getCurrentVersion(versionable);
    }

    /**
     * Gets the base version of comparison. By default, it is the last validated one.
     * @param <C> The type of the {@link VersionAwareAmetysObject} {@link Content}
     * @param versionable The {@link Content}
     * @return the target version
     */
    protected <C extends Content & VersionAwareAmetysObject> String _getBaseVersion(C versionable)
    {
        return _compareVersionHelper.getLastVersionWithLabel(versionable, CmsConstants.LIVE_LABEL);
    }

    /**
     * SAX attributes of the given content which changed since the last validation
     * @param ch The content handler
     * @param content The content
     * @param task the current task
     * @param changes The changes
     * @throws SAXException If an error occurred
     */
    protected void _saxAttributeChanges(ContentHandler ch, Content content, Task task, List<ContentComparatorChange> changes) throws SAXException
    {
        /*
         * SAX as stringified JSON as ExtJS XML reader seem to not be able to read multiple values
         * if format is 
         * <change name="degree" isImportant="true">
         *   <label>
         *     <i18n:text key="" catalogue=""/>
         *   </label>
         * </change>
         * <change name="domain" isImportant="false">
         *   <label>
         *     <i18n:text key="" catalogue=""/>
         *   </label>
         * </change>
         * 
         * for instance.
         * 
         * So use JSON for easier deserialization
         */
        List<Map<String, Object>> changesAsJson = _getAttributeChanges(task, changes);
        String changesAsString = _jsonUtils.convertObjectToJson(changesAsJson);
        XMLUtils.createElement(ch, "attribute-changes", changesAsString);
    }

    /**
     * Gets the attribute changes (as JSON) of the given content which changed since the last validation
     * @param task the current task
     * @param changes The changes
     * @return The changes as JSON
     */
    protected List<Map<String, Object>> _getAttributeChanges(Task task, List<ContentComparatorChange> changes)
    {
        return _compareVersionHelper.getChangedModelItems(changes)
                .map(attributeChange ->
                {
                    boolean isImportant = _isImportant(task, attributeChange);
                    return _toJson(attributeChange, isImportant);
                })
                .collect(Collectors.toList());
    }

    private Map<String, Object> _toJson(ModelItem modelItem, boolean isImportant)
    {
        try
        {
            return Map.of(
                    "modelItem", modelItem.toJSON(DefinitionContext.newInstance()),
                    "isImportant", isImportant);
        }
        catch (ProcessingException e)
        {
            throw new IllegalStateException("Cannot transform modelItem to JSON", e);
        }
    }

    /**
     * Determines if a change is important
     * @param task the current task
     * @param changedModelItem The changed {@link ModelItem}
     * @return <code>true</code> if the change is important
     */
    protected boolean _isImportant(Task task, ModelItem changedModelItem)
    {
        if (_importantAttributes.containsKey(task.getId()))
        {
            return _importantAttributes.get(task.getId()).contains(changedModelItem.getPath());
        }
        return false;
    }

}
