/*
 *  Copyright 2018 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.odf.coursepart;

import java.util.Arrays;
import java.util.List;
import java.util.Optional;
import java.util.stream.Collectors;

import javax.jcr.Node;

import org.apache.commons.lang3.StringUtils;

import org.ametys.cms.data.ContentDataHelper;
import org.ametys.cms.data.ContentValue;
import org.ametys.cms.repository.ModifiableDefaultContent;
import org.ametys.odf.content.code.DisplayCodeProperty;
import org.ametys.odf.course.Course;
import org.ametys.plugins.repository.AmetysRepositoryException;

/**
 * Class representing a {@link CoursePart}
 */
public class CoursePart extends ModifiableDefaultContent<CoursePartFactory>
{
    /** The attribute holding the catalog */
    public static final String CATALOG = "catalog";
    
    /** The attribute holding the code */
    public static final String CODE = "code";
    
    /** Constants for attribute 'nature' */
    public static final String NATURE = "nature";

    /** Constants for attribute 'nbHours' */
    public static final String NB_HOURS = "nbHours";

    /** Name of attribute for course holder */
    public static final String COURSE_HOLDER = "courseHolder";

    /** Name of attribute for parent courses */
    public static final String PARENT_COURSES = "courses";
    
    /**
     * Constructor.
     * @param node the JCR Node.
     * @param parentPath the parent path
     * @param factory the corresponding factory.
     */
    public CoursePart(Node node, String parentPath, CoursePartFactory factory)
    {
        super(node, parentPath, factory);
    }
    
    /**
     * Get the catalog.
     * @return the catalog
     */
    public String getCatalog()
    {
        return getValue(CATALOG);
    }
    
    /**
     * Set the catalog.
     * @param catalog The catalog
     * @throws AmetysRepositoryException if an error occurs
     */
    public void setCatalog(String catalog) throws AmetysRepositoryException
    {
        setValue(CATALOG, catalog);
    }

    /**
     * Get the code.
     * @return the code
     */
    public String getCode()
    {
        return getValue(CODE, false, StringUtils.EMPTY);
    }

    /**
     * Set the code.
     * @param code The code
     * @throws AmetysRepositoryException if an error occurs
     */
    public void setCode(String code) throws AmetysRepositoryException
    {
        setValue(CODE, code);
    }

    /**
     * Get the code to display
     * @return the code to display
     */
    public String getDisplayCode()
    {
        return getValue(DisplayCodeProperty.PROPERTY_NAME, false, StringUtils.EMPTY);
    }
    
    /**
     * Get the nature.
     * @return the nature
     */
    public String getNature()
    {
        return ContentDataHelper.getContentIdFromContentData(this,  NATURE);
    }

    /**
     * Get the number of hours
     * @return the number of hours
     */
    public double getNumberOfHours()
    {
        return getValue(NB_HOURS, false, 0D);
    }

    /**
     * Get the course holder
     * @return the course holder
     */
    public Course getCourseHolder()
    {
        return Optional.ofNullable((ContentValue) getValue(COURSE_HOLDER))
                                .flatMap(ContentValue::getContentIfExists)
                                .map(Course.class::cast)
                                .orElse(null);
    }
    
    /**
     * Get the parent courses.
     * @return The {@link List} of parent {@link Course}s
     */
    public List<Course> getCourses()
    {
        return Arrays.stream(getValue(PARENT_COURSES, false, new ContentValue[0]))
                .map(ContentValue::getContentIfExists)
                .flatMap(Optional::stream)
                .map(Course.class::cast)
                .collect(Collectors.toList());
    }
}
