/*
 *  Copyright 2025 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.odf.restriction;

import java.util.Arrays;
import java.util.List;
import java.util.Set;

import org.apache.avalon.framework.activity.Initializable;
import org.apache.avalon.framework.configuration.Configuration;
import org.apache.avalon.framework.configuration.ConfigurationException;
import org.apache.avalon.framework.context.Context;
import org.apache.avalon.framework.context.ContextException;
import org.apache.avalon.framework.context.Contextualizable;
import org.apache.cocoon.components.ContextHelper;
import org.apache.cocoon.environment.Request;
import org.apache.commons.lang3.StringUtils;

import org.ametys.cms.model.restrictions.RestrictedModelItem;
import org.ametys.cms.repository.Content;
import org.ametys.core.right.RightManager.RightResult;
import org.ametys.odf.data.EducationalPath;
import org.ametys.odf.rights.ODFRightHelper;
import org.ametys.odf.rights.ODFRightHelper.ContextualizedContent;
import org.ametys.runtime.config.Config;
import org.ametys.runtime.model.ModelItem;

/**
 * Restrictions for a repeater with educational path or for data in a repeater with educational
 */
public class RepeaterWithEducationalPathRestriction extends SharedOnlyRestriction implements Contextualizable, Initializable
{
    /** The fields handled by this restriction */
    protected List<String> _fields;
    
    private Context _context;

    @Override
    public void configure(Configuration configuration) throws ConfigurationException
    {
        super.configure(configuration);
        
        _checkWriteRights();
    }
    
    /**
     * Ensure the configuration of write rights is correct
     * @throws ConfigurationException If the configuration is incorrect
     */
    protected void _checkWriteRights() throws ConfigurationException
    {
        if (_writeRightIds == null || _writeRightIds.isEmpty())
        {
            throw new ConfigurationException("At least one write right ids must be set for the " + getClass().getName() + " restriction: you should at least set the classic edition right.");
        }
    }

    public void contextualize(Context context) throws ContextException
    {
        _context = context;
    }
    
    public void initialize() throws Exception
    {
        String[] fields = StringUtils.split(Config.getInstance().getValue("odf.consumercaneditcoursedatabyeducationalpath", false, ""), ",");
        _fields = Arrays.stream(fields).filter(StringUtils::isNotBlank).toList();
    }
    
    @Override
    public RestrictionResult canWrite(Content content, RestrictedModelItem modelItem)
    {
        if (canRead(content, modelItem) == RestrictionResult.FALSE)
        {
            return RestrictionResult.FALSE;
        }
        return super.canWrite(content, modelItem);
    }
    
    
    @Override
    protected boolean _hasRights(Content content, RestrictedModelItem modelItem, Set<String> rightLimitations)
    {
        if (super._hasRights(content, modelItem, rightLimitations))
        {
            // User is producer of content with needed rights for restrictions
            return true;
        }
        
        if (_fields.contains(StringUtils.substringBefore(modelItem.getPath(), ModelItem.ITEM_PATH_SEPARATOR))) // will match exact fields or repeaters
        {
            Request request = ContextHelper.getRequest(_context);
            @SuppressWarnings("unchecked")
            List<EducationalPath> educationalPaths = (List<EducationalPath>) request.getAttribute(ODFRightHelper.REQUEST_ATTR_EDUCATIONAL_PATHS);
            
            if (educationalPaths != null)
            {
                // Test rights as a consumer of the content (ie. allowed to edit repeater data for one of the given educational paths)
                for (String rightId : rightLimitations)
                {
                    for (EducationalPath educationalPath : educationalPaths)
                    {
                        if (_rightManager.currentUserHasRight(rightId, new ContextualizedContent(content, educationalPath)) == RightResult.RIGHT_ALLOW)
                        {
                            return true;
                        }
                    }
                }
            }
        }
        
        return false;
    }
}
