/*
 *  Copyright 2020 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.odf.content;

import java.io.IOException;
import java.util.Map;
import java.util.Set;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.cocoon.ProcessingException;
import org.apache.cocoon.environment.ObjectModelHelper;
import org.apache.cocoon.environment.Request;
import org.apache.cocoon.generation.ServiceableGenerator;
import org.apache.cocoon.xml.AttributesImpl;
import org.apache.cocoon.xml.XMLUtils;
import org.apache.commons.lang.StringUtils;
import org.xml.sax.SAXException;

import org.ametys.cms.repository.Content;
import org.ametys.odf.course.Course;
import org.ametys.odf.enumeration.OdfReferenceTableEntry;
import org.ametys.odf.program.AbstractProgram;
import org.ametys.odf.program.Program;
import org.ametys.odf.program.SubProgram;
import org.ametys.odf.skill.ODFSkillsHelper;

/**
 * SAX the skills hold by child courses of a {@link Program} or a not mutualised {@link SubProgram}
 */
public class ProgramSkillsGenerator extends ServiceableGenerator
{
    private ODFSkillsHelper _skillsHelper;
    
    @Override
    public void service(ServiceManager smanager) throws ServiceException
    {
        super.service(smanager);
        _skillsHelper = (ODFSkillsHelper) smanager.lookup(ODFSkillsHelper.ROLE);
    }
    
    public void generate() throws IOException, SAXException, ProcessingException
    {
        Request request = ObjectModelHelper.getRequest(objectModel);
        Content content = (Content) request.getAttribute(Content.class.getName());

        String viewName = request.getParameter("viewName");
        
        contentHandler.startDocument();
        
        if ("main".equals(viewName) || StringUtils.isEmpty(viewName))
        {
            if (content instanceof AbstractProgram abstractProgram)
            {
                Map<Content, Map<Content, Set<Content>>> skillsDistribution = _skillsHelper.getSkillsDistribution(abstractProgram, 2);
                if (skillsDistribution != null)
                {
                    saxSkillsDistribution(skillsDistribution);
                }
            }
            else
            {
                getLogger().warn("Cannot get the skills of a non program '" + content.getId() + "'");
            }
        }
        
        contentHandler.endDocument();
    }
    
    /**
     * SAX the skills distribution
     * @param distribution the skills distribution
     * @throws IOException if an I/O error occurred
     * @throws SAXException if an error occurred while saxing
     */
    protected void saxSkillsDistribution(Map<Content, Map<Content, Set<Content>>> distribution) throws IOException, SAXException
    {
        XMLUtils.startElement(contentHandler, "skills");
        
        // Iterate on macro skills
        for (Map.Entry<Content, Map<Content, Set<Content>>> macroSkillEntry : distribution.entrySet())
        {
            Content macroSkill = macroSkillEntry.getKey();
            
            AttributesImpl attrs = new AttributesImpl();
            attrs.addCDATAAttribute("id", macroSkill.getId());
            attrs.addCDATAAttribute("title", macroSkill.getTitle());
            attrs.addCDATAAttribute("code", macroSkill.getValue(OdfReferenceTableEntry.CODE, false, ""));
            
            XMLUtils.startElement(contentHandler, "skillSet", attrs);
            
            // Iterate on micro skills
            for (Map.Entry<Content, Set<Content>> microSkillEntry : macroSkillEntry.getValue().entrySet())
            {
                Content microSkill = microSkillEntry.getKey();
                
                attrs.clear();
                attrs.addCDATAAttribute("id", microSkill.getId());
                attrs.addCDATAAttribute("title", microSkill.getTitle());
                attrs.addCDATAAttribute("code", microSkill.getValue(OdfReferenceTableEntry.CODE, false, StringUtils.EMPTY));
                
                XMLUtils.startElement(contentHandler, "skill", attrs);
                
                // Iterate on courses
                for (Content course : microSkillEntry.getValue())
                {
                    attrs.clear();
                    attrs.addCDATAAttribute("id", course.getId());
                    attrs.addCDATAAttribute("title", course.getTitle());
                    attrs.addCDATAAttribute("code", ((Course) course).getCode());
                    
                    XMLUtils.startElement(contentHandler, "course", attrs);
                    
                    XMLUtils.endElement(contentHandler, "course");
                }
                
                XMLUtils.endElement(contentHandler, "skill");
            }
            
            XMLUtils.endElement(contentHandler, "skillSet");
        }
        
        XMLUtils.endElement(contentHandler, "skills");
    }
}
