/*
 *  Copyright 2020 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.odf.export.pdf;

import java.io.IOException;
import java.util.Map;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;
import org.apache.cocoon.ProcessingException;
import org.apache.cocoon.environment.ObjectModelHelper;
import org.apache.cocoon.generation.AbstractGenerator;
import org.apache.cocoon.serialization.ZipArchiveSerializer;
import org.apache.cocoon.xml.AttributesImpl;
import org.apache.cocoon.xml.XMLUtils;
import org.apache.commons.lang.StringUtils;
import org.xml.sax.SAXException;

import org.ametys.cms.repository.Content;
import org.ametys.core.util.I18nUtils;
import org.ametys.plugins.repository.AmetysObjectResolver;
import org.ametys.runtime.i18n.I18nizableText;

/**
 * Generate the list of booklet to include in a zip for downloading
 */
public class EducationalBookletZipGenerator extends AbstractGenerator implements Serviceable
{
    /** The Ametys Object resolver */
    protected AmetysObjectResolver _resolver;
    
    /** The i18n utils */
    protected I18nUtils _i18nUtils;
    
    public void service(ServiceManager manager) throws ServiceException
    {
        _resolver = (AmetysObjectResolver) manager.lookup(AmetysObjectResolver.ROLE);
        _i18nUtils = (I18nUtils) manager.lookup(I18nUtils.ROLE);
    }
    
    public void generate() throws IOException, SAXException, ProcessingException
    {
        Map parentContextParameters = (Map) objectModel.get(ObjectModelHelper.PARENT_CONTEXT);
        if (parentContextParameters == null)
        {
            throw new ProcessingException("Missing parent context");
        }

        String programItemIdsAsString = (String) parentContextParameters.get("programItemIds");
        String[] programItemIds = StringUtils.split(programItemIdsAsString, ",");
        
        contentHandler.startDocument();
        contentHandler.startPrefixMapping("zip", ZipArchiveSerializer.ZIP_NAMESPACE);
        XMLUtils.startElement(contentHandler, ZipArchiveSerializer.ZIP_NAMESPACE, "archive");
        for (String programItemId : programItemIds)
        {
            Content content = _resolver.resolveById(programItemId);
            String prefix = _i18nUtils.translate(new I18nizableText("plugin.odf", "PLUGINS_ODF_EDUCATIONAL_BOOKLET_PROGRAMITEM_PDF_NAME_PREFIX"), content.getLanguage());
            String fileName = prefix + " " + content.getTitle() + ".pdf";
            
            AttributesImpl zipAttrs = new AttributesImpl();
            zipAttrs.addAttribute("", "name", "name", "CDATA", fileName);
            zipAttrs.addAttribute("", "src", "src", "CDATA", "ametys-home://data/odf/booklet/" + content.getName() + "/" + content.getLanguage() + "/educational-booklet.pdf");
            XMLUtils.startElement(contentHandler, ZipArchiveSerializer.ZIP_NAMESPACE, "entry", zipAttrs);
            XMLUtils.endElement(contentHandler, ZipArchiveSerializer.ZIP_NAMESPACE, "entry");
        }
        
        XMLUtils.endElement(contentHandler, ZipArchiveSerializer.ZIP_NAMESPACE, "archive");
        contentHandler.endPrefixMapping("zip");
        contentHandler.endDocument();
    }
}
