/*
 *  Copyright 2010 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */

package org.ametys.odf.cdmfr;

import java.io.IOException;
import java.io.InputStream;
import java.time.LocalDate;
import java.time.format.DateTimeFormatter;
import java.util.HashMap;

import org.apache.cocoon.xml.AttributesImpl;
import org.apache.cocoon.xml.XMLUtils;
import org.apache.excalibur.source.Source;
import org.apache.excalibur.source.SourceResolver;
import org.apache.excalibur.xml.sax.XMLizable;
import org.xml.sax.ContentHandler;
import org.xml.sax.SAXException;

import org.ametys.cms.data.RichText;
import org.ametys.runtime.model.type.DataContext;

/**
 * Helper for export tasks.
 */
public final class CDMHelper
{
    /** The CDM date formatter. */
    public static final DateTimeFormatter CDM_DATE_FORMATTER = DateTimeFormatter.ofPattern("yyyy-MM-dd");
    
    private CDMHelper()
    {
        // empty constructor
    }

    
    /**
     * Transform a docbook in a CDM infoBlock way.
     * @param contentHandler the handler to SAX CDM.
     * @param tag the surrounding tag.
     * @param richText the richText. If <code>null</code>, nothing is SAXed.
     * @param richTextContext The context of the rich 
     * @param sourceResolver the Cocoon SourceResolver.
     * @throws SAXException if an error occurs. 
     */
    public static void richText2CDM(ContentHandler contentHandler, String tag, RichText richText, DataContext richTextContext, SourceResolver sourceResolver) throws SAXException
    {
        richText2CDM(contentHandler, tag, richText, richTextContext, sourceResolver, false, new AttributesImpl());
    }
    
    /**
     * Transform a docbook in a CDM infoBlock way.
     * @param contentHandler the handler to SAX CDM.
     * @param tag the surrounding tag.
     * @param richText the richText. If <code>null</code>, nothing is SAXed.
     * @param richTextContext The context of the rich 
     * @param sourceResolver the Cocoon SourceResolver.
     * @param attrs The attributes to SAX
     * @throws SAXException if an error occurs. 
     */
    public static void richText2CDM(ContentHandler contentHandler, String tag, RichText richText, DataContext richTextContext, SourceResolver sourceResolver, AttributesImpl attrs) throws SAXException
    {
        richText2CDM(contentHandler, tag, richText, richTextContext, sourceResolver, false, attrs);
    }

    
    /**
     * Transform a docbook in a CDM infoBlock way.
     * @param contentHandler the handler to SAX CDM.
     * @param tag the surrounding tag.
     * @param richText the richText. If <code>null</code>, nothing is SAXed.
     * @param richTextContext The context of the rich  
     * @param sourceResolver the Cocoon SourceResolver.
     * @param writeIfEmpty true to sax CDM info block even if the rich text is empty
     * @throws SAXException if an error occurs. 
     */
    public static void richText2CDM(ContentHandler contentHandler, String tag, RichText richText, DataContext richTextContext, SourceResolver sourceResolver, boolean writeIfEmpty) throws SAXException
    {
        richText2CDM(contentHandler, tag, richText, richTextContext, sourceResolver, writeIfEmpty, false, new AttributesImpl());
    }
    
    /**
     * Transform a docbook in a CDM infoBlock way.
     * @param contentHandler the handler to SAX CDM.
     * @param tag the surrounding tag.
     * @param richText the richText. If <code>null</code>, nothing is SAXed.
     * @param richTextContext The context of the rich 
     * @param sourceResolver the Cocoon SourceResolver.
     * @param writeIfEmpty true to sax CDM info block even if the rich text is empty
     * @param rawHtmlExpert true to export HTML expert as raw CDATA. If false, parse and resolve HTML expert code.
     * @throws SAXException if an error occurs. 
     */
    public static void richText2CDM(ContentHandler contentHandler, String tag, RichText richText, DataContext richTextContext, SourceResolver sourceResolver, boolean writeIfEmpty, boolean rawHtmlExpert) throws SAXException
    {
        richText2CDM(contentHandler, tag, richText, richTextContext, sourceResolver, writeIfEmpty, rawHtmlExpert, new AttributesImpl());
    }

    /**
     * Transform a docbook in a CDM infoBlock way.
     * @param contentHandler the handler to SAX CDM.
     * @param tag the surrounding tag.
     * @param richText the richText. If <code>null</code>, nothing is SAXed.
     * @param richTextContext The context of the rich 
     * @param writeIfEmpty true to sax CDM info block even if the rich text is empty
     * @param attrs The attributes to SAX
     * @param sourceResolver the Cocoon SourceResolver.
     * @throws SAXException if an error occurs. 
     */
    public static void richText2CDM(ContentHandler contentHandler, String tag, RichText richText, DataContext richTextContext, SourceResolver sourceResolver, boolean writeIfEmpty, AttributesImpl attrs) throws SAXException
    {
        richText2CDM(contentHandler, tag, richText, richTextContext, sourceResolver, writeIfEmpty, false, attrs);
    }
    
    /**
     * Transform a docbook in a CDM infoBlock way.
     * @param contentHandler the handler to SAX CDM.
     * @param tag the surrounding tag.
     * @param richText the richText. If <code>null</code>, nothing is SAXed.
     * @param richTextContext The context of the rich 
     * @param writeIfEmpty true to sax CDM info block even if the rich text is empty
     * @param rawHtmlExpert true to export HTML expert as raw CDATA. If false, parse and resolve HTML expert code.
     * @param attrs The attributes to SAX
     * @param sourceResolver the Cocoon SourceResolver.
     * @throws SAXException if an error occurs. 
     */
    public static void richText2CDM(ContentHandler contentHandler, String tag, RichText richText, DataContext richTextContext, SourceResolver sourceResolver, boolean writeIfEmpty, boolean rawHtmlExpert, AttributesImpl attrs) throws SAXException
    {
        if (richText != null)
        {
            XMLUtils.startElement(contentHandler, tag, attrs);
            richText2CDM(contentHandler, richText, richTextContext, sourceResolver, rawHtmlExpert);
            XMLUtils.endElement(contentHandler, tag);
        }
        else if (attrs.getLength() > 0)
        {
            XMLUtils.createElement(contentHandler, tag, attrs);
        }
        else if (writeIfEmpty)
        {
            XMLUtils.createElement(contentHandler, tag, attrs);
        }
    }
    
    /**
     * Transform a docbook in a CDM infoBlock way.
     * @param contentHandler the handler to SAX CDM.
     * @param richText the richText. If <code>null</code>, nothing is SAXed
     * @param richTextContext The context of the rich
     * @param sourceResolver the Cocoon SourceResolver.
     * @throws SAXException if an error occurs. 
     */
    public static void richText2CDM(ContentHandler contentHandler, RichText richText, DataContext richTextContext, SourceResolver sourceResolver) throws SAXException
    {
        richText2CDM(contentHandler, richText, richTextContext, sourceResolver, false);
    }
    
    /**
     * Transform a docbook in a CDM infoBlock way.
     * @param contentHandler the handler to SAX CDM.
     * @param richText the richText. If <code>null</code>, nothing is SAXed.
     * @param richTextContext The context of the rich 
     * @param sourceResolver the Cocoon SourceResolver.
     * @param rawHtmlExpert true to export HTML expert as raw CDATA. If false, parse and resolve HTML expert code.
     * @throws SAXException if an error occurs. 
     */
    public static void richText2CDM(ContentHandler contentHandler, RichText richText, DataContext richTextContext, SourceResolver sourceResolver, boolean rawHtmlExpert) throws SAXException
    {
        if (richText != null)
        {
            try (InputStream is = richText.getInputStream())
            {
                HashMap<String, Object> params = new HashMap<>();
                params.put("source", is);
                params.put("rawHtmlExpert", rawHtmlExpert);
                params.put("dataContext", richTextContext);
                
                Source source = sourceResolver.resolveURI("cocoon://_plugins/odf/docbook2cdm", null, params);
                
                ((XMLizable) source).toSAX(contentHandler);
    
            }
            catch (IOException ex)
            {
                throw new RuntimeException(ex);
            }
        }
    }
    
    /**
     * SAX a date in a CDM document.
     * @param contentHandler The content handler to SAX into.
     * @param tagName The element name.
     * @param value The date value.
     * @throws SAXException if an error occurs.
     */
    public static void date2CDM(ContentHandler contentHandler, String tagName, LocalDate value) throws SAXException
    {
        if (value != null)
        {
            AttributesImpl attrs = new AttributesImpl();
            attrs.addCDATAAttribute(CDMFRTagsConstants.ATTRIBUTE_DATE, value.format(CDM_DATE_FORMATTER));
            XMLUtils.createElement(contentHandler, tagName, attrs);
        }
    }
    
}
