/*
 *  Copyright 2021 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.odf.schedulable;

import java.io.IOException;
import java.util.HashMap;
import java.util.Map;
import java.util.Objects;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.exception.ExceptionUtils;
import org.quartz.JobDataMap;
import org.quartz.JobExecutionContext;
import org.quartz.JobExecutionException;

import org.ametys.cms.schedule.AbstractSendingMailSchedulable;
import org.ametys.core.schedule.progression.ContainerProgressionTracker;
import org.ametys.core.schedule.progression.SimpleProgressionTracker;
import org.ametys.core.ui.mail.StandardMailBodyHelper;
import org.ametys.odf.catalog.Catalog;
import org.ametys.odf.catalog.CatalogsManager;
import org.ametys.plugins.core.schedule.Scheduler;
import org.ametys.runtime.i18n.I18nizableText;
import org.ametys.runtime.i18n.I18nizableTextParameter;

/**
 * Schedulable to copy a catalog.
 */
public class CopyCatalogSchedulable extends AbstractSendingMailSchedulable
{

    /** Schedulable ID */
    public static final String SCHEDULABLE_ID = CopyCatalogSchedulable.class.getName();
    
    /** The key for the source catalog */
    public static final String JOBDATAMAP_SRC_CATALOG_KEY = "srcCatalog";
    
    /** The key for the destinationcatalog */
    public static final String JOBDATAMAP_DEST_CATALOG_KEY = "destCatalog";
    
    private static final String __RESTART_SOLR_COMMIT = "restart-solr-commit";
    private static final String __UPDATE_WORKFLOW_STEP = "update-workflow-step";
    private static final String __UPDATES_AFTER_COPY_STEP = "updates-after-copy-step";
    private static final String __COPY_STEP = "copy-step";
    private static final String __BEFORE_COPY_CHECKS_STEP = "before-copy-checks-step";
    
    /** The catalogs manager */
    protected CatalogsManager _catalogsManager;
    
    @Override
    public void service(ServiceManager manager) throws ServiceException
    {
        super.service(manager);
        _catalogsManager = (CatalogsManager) manager.lookup(CatalogsManager.ROLE);
    }
    
    @Override
    protected void _doExecute(JobExecutionContext context, ContainerProgressionTracker progressionTracker) throws Exception
    {
        SimpleProgressionTracker beforeCopyChecks = progressionTracker.addSimpleStep(__BEFORE_COPY_CHECKS_STEP, new I18nizableText("plugin.odf", "PLUGINS_ODF_SCHEDULABLE_COPY_CATALOG_BEFORE_COPY_CHECKS_STEP_LABEL"));
        progressionTracker.addSimpleStep(__COPY_STEP, new I18nizableText("plugin.odf", "PLUGINS_ODF_SCHEDULABLE_COPY_CATALOG_COPY_STEP_LABEL"), 10);
        progressionTracker.addSimpleStep(__UPDATES_AFTER_COPY_STEP, new I18nizableText("plugin.odf", "PLUGINS_ODF_SCHEDULABLE_COPY_CATALOG_UPDATE_AFTER_COPY_STEP_LABEL"), 5);
        progressionTracker.addSimpleStep(__UPDATE_WORKFLOW_STEP, new I18nizableText("plugin.odf", "PLUGINS_ODF_SCHEDULABLE_COPY_CATALOG_UPDATE_WORKFLOW_STEP_LABEL"), 10);
        progressionTracker.addSimpleStep(__RESTART_SOLR_COMMIT, new I18nizableText("plugin.odf", "PLUGINS_ODF_SCHEDULABLE_COPY_CATALOG_RESTART_SOLR_COMMIT_STEP_LABEL"), 10);
        
        beforeCopyChecks.setSize(1);

        JobDataMap jobDataMap = context.getJobDetail().getJobDataMap();
        
        // Check source catalog
        String srcCatalogName = jobDataMap.getString(Scheduler.PARAM_VALUES_PREFIX + JOBDATAMAP_SRC_CATALOG_KEY);
        if (StringUtils.isEmpty(srcCatalogName))
        {
            throw new JobExecutionException("The source catalog name cannot be empty.");
        }
        
        // Check destination catalog
        String destCatalogName = jobDataMap.getString(Scheduler.PARAM_VALUES_PREFIX + JOBDATAMAP_DEST_CATALOG_KEY);
        if (StringUtils.isEmpty(destCatalogName))
        {
            throw new JobExecutionException("The destination catalog name cannot be empty.");
        }
        
        // Check that source and destination catalog are different
        if (Objects.equals(srcCatalogName, destCatalogName))
        {
            throw new JobExecutionException("The source and destination catalogs cannot be the same.");
        }
        
        // Check the existence of the source catalog
        Catalog srcCatalog = _catalogsManager.getCatalog(srcCatalogName);
        if (srcCatalog == null)
        {
            throw new JobExecutionException("The source catalog should exist.");
        }

        // Check the existence of the destination catalog
        Catalog destCatalog = _catalogsManager.getCatalog(destCatalogName);
        if (destCatalog == null)
        {
            destCatalog = _catalogsManager.createCatalog(destCatalogName, destCatalogName);
        }
        
        beforeCopyChecks.increment();
        
        _catalogsManager.copyCatalog(destCatalog, srcCatalog, progressionTracker);
    }

    @Override
    protected I18nizableText _getSuccessMailSubject(JobExecutionContext context) throws Exception
    {
        return new I18nizableText("plugin.odf", "PLUGINS_ODF_SCHEDULABLE_COPY_CATALOG_SUCCESS_MAIL_SUBJECT");
    }
    
    @Override
    protected boolean _isMailBodyInHTML(JobExecutionContext context) throws Exception
    {
        return true;
    }

    @Override
    protected String _getSuccessMailBody(JobExecutionContext context, String language) throws Exception
    {
        try
        {
            return StandardMailBodyHelper.newHTMLBody()
                    .withTitle(_getSuccessMailSubject(context))
                    .withMessage(new I18nizableText("plugin.odf", "PLUGINS_ODF_SCHEDULABLE_COPY_CATALOG_SUCCESS_MAIL_BODY", _getCommonI18nParams(context)))
                    .withLanguage(language)
                    .build();
        }
        catch (IOException e)
        {
            getLogger().warn("Failed to build HTML email body for catalog copy result. Fallback to no wrapped email", e);
            return _i18nUtils.translate(new I18nizableText("plugin.odf", "PLUGINS_ODF_SCHEDULABLE_COPY_CATALOG_SUCCESS_MAIL_BODY", _getCommonI18nParams(context)), language);
        }
    }

    @Override
    protected I18nizableText _getErrorMailSubject(JobExecutionContext context) throws Exception
    {
        return new I18nizableText("plugin.odf", "PLUGINS_ODF_SCHEDULABLE_COPY_CATALOG_ERROR_MAIL_SUBJECT");
    }

    @Override
    protected String _getErrorMailBody(JobExecutionContext context, String language, Throwable throwable) throws Exception
    {
        try
        {
            String error = ExceptionUtils.getStackTrace(throwable);
            
            return StandardMailBodyHelper.newHTMLBody()
                    .withTitle(_getErrorMailSubject(context))
                    .withMessage(new I18nizableText("plugin.odf", "PLUGINS_ODF_SCHEDULABLE_COPY_CATALOG_ERROR_MAIL_BODY", _getCommonI18nParams(context)))
                    .withDetails(null, error, true)
                    .withLanguage(language)
                    .build();
        }
        catch (IOException e)
        {
            getLogger().warn("Failed to build HTML email body for catalog copy result. Fallback to no wrapped email", e);
            return _i18nUtils.translate(new I18nizableText("plugin.odf", "PLUGINS_ODF_SCHEDULABLE_COPY_CATALOG_ERROR_MAIL_BODY", _getCommonI18nParams(context)), language);
        }
    }
    
    private Map<String, I18nizableTextParameter> _getCommonI18nParams(JobExecutionContext context)
    {
        JobDataMap jobDataMap = context.getJobDetail().getJobDataMap();
        
        Map<String, I18nizableTextParameter> i18nParams = new HashMap<>();
        i18nParams.put("srcCatalog", new I18nizableText(jobDataMap.getString(Scheduler.PARAM_VALUES_PREFIX + JOBDATAMAP_SRC_CATALOG_KEY)));
        i18nParams.put("destCatalog", new I18nizableText(jobDataMap.getString(Scheduler.PARAM_VALUES_PREFIX + JOBDATAMAP_DEST_CATALOG_KEY)));
        return i18nParams;
    }
}
