/*
 *  Copyright 2022 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.odf.init;

import java.util.Collections;
import java.util.Map;
import java.util.Set;

import org.apache.avalon.framework.context.Context;
import org.apache.avalon.framework.context.ContextException;
import org.apache.avalon.framework.context.Contextualizable;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;
import org.apache.cocoon.components.ContextHelper;
import org.apache.cocoon.environment.Request;

import org.ametys.cms.content.ContentHelper;
import org.ametys.cms.repository.Content;
import org.ametys.cms.rights.ContentTypeAccessController;
import org.ametys.core.group.GroupIdentity;
import org.ametys.core.right.AccessController;
import org.ametys.core.right.AccessExplanation;
import org.ametys.core.right.RightManager;
import org.ametys.core.right.RightManager.RightResult;
import org.ametys.core.user.UserIdentity;
import org.ametys.runtime.i18n.I18nizableText;
import org.ametys.runtime.plugin.component.PluginAware;
import org.ametys.runtime.plugins.admin.rights.AdminAccessController;

/**
 * Access controller to allow users with ODF_Rights_RefTableData_Sync right on context /admin to modify reference tables during initialization only.
 */
public class OdfRefTableDataSynchronizationAccessController implements AccessController, Contextualizable, Serviceable, PluginAware
{
    /** Attribute to add to activate the current access controller */
    public static final String ODF_REF_TABLE_SYNCHRONIZATION = OdfRefTableDataSynchronizationAccessController.class.getName() + "$isSynchronization";
    
    /** Right to initialize ODF reference table data */
    public static final String ODF_REF_TABLE_SYNCHRONIZATION_RIGHT = "ODF_Rights_RefTableData_Sync";
    
    private static final Set<String> __SUPPORTED_RIGHTS = Set.of("Workflow_Rights_Edition_Online");
    
    /** The helper for contents */
    protected ContentHelper _contentHelper;
    
    /** The right manager */
    protected RightManager _rightManager;

    private Context _context;
    private String _id;
    
    @Override
    public void contextualize(Context context) throws ContextException
    {
        _context = context;
    }

    public void service(ServiceManager manager) throws ServiceException
    {
        _contentHelper = (ContentHelper) manager.lookup(ContentHelper.ROLE);
        _rightManager = (RightManager) manager.lookup(RightManager.ROLE);
    }
    
    public void setPluginInfo(String pluginName, String featureName, String id)
    {
        _id = id;
    }
    
    public boolean supports(Object object)
    {
        Request request = ContextHelper.getRequest(_context);
        if (request.getAttribute(ODF_REF_TABLE_SYNCHRONIZATION) != null)
        {
            return object instanceof Content && _contentHelper.isReferenceTable((Content) object)
                    || ContentTypeAccessController.REFERENCE_TABLE_ROOT_CONTEXT.equals(object);
        }
        
        return false;
    }
    
    public AccessResult getPermission(UserIdentity user, Set<GroupIdentity> userGroups, String rightId, Object object)
    {
        if (__SUPPORTED_RIGHTS.contains(rightId))
        {
            RightResult rightResult = _rightManager.hasRight(user, ODF_REF_TABLE_SYNCHRONIZATION_RIGHT, AdminAccessController.ADMIN_RIGHT_CONTEXT);
            return switch (rightResult)
            {
                case RIGHT_ALLOW -> AccessResult.USER_ALLOWED;
                case RIGHT_DENY -> AccessResult.USER_DENIED;
                default -> AccessResult.UNKNOWN;
            };
        }
        
        return AccessResult.UNKNOWN;
    }
    
    public AccessResult getPermissionForAnyConnectedUser(String rightId, Object object)
    {
        return AccessResult.UNKNOWN;
    }
    
    public AccessResult getPermissionForAnonymous(String rightId, Object object)
    {
        return AccessResult.UNKNOWN;
    }
    
    public Map<GroupIdentity, AccessResult> getPermissionByGroup(String rightId, Object object)
    {
        return Collections.EMPTY_MAP;
    }
    
    public Map<UserIdentity, AccessResult> getPermissionByUser(String rightId, Object object)
    {
        return Collections.EMPTY_MAP;
    }
    
    public Map<String, AccessResult> getPermissionByRight(UserIdentity user, Set<GroupIdentity> userGroups, Object object)
    {
        return Collections.EMPTY_MAP;
    }
    
    public AccessResult getReadAccessPermission(UserIdentity user, Set<GroupIdentity> userGroups, Object object)
    {
        return AccessResult.UNKNOWN;
    }
    
    public Map<GroupIdentity, AccessResult> getReadAccessPermissionByGroup(Object object)
    {
        return Collections.EMPTY_MAP;
    }
    
    public Map<UserIdentity, AccessResult> getReadAccessPermissionByUser(Object object)
    {
        return Collections.EMPTY_MAP;
    }
    
    public AccessResult getReadAccessPermissionForAnyConnectedUser(Object object)
    {
        return AccessResult.UNKNOWN;
    }
    
    public AccessResult getReadAccessPermissionForAnonymous(Object object)
    {
        return AccessResult.UNKNOWN;
    }
    
    public boolean hasUserAnyReadAccessPermissionOnWorkspace(Set<Object> workspacesContexts, UserIdentity user, Set<GroupIdentity> userGroups)
    {
        return false;
    }
    
    public boolean hasAnyConnectedUserAnyReadAccessPermissionOnWorkspace(Set<Object> workspacesContexts)
    {
        return false;
    }
    
    public boolean hasAnonymousAnyReadAccessPermissionOnWorkspace(Set<Object> workspacesContexts)
    {
        return false;
    }
    
    public boolean hasUserAnyPermissionOnWorkspace(Set<Object> workspacesContexts, UserIdentity user, Set<GroupIdentity> userGroups, String rightId)
    {
        return false;
    }
    
    public boolean hasAnyConnectedUserAnyPermissionOnWorkspace(Set<Object> workspacesContexts, String rightId)
    {
        return false;
    }
    
    public boolean hasAnonymousAnyPermissionOnWorkspace(Set<Object> workspacesContexts, String rightId)
    {
        return false;
    }
    
    @Override
    public AccessExplanation explainPermission(UserIdentity user, Set<GroupIdentity> groups, String rightId, Object object)
    {
        AccessResult permission = getPermission(user, groups, rightId, object);
        if (permission == AccessResult.USER_ALLOWED)
        {
            return new AccessExplanation(getId(), permission , new I18nizableText("plugin.odf", "PLUGINS_ODF_REF_TABLE_DATA_ACCESS_CONTROLLER_ALLOWED_PERMISSION"));
            
        }
        else if (permission == AccessResult.USER_DENIED)
        {
            return new AccessExplanation(getId(), permission , new I18nizableText("plugin.odf", "PLUGINS_ODF_REF_TABLE_DATA_ACCESS_CONTROLLER_DENIED_PERMISSION"));
        }
        else // unknown
        {
            return AccessController.getDefaultAccessExplanation(getId(), permission);
        }
    }
    
    @Override
    public Map<ExplanationObject, Map<Permission, AccessExplanation>> explainAllPermissions(UserIdentity identity, Set<GroupIdentity> groups, Set<Object> workspacesContexts)
    {
        // Only used in admin context
        return Map.of();
    }
    
    public Map<Permission, AccessExplanation> explainAllPermissionsForAnonymous(Object object)
    {
        // Only used in admin context
        return Map.of();
    }
    
    public Map<Permission, AccessExplanation> explainAllPermissionsForAnyConnected(Object object)
    {
        // Only used in admin context
        return Map.of();
    }
    
    public Map<UserIdentity, Map<Permission, AccessExplanation>> explainAllPermissionsByUser(Object object)
    {
        // Only used in admin context
        return Map.of();
    }

    public Map<GroupIdentity, Map<Permission, AccessExplanation>> explainAllPermissionsByGroup(Object object)
    {
        // Only used in admin context
        return Map.of();
    }
    
    public String getId()
    {
        return _id;
    }

    public I18nizableText getObjectLabel(Object object)
    {
        // explainAllPermissions is not implemented
        throw new UnsupportedOperationException();
    }

    public I18nizableText getObjectCategory(Object object)
    {
        // explainAllPermissions is not implemented
        throw new UnsupportedOperationException();
    }
}
