/*
 *  Copyright 2010 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.odf.program;

import java.time.LocalDate;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashSet;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.stream.Collectors;

import javax.jcr.Node;

import org.apache.commons.lang3.ArrayUtils;
import org.apache.commons.lang3.StringUtils;

import org.ametys.cms.data.ContentDataHelper;
import org.ametys.cms.data.ContentValue;
import org.ametys.cms.data.Geocode;
import org.ametys.cms.data.RichText;
import org.ametys.odf.courselist.CourseList;
import org.ametys.odf.courselist.CourseListContainer;
import org.ametys.odf.data.EducationalPath;
import org.ametys.plugins.repository.data.holder.group.ModelAwareRepeater;
import org.ametys.plugins.repository.data.holder.group.ModelAwareRepeaterEntry;
import org.ametys.runtime.model.ElementDefinition;
import org.ametys.runtime.model.exception.UndefinedItemPathException;

/**
 * Abstract common superclass for {@link Program} and {@link SubProgram}.
 * @param <F> The actual type of the factory
 */
public abstract class AbstractProgram<F extends ProgramFactory> extends AbstractTraversableProgramPart<F> implements CourseListContainer
{
    // ------------------------------------------------//
    //
    // PROGRAM ATTRIBUTES
    //
    // -----------------------------------------------//
    /** Constants for attribute 'degree' */
    public static final String DEGREE = "degree";
    
    /** Constants for attribute 'certified' */
    public static final String CERTIFIED = "certified";

    /** Constants for attribute 'domain' */
    public static final String DOMAIN = "domain";
    
    /** Constants for attribute 'mention' */
    public static final String MENTION = "mention";

    /** Constants for attribute 'speciality' */
    public static final String SPECIALITY = "speciality";

    /** Constants for attribute 'educationLevel' */
    public static final String LEVEL = "educationLevel";

    /** Constants for attribute 'duration' */
    public static final String DURATION = "duration";
    
    /** Constants for attribute 'educationLanguage' */
    public static final String EDUC_LANGUAGE = "educationLanguage";

    /** Constants for attribute 'presentation' */
    public static final String PRESENTATION = "presentation";

    /** Constants for attribute 'objectives' */
    public static final String OBJECTIVES = "objectives";

    /** Constants for attribute 'qualification' */
    public static final String QUALIFICATION = "qualification";

    /** Constants for attribute 'teachingOrganization' */
    public static final String TEACHING_ORGANIZATION = "teachingOrganization";
    
    /** Constants for attribute 'apprenticeshipModalities' */
    public static final String APPRENTICESHIP_MODALITIES = "apprenticeshipModalities";

    /** Constants for attribute 'accessCondition' */
    public static final String ACCESS_CONDITION = "accessCondition";

    /** Constants for attribute 'neededPrerequisite' */
    public static final String NEEDED_PREREQUISITE = "neededPrerequisite";

    /** Constants for attribute 'recommendedPrerequisite' */
    public static final String RECOMMENDED_PREREQUISITE = "recommendedPrerequisite";

    /** Constants for attribute 'expectedResults' */
    public static final String EXPECTED_RESULTS = "expectedResults";
    
    /** Constants for attribute 'furtherStudy' */
    public static final String FURTHER_STUDY = "furtherStudy";

    /** Constants for attribute 'studyAbroad' */
    public static final String STUDY_ABROAD = "studyAbroad";

    /** Constants for attribute 'targetGroup' */
    public static final String TARGET_GROUP = "targetGroup";

    /** Constants for attribute 'jobOpportunities' */
    public static final String JOB_OPPORTUNITIES = "jobOpportunities";

    /** Constants for attribute 'trainingStrategy' */
    public static final String TRAINING_STRATEGY = "trainingStrategy";
    
    /** Constants for attribute 'knowledgeCheck' */
    public static final String KNOWLEDGE_CHECK = "knowledgeCheck";

    /** Constants for attribute 'universalAdjustment' */
    public static final String UNIVERSAL_ADJUSTMENT = "universalAdjustment";

    /** Constants for attribute 'certifying' */
    public static final String CERTIFYING = "certifying";

    /** Constants for attribute 'expenses' */
    public static final String EXPENSES = "expenses";

    /** Constants for attribute 'additionalInformations' */
    public static final String ADDITIONNAL_INFORMATIONS = "additionalInformations";

    /** Constants for attribute 'jointOrgUnit' */
    public static final String JOINT_ORGUNIT = "jointOrgUnit";

    /** Constants for attribute 'place' */
    public static final String PLACE = "place";

    /** Constants for attribute 'distanceLearning' */
    public static final String DISTANCE_LEARNING = "distanceLearning";

    /** Constants for attribute 'educationKind' */
    public static final String EDUCATION_KIND = "educationKind";

    /** Constant for attribute 'formofteachingOrg' */
    public static final String FORM_OF_TEACHING_ORG = "formofteachingOrg";

    /** Constants for attribute 'ects' */
    public static final String ECTS = "ects";

    /** Constants for attribute 'internship' */
    public static final String INTERNSHIP = "internship";

    /** Constants for attribute 'internshipDuration' */
    public static final String INTERNSHIP_DURATION = "internshipDuration";

    /** Constants for attribute 'internshipAbroad' */
    public static final String INTERNSHIP_ABROAD = "internshipAbroad";

    /** Constants for attribute 'internshipAbroadDuration' */
    public static final String INTERNSHIP_ABROAD_DURATION = "internshipAbroadDuration";
    
    /** Constants for attribute 'registrationStart' */
    public static final String REGISTRATION_START = "registrationStart";
    
    /** Constants for attribute 'registrationDeadline' */
    public static final String REGISTRATION_DEADLINE = "registrationDeadline";
    
    /** Constants for attribute 'teachingStart' */
    public static final String TEACHING_START = "teachingStart";
    
    /** Constants for attribute 'teachingEnd' */
    public static final String TEACHING_END = "teachingEnd";
    
    /** Constants for attribute 'partnerSchools' */
    public static final String PARTNER_SCHOOLS = "partnerSchools";

    /** Constants for attribute 'partnerSchools/linkUrl' */
    public static final String PARTNER_SCHOOLS_LINK_URL = "linkUrl";

    /** Constants for attribute 'partnerSchools/linkLabel' */
    public static final String PARTNER_SCHOOLS_LINK_LABEL = "linkLabel";

    /** Constants for attribute 'partnerLaboratories' */
    public static final String PARTNER_LABORATORIES = "partnerLaboratories";

    /** Constants for attribute 'partnerLaboratories/linkUrl' */
    public static final String PARTNER_LABORATORIES_LINK_URL = "linkUrl";

    /** Constants for attribute 'partnerLaboratories/linkLabel' */
    public static final String PARTNER_LABORATORIES_LINK_LABEL = "linkLabel";

    /** Constants for attribute 'rncpCode' */
    public static final String RNCP_CODE = "rncpCode";
    
    /** Constants for attribute 'rncpLevel' */
    public static final String RNCP_LEVEL = "rncpLevel";

    /** Constants for attribute 'siseCode' */
    public static final String SISE_CODE = "siseCode";

    /** Constants for attribute 'Cite 97' */
    public static final String CITE97_CODE = "cite97Code";

    /** Constants for attribute 'erasmusCode' */
    public static final String ERASMUS_CODE = "erasmusCode";
    
    /** Constants for attribute 'erasmusCode' */
    public static final String DGESIP_CODE = "dgesipCode";
    
    /** Constants for attribute 'formacode' */
    public static final String FORMACODE = "formacode";
    
    /** Constants for attribute 'romeCode' */
    public static final String ROME_CODE = "romeCode";
    
    /** Constants for attribute 'fapCode' */
    public static final String FAP_CODE = "fapCode";

    /** Constants for attribute 'nsfCode' */
    public static final String NSF_CODE = "nsfCode";

    /** Constants for attribute 'programWebSite' */
    public static final String PROGRAM_WEBSITE = "programWebSite";
    
    /** Constants for attribute 'programWebSiteLabel' */
    public static final String PROGRAM_WEBSITE_LABEL = "programWebSiteLabel";

    /** Constants for attribute 'programWebSiteUrl' */
    public static final String PROGRAM_WEBSITE_URL = "programWebSiteUrl";

    /** Constants for attribute 'attachments' */
    public static final String ATTACHMENTS = "attachments";

    /** Constants for attribute 'attachments/attachment' */
    public static final String ATTACHMENTS_ATTACHMENT = "attachment";

    /** Constants for attribute 'attachments/attachment-text' */
    public static final String ATTACHMENTS_ATTACHMENT_TEXT = "attachment-text";

    /** Constants for attribute 'numberOfStudents' */
    public static final String NUMBER_OF_STUDENTS = "numberOfStudents";
    
    /** Constants for attribute 'successRate' */
    public static final String SUCCESSRATE = "successRate";

    /** Constants for attribute 'reorientation' */
    public static final String REORIENTATION = "reorientation";
    
    /** Constants for attribute 'keywords' */
    public static final String KEYWORDS = "keywords";
    
    /** Constants for attribute 'educationEntryLevel' */
    public static final String EDUCATION_ENTRY_LEVEL = "educationEntryLevel";
    
    /** Constants for attribute 'mandatoryEntryLevel' */
    public static final String MANDATORY_ENTRY_LEVEL = "mandatoryEntryLevel";
    
    /** Constants for attribute 'programField' */
    public static final String PROGRAM_FIELD = "programField";
    
    /** Constants for attribute 'availableCertification' */
    public static final String AVAILABLE_CERTIFICATION = "availableCertification";
    
    /** Constants for attribute 'disciplines' */
    public static final String DISCIPLINES = "disciplines";
    
    /** Constants for attribute 'sectors' */
    public static final String SECTORS = "sectors";
    
    /** Constants for attribute 'internshipOpen' */
    public static final String INTERNSHIP_OPEN = "internshipOpen";

    /** Constants for attribute 'internshipDescription' */
    public static final String INTERNSHIP_DESCRIPTION = "internshipDescription";
    
    /** Constants for attribute 'internshipDescription/title' */
    public static final String INTERNSHIP_DESCRIPTION_TITLE = "title";
    
    /** Constants for attribute 'internshipDescription/duration' */
    public static final String INTERNSHIP_DESCRIPTION_DURATION = "duration";
    
    /** Constants for attribute 'internshipDescription/period' */
    public static final String INTERNSHIP_DESCRIPTION_PERIOD = "period";
    
    /** Constants for attribute 'internshipDescription/kind' */
    public static final String INTERNSHIP_DESCRIPTION_KIND = "kind";
    
    /** Constants for attribute 'apprenticeshipOpen' */
    public static final String APPRENTICESHIP_OPEN = "apprenticeshipOpen";
    
    /** Constants for attribute 'apprenticeshipPeriod' */
    public static final String APPRENTICESHIP_PERIOD = "apprenticeshipPeriod";
    
    /** Constants for attribute 'apprenticeshipContract' */
    public static final String APPRENTICESHIP_CONTRACT = "apprenticeshipContract";
    
    /** Constants for attribute 'internationalEducation' */
    public static final String INTERNATIONAL_EDUCATION = "internationalEducation";
    
    /** Constants for attribute 'internationalDimension' */
    public static final String INTERNATIONAL_DIMENSION = "internationalDimension";
    
    /** Constants for attribute 'geoCode' */
    public static final String GEOCODE = "geoCode";
    
    /** Constants for attribute 'otherPartners' */
    public static final String OTHER_PARTNERS = "otherPartners";
    
    /** Constants for attribute 'otherPartners' */
    public static final String OTHER_CONTACT = "otherContact";
    
    /** Constants for attribute 'campus' */
    public static final String CAMPUS = "campus";
    
    /** Constants for attribute 'foreignPlace' */
    public static final String FOREIGN_PLACE = "foreignPlace";
    
    /** Constants for attribute 'inscription' */
    public static final String INSCRIPTION = "inscription";
    
    /** Constants for attribute 'furtherStudyPrograms' */
    public static final String FURTHER_STUDY_PROGRAMS = "furtherStudyPrograms";
    
    /**
     * References
     */
       
    /** Constants for attribute 'contacts' */
    public static final String CONTACTS = "contacts";
    
    /** Constants for attribute 'contacts/role' */
    public static final String CONTACTS_ROLE = "role";
    
    /** Constants for attribute 'contacts/persons' */
    public static final String CONTACTS_PERSONS = "persons";

    private String _contextPath;

    private List<EducationalPath> _currentEducationalPaths;

    // ------------------------------------------------//
    //
    // PROGRAM METHODS
    //
    // -----------------------------------------------//
    /**
     * Constructor
     * @param node The JCR node
     * @param parentPath The parent path
     * @param factory The factory
     */
    public AbstractProgram(Node node, String parentPath, F factory)
    {
        super(node, parentPath, factory);
    }
    
    // --------------------------------------------------------------------------------------//
    // CONTEXT
    // --------------------------------------------------------------------------------------//
    
    // Méthodes utilisées lors du parcours d'une maquette uniquement, afin de contextualiser l'élément
    // A ne pas utiliser n'importe ou ni n'importe comment
    
    /**
     * Set the parent path for links and breadcrumb
     * @param path the parent path
     */
    public void setContextPath (String path)
    {
        _contextPath = path;
    }
    
    /**
     * Get the parent path. Can be null.
     * @return the parent path
     */
    public String getContextPath ()
    {
        return _contextPath;
    }
    
    /**
     * Set the current educational path of this abstract program
     * @param paths the current educational path
     */
    public void setCurrentEducationalPaths(List<EducationalPath> paths)
    {
        _currentEducationalPaths = paths;
    }
    
    /**
     * Get the current educational path of this abstract program
     * @return the current educational path
     */
    public List<EducationalPath> getCurrentEducationalPaths()
    {
        return _currentEducationalPaths;
    }
    
    // --------------------------------------------------------------------------------------//
    // CONTACTS
    // --------------------------------------------------------------------------------------//

    /**
     * Return the list of Persons in charge binded to this program
     * @return a list of roles and UUID
     */
    public Set<String> getContacts()
    {
        Set<String> contactIds = new HashSet<>();
        
        ModelAwareRepeater contacts = getRepeater(CONTACTS);
        if (contacts != null)
        {
            for (ModelAwareRepeaterEntry contactEntry : contacts.getEntries())
            {
                // Remove empty values
                contactIds.addAll(ContentDataHelper.getContentIdsStreamFromMultipleContentData(contactEntry, CONTACTS_PERSONS)
                        .filter(contentId -> !contentId.isEmpty())
                        .collect(Collectors.toSet()));
            }
        }

        return contactIds;
    }
    
    /**
     * Return the list of Persons in charge binded to this program
     * @return a list of roles and UUID
     */
    public Map<String, List<String>> getContactsByRole()
    {
        Map<String, List<String>> contactsByRole = new LinkedHashMap<>();
        
        ModelAwareRepeater contacts = getRepeater(CONTACTS);
        if (contacts != null)
        {
            for (ModelAwareRepeaterEntry contactEntry : contacts.getEntries())
            {
                // Remove empty values
                List<String> persons = ContentDataHelper.getContentIdsStreamFromMultipleContentData(contactEntry, CONTACTS_PERSONS)
                                                    .filter(contentId -> !contentId.isEmpty())
                                                    .collect(Collectors.toList());
                if (!persons.isEmpty())
                {
                    String role = ContentDataHelper.getContentIdFromContentData(contactEntry, CONTACTS_ROLE);
                    List<String> personList = contactsByRole.getOrDefault(role, new ArrayList<>());
                    personList.addAll(persons);
                    contactsByRole.put(role, personList);
                }
            }
        }

        return contactsByRole;
    }

    // --------------------------------------------------------------------------------------//
    // GETTERS AND SETTERS
    // --------------------------------------------------------------------------------------//
    /**
     * Returns <code>true</code> if the {@link AbstractProgram} is certified
     * @return <code>true</code> if certified or default value (true in the kernel) if not defined
     */
    public boolean isCertified()
    {
        return getValue(CERTIFIED, true, true);
    }
    
    /**
     * Get the domain
     * @return the domain or an empty array
     */
    public String[] getDomain()
    {
        try
        {
            ElementDefinition definition = (ElementDefinition) getDefinition(DOMAIN);
            if (definition.isMultiple())
            {
                return ContentDataHelper.getContentIdsArrayFromMultipleContentData(this, DOMAIN);
            }
            else
            {
                return new String[] {ContentDataHelper.getContentIdFromContentData(this, DOMAIN)};
            }
        }
        catch (UndefinedItemPathException e)
        {
            return null; // this attribute is not part of model
        }
        
    }

    /**
     * Get the education presentation
     * @return the education presentation or null if not set
     */
    public RichText getPresentation()
    {
        try
        {
            return getValue(PRESENTATION);
        }
        catch (UndefinedItemPathException e)
        {
            return null; // this attribute is not part of model
        }
    }

    /**
     * Get the education objectives
     * @return the education objectives or null if not set
     */
    public RichText getObjectives()
    {
        try
        {
            return getValue(OBJECTIVES);
        }
        catch (UndefinedItemPathException e)
        {
            return null; // this attribute is not part of model
        }
    }

    /**
     * Get the education qualification
     * @return the education qualification or null if not set
     */
    public RichText getQualification()
    {
        try
        {
            return getValue(QUALIFICATION);
        }
        catch (UndefinedItemPathException e)
        {
            return null; // this attribute is not part of model
        }
    }

    /**
     * Get the education organization
     * @return the education organization or null if not set
     */
    public RichText getTeachingOrganization()
    {
        try
        {
            return getValue(TEACHING_ORGANIZATION);
        }
        catch (UndefinedItemPathException e)
        {
            return null; // this attribute is not part of model
        }
    }
    
    /**
     * Get the apprenticeship modalities
     * @return the apprenticeship modalities or null if not set
     */
    public RichText getApprenticeshipModalities()
    {
        try
        {
            return getValue(APPRENTICESHIP_MODALITIES);
        }
        catch (UndefinedItemPathException e)
        {
            return null; // this attribute is not part of model
        }
    }

    /**
     * Get the access conditions
     * @return  the access conditions  or null if not set
     */
    public RichText getAccessCondition()
    {
        try
        {
            return getValue(ACCESS_CONDITION);
        }
        catch (UndefinedItemPathException e)
        {
            return null; // this attribute is not part of model
        }
    }

    /**
     * Get the education requirements
     * @return the education requirements or null if not set
     */
    public RichText getNeededPrerequisite()
    {
        try
        {
            return getValue(NEEDED_PREREQUISITE);
        }
        catch (UndefinedItemPathException e)
        {
            return null; // this attribute is not part of model
        }
    }

    /**
     * Get the education recommended prerequisite
     * @return the education recommended prerequisite or null if not set
     */
    public RichText getRecommendedPrerequisite()
    {
        try
        {
            return getValue(RECOMMENDED_PREREQUISITE);
        }
        catch (UndefinedItemPathException e)
        {
            return null; // this attribute is not part of model
        }
    }
    
    /**
     * Get the education expected results
     * @return the education expected results or null if not set
     */
    public RichText getExpectedResults()
    {
        try
        {
            return getValue(EXPECTED_RESULTS);
        }
        catch (UndefinedItemPathException e)
        {
            return null; // this attribute is not part of model
        }
    }

    /**
     * Get the education further study
     * @return the education further study or null if not set
     */
    public RichText getFurtherStudy()
    {
        try
        {
            return getValue(FURTHER_STUDY);
        }
        catch (UndefinedItemPathException e)
        {
            return null; // this attribute is not part of model
        }
    }

    /**
     * Get the education study abroard
     * @return the education study abroard or null if not set
     */
    public RichText getStudyAbroad()
    {
        try
        {
            return getValue(STUDY_ABROAD);
        }
        catch (UndefinedItemPathException e)
        {
            return null; // this attribute is not part of model
        }
    }

    /**
     * Get the education target
     * @return the education target or null if not set
     */
    public RichText getTargetGroup()
    {
        try
        {
            return getValue(TARGET_GROUP);
        }
        catch (UndefinedItemPathException e)
        {
            return null; // this attribute is not part of model
        }
    }

    /**
     * Get the job opportunities
     * @return the job opportunities or null if not set
     */
    public RichText getJobOpportunities()
    {
        try
        {
            return getValue(JOB_OPPORTUNITIES);
        }
        catch (UndefinedItemPathException e)
        {
            return null; // this attribute is not part of model
        }
    }
    
    /**
     * Get the training strategy
     * @return the training strategy or null if not set
     */
    public RichText getTrainingStrategy()
    {
        try
        {
            return getValue(TRAINING_STRATEGY);
        }
        catch (UndefinedItemPathException e)
        {
            return null; // this attribute is not part of model
        }
    }
    
    /**
     * Get acknowledgments
     * @return acknowledgments or null if not set
     */
    public RichText getKnowledgeCheck()
    {
        try
        {
            return getValue(KNOWLEDGE_CHECK);
        }
        catch (UndefinedItemPathException e)
        {
            return null; // this attribute is not part of model
        }
    }

    /**
     * Get universal adjustment
     * @return universal adjustment or null if not set
     */
    public RichText getUniversalAdjustment()
    {
        try
        {
            return getValue(UNIVERSAL_ADJUSTMENT);
        }
        catch (UndefinedItemPathException e)
        {
            return null; // this attribute is not part of model
        }
    }
    
    /**
     * Is certifying
     * @return <code>true</code> if the progam is certifying
     */
    public boolean isCertifying()
    {
        try
        {
            return getValue(CERTIFYING, false, false);
        }
        catch (UndefinedItemPathException e)
        {
            return false; // this attribute is not part of model
        }
    }

    /**
     * Get the additional informations
     * @return the additional informations or null if not set
     */
    public RichText getAdditionalInformations()
    {
        try
        {
            return getValue(ADDITIONNAL_INFORMATIONS);
        }
        catch (UndefinedItemPathException e)
        {
            return null; // this attribute is not part of model
        }
    }

    /**
     * Get the education level
     * @return the education level or null
     */
    public String getEducationLevel()
    {
        try
        {
            return ContentDataHelper.getContentIdFromContentData(this, LEVEL);
        }
        catch (UndefinedItemPathException e)
        {
            return null; // this attribute is not part of model
        }
    }

    /**
     * Get the RNCP code
     * @return the RNCP code
     */
    public String[] getRncpCode()
    {
        try
        {
            return getValue(RNCP_CODE, false, ArrayUtils.EMPTY_STRING_ARRAY);
        }
        catch (UndefinedItemPathException e)
        {
            return ArrayUtils.EMPTY_STRING_ARRAY; // this attribute is not part of model
        }
    }
    
    /**
     * Get the RNCP level
     * @return the RNCP level
     */
    public String[] getRncpLevel()
    {
        try
        {
            return ContentDataHelper.getContentIdsArrayFromMultipleContentData(this, RNCP_LEVEL);
        }
        catch (UndefinedItemPathException e)
        {
            return ArrayUtils.EMPTY_STRING_ARRAY; // this attribute is not part of model
        }
    }
    
    /**
     * Get the SISE code
     * @return the SISE code
     */
    public String[] getSiseCode()
    {
        try
        {
            return ContentDataHelper.getContentIdsArrayFromMultipleContentData(this, SISE_CODE);
        }
        catch (UndefinedItemPathException e)
        {
            return ArrayUtils.EMPTY_STRING_ARRAY; // this attribute is not part of model
        }
    }

    /**
     * Get the CITE97 code
     * @return the CITE97 code
     */
    public String[] getCite97Code()
    {
        try
        {
            return ContentDataHelper.getContentIdsArrayFromMultipleContentData(this, CITE97_CODE);
        }
        catch (UndefinedItemPathException e)
        {
            return ArrayUtils.EMPTY_STRING_ARRAY; // this attribute is not part of model
        }
    }

    /**
     * Get the DGESIP code
     * @return the DGESIP code
     */
    public String[] getDGESIPCode()
    {
        try
        {
            return ContentDataHelper.getContentIdsArrayFromMultipleContentData(this, DGESIP_CODE);
        }
        catch (UndefinedItemPathException e)
        {
            return ArrayUtils.EMPTY_STRING_ARRAY; // this attribute is not part of model
        }
    }

    /**
     * Get the Erasmus code
     * @return the Erasmus code
     */
    public String[] getErasmusCode()
    {
        try
        {
            return ContentDataHelper.getContentIdsArrayFromMultipleContentData(this, ERASMUS_CODE);
        }
        catch (UndefinedItemPathException e)
        {
            return ArrayUtils.EMPTY_STRING_ARRAY; // this attribute is not part of model
        }
    }

    /**
     * Get the FORMACODE
     * @return the FORMACODE
     */
    public String[] getFORMACODE()
    {
        try
        {
            return getValue(FORMACODE, false, ArrayUtils.EMPTY_STRING_ARRAY);
        }
        catch (UndefinedItemPathException e)
        {
            return ArrayUtils.EMPTY_STRING_ARRAY; // this attribute is not part of model
        }
    }
    
    /**
     * Get the ROME code
     * @return the ROME code
     */
    public String[] getRomeCode()
    {
        try
        {
            return ContentDataHelper.getContentIdsArrayFromMultipleContentData(this, ROME_CODE);
        }
        catch (UndefinedItemPathException e)
        {
            return ArrayUtils.EMPTY_STRING_ARRAY; // this attribute is not part of model
        }
    }
    
    /**
     * Get the FAP code
     * @return the FAP code
     */
    public String[] getFapCode()
    {
        try
        {
            return ContentDataHelper.getContentIdsArrayFromMultipleContentData(this, FAP_CODE);
        }
        catch (UndefinedItemPathException e)
        {
            return ArrayUtils.EMPTY_STRING_ARRAY; // this attribute is not part of model
        }
    }
    
    /**
     * Get the NSF code
     * @return the NSF code
     */
    public String getNSFCode()
    {
        try
        {
            return ContentDataHelper.getContentIdFromContentData(this, NSF_CODE);
        }
        catch (UndefinedItemPathException e)
        {
            return null; // this attribute is not part of model
        }
    }
    
    /**
     * Get the mention
     * @return the mention or null
     */
    public String getMention()
    {
        try
        {
            return ContentDataHelper.getContentIdFromContentData(this, MENTION);
        }
        catch (UndefinedItemPathException e)
        {
            return null; // this attribute is not part of model
        }
    }

    /**
     * Get the speciality
     * @return the speciality or null
     */
    public String getSpeciality()
    {
        try
        {
            return getValue(SPECIALITY, false, StringUtils.EMPTY);
        }
        catch (UndefinedItemPathException e)
        {
            return null; // this attribute is not part of model
        }
    }

    /**
     * Get the org units
     * @return the org units
     */
    public String[] getJointOrgUnit()
    {
        try
        {
            return ContentDataHelper.getContentIdsArrayFromMultipleContentData(this, JOINT_ORGUNIT);
        }
        catch (UndefinedItemPathException e)
        {
            return ArrayUtils.EMPTY_STRING_ARRAY; // this attribute is not part of model
        }
    }

    /**
     * Get the places
     * @return the places as an Array of String
     */
    public String[] getPlace()
    {
        try
        {
            return ContentDataHelper.getContentIdsArrayFromMultipleContentData(this, PLACE);
        }
        catch (UndefinedItemPathException e)
        {
            return ArrayUtils.EMPTY_STRING_ARRAY; // this attribute is not part of model
        }
    }

    /**
     * Get the list of websites
     * @return the list of website or an empty list
     */
    public Set<WebsiteLink> getWebsiteLinks()
    {
        Set<WebsiteLink> websites = new HashSet<>();
        
        try
        {
            ModelAwareRepeater webSites = getRepeater(PROGRAM_WEBSITE);
            if (webSites != null)
            {
                for (ModelAwareRepeaterEntry entry : webSites.getEntries())
                {
                    WebsiteLink website = new WebsiteLink(entry.getValue(PROGRAM_WEBSITE_URL, false, StringUtils.EMPTY), entry.getValue(PROGRAM_WEBSITE_LABEL, false, StringUtils.EMPTY));
                    websites.add(website);
                }
            }
        }
        catch (UndefinedItemPathException e)
        {
            // this attribute is not part of model
        }
        
        return websites;
    }

    /**
     * Get the ECTS credits
     * @return the ECTS credits
     */
    public String getEcts()
    {
        try
        {
            return ContentDataHelper.getContentIdFromContentData(this, ECTS);
        }
        catch (UndefinedItemPathException e)
        {
            return null; // this attribute is not part of model
        }
    }

    /**
     * Get the kind of education
     * @return the kind of education or null
     */
    public String getEducationKind()
    {
        try
        {
            return ContentDataHelper.getContentIdFromContentData(this, EDUCATION_KIND);
        }
        catch (UndefinedItemPathException e)
        {
            return null; // this attribute is not part of model
        }
    }
    
    /**
     * Get the duration
     * @return the duration
     */
    public String getDuration()
    {
        try
        {
            return ContentDataHelper.getContentIdFromContentData(this, DURATION);
        }
        catch (UndefinedItemPathException e)
        {
            return null; // this attribute is not part of model
        }
    }

    /**
     * Get the education languages
     * @return the education languages
     */
    public String[] getEducationLanguage()
    {
        try
        {
            return ContentDataHelper.getContentIdsArrayFromMultipleContentData(this, EDUC_LANGUAGE);
        }
        catch (UndefinedItemPathException e)
        {
            return ArrayUtils.EMPTY_STRING_ARRAY; // this attribute is not part of model
        }
    }

    /**
     * Get the effectives
     * @return the effectives
     */
    public RichText getEffectives()
    {
        try
        {
            return getValue(NUMBER_OF_STUDENTS);
        }
        catch (UndefinedItemPathException e)
        {
            return null; // this attribute is not part of model
        }
    }
    
    /**
     * Get the SuccessRate
     * @return the SuccessRate
     */
    public String getSuccessRate()
    {
        try
        {
            return getValue(SUCCESSRATE, false, StringUtils.EMPTY);
        }
        catch (UndefinedItemPathException e)
        {
            return null; // this attribute is not part of model
        }
    }
    
    /**
     * Get the expenses
     * @return the expenses
     */
    public RichText getExpenses()
    {
        try
        {
            return getValue(EXPENSES);
        }
        catch (UndefinedItemPathException e)
        {
            return null; // this attribute is not part of model
        }
    }
    
    /**
     * Get the form of teaching organization
     * @return the form of teaching organization
     */
    public String[] getFormOfTeachingOrgs()
    {
        try
        {
            return ContentDataHelper.getContentIdsArrayFromMultipleContentData(this, FORM_OF_TEACHING_ORG);
        }
        catch (UndefinedItemPathException e)
        {
            return ArrayUtils.EMPTY_STRING_ARRAY; // this attribute is not part of model
        }
    }
    
    /**
     * Get the REORIENTATION
     * @return RichText
     */
    public RichText getReorientation()
    {
        try
        {
            return getValue(REORIENTATION);
        }
        catch (UndefinedItemPathException e)
        {
            return null; // this attribute is not part of model
        }
    }
    
    /**
     * Get the distance learning
     * @return the distance learning or null
     */
    public String getDistanceLearning()
    {
        try
        {
            return ContentDataHelper.getContentIdFromContentData(this, DISTANCE_LEARNING);
        }
        catch (UndefinedItemPathException e)
        {
            return null; // this attribute is not part of model
        }
    }

    /**
     * Get the internship
     * @return the internship or null
     */
    public String getInternship()
    {
        try
        {
            return ContentDataHelper.getContentIdFromContentData(this, INTERNSHIP);
        }
        catch (UndefinedItemPathException e)
        {
            return null; // this attribute is not part of model
        }
    }

    /**
     * Get the internship duration
     * @return the internship duration or null
     */
    public String getInternshipDuration()
    {
        try
        {
            return getValue(INTERNSHIP_DURATION, false, StringUtils.EMPTY);
        }
        catch (UndefinedItemPathException e)
        {
            return null; // this attribute is not part of model
        }
    }

    /**
     * Get the internship abroad
     * @return the internship abroad or null
     */
    public String getInternshipAbroad()
    {
        try
        {
            return ContentDataHelper.getContentIdFromContentData(this, INTERNSHIP_ABROAD);
        }
        catch (UndefinedItemPathException e)
        {
            return null; // this attribute is not part of model
        }
    }

    /**
     * Get the internship abroad duration
     * @return  the internship abroad duration or null
     */
    public String getAbroadInternshipDuration()
    {
        try
        {
            return getValue(INTERNSHIP_ABROAD_DURATION, false, StringUtils.EMPTY);
        }
        catch (UndefinedItemPathException e)
        {
            return null; // this attribute is not part of model
        }
    }

    /**
     * Get the registration start date.
     * @return The registration start date, can be null.
     */
    public LocalDate getRegistrationStart()
    {
        try
        {
            return getValue(REGISTRATION_START);
        }
        catch (UndefinedItemPathException e)
        {
            return null; // this attribute is not part of model
        }
    }
    
    /**
     * Get the registration deadline date.
     * @return The registration deadline date, can be null.
     */
    public LocalDate getRegistrationDeadline()
    {
        try
        {
            return getValue(REGISTRATION_DEADLINE);
        }
        catch (UndefinedItemPathException e)
        {
            return null; // this attribute is not part of model
        }
    }
    
    /**
     * Get the teaching start date.
     * @return The teaching start date, can be null.
     */
    public LocalDate getTeachingStart()
    {
        try
        {
            return getValue(TEACHING_START);
        }
        catch (UndefinedItemPathException e)
        {
            return null; // this attribute is not part of model
        }
    }
    
    /**
     * Get the teaching end date.
     * @return The teaching end date, can be null.
     */
    public LocalDate getTeachingEnd()
    {
        try
        {
            return getValue(TEACHING_END);
        }
        catch (UndefinedItemPathException e)
        {
            return null; // this attribute is not part of model
        }
    }
    
    /**
     * Get the keywords
     * @return the keywords
     */
    public String[] getKeywords()
    {
        try
        {
            return getValue(KEYWORDS, false, ArrayUtils.EMPTY_STRING_ARRAY);
        }
        catch (UndefinedItemPathException e)
        {
            return ArrayUtils.EMPTY_STRING_ARRAY; // this attribute is not part of model
        }
    }

    /**
     * Get the education level entry
     * @return the education level entry
     */
    public String[] getEducationLevelEntry()
    {
        try
        {
            return ContentDataHelper.getContentIdsArrayFromMultipleContentData(this, EDUCATION_ENTRY_LEVEL);
        }
        catch (UndefinedItemPathException e)
        {
            return ArrayUtils.EMPTY_STRING_ARRAY; // this attribute is not part of model
        }
    }
    
    /**
     * Is the entry level mandatory
     * @return <code>true</code> if the entry level is mandatory
     */
    public boolean isMandatoryEntryLevel()
    {
        try
        {
            return getValue(MANDATORY_ENTRY_LEVEL, false, false);
        }
        catch (UndefinedItemPathException e)
        {
            return false; // this attribute is not part of model
        }
    }
    
    /**
     * Get the program fields
     * @return the program fields
     */
    public String[] getProgramField()
    {
        try
        {
            return ContentDataHelper.getContentIdsArrayFromMultipleContentData(this, PROGRAM_FIELD);
        }
        catch (UndefinedItemPathException e)
        {
            return ArrayUtils.EMPTY_STRING_ARRAY; // this attribute is not part of model
        }
    }
    
    /**
     * Get the available certifications
     * @return the available certifications
     */
    public String[] getAvailableCertification()
    {
        try
        {
            return ContentDataHelper.getContentIdsArrayFromMultipleContentData(this, AVAILABLE_CERTIFICATION);
        }
        catch (UndefinedItemPathException e)
        {
            return ArrayUtils.EMPTY_STRING_ARRAY; // this attribute is not part of model
        }
    }
    
    /**
     * Get the disciplines
     * @return the disciplines
     */
    public String[] getDisciplines()
    {
        try
        {
            return ContentDataHelper.getContentIdsArrayFromMultipleContentData(this, DISCIPLINES);
        }
        catch (UndefinedItemPathException e)
        {
            return ArrayUtils.EMPTY_STRING_ARRAY; // this attribute is not part of model
        }
    }
    
    /**
     * Get the sectors
     * @return the sectors
     */
    public String[] getSectors()
    {
        try
        {
            return ContentDataHelper.getContentIdsArrayFromMultipleContentData(this, SECTORS);
        }
        catch (UndefinedItemPathException e)
        {
            return ArrayUtils.EMPTY_STRING_ARRAY; // this attribute is not part of model
        }
    }

    /**
     * Is intership open
     * @return <code>true</code> if internship is open
     */
    public boolean isInternshipOpen()
    {
        try
        {
            return getValue(INTERNSHIP_OPEN, false, false);
        }
        catch (UndefinedItemPathException e)
        {
            return false; // this attribute is not part of model
        }
    }

    /**
     * Is apprenticeship open
     * @return <code>true</code> if apprenticeship is open
     */
    public boolean isApprenticeshipOpen()
    {
        try
        {
            return getValue(APPRENTICESHIP_OPEN, false, false);
        }
        catch (UndefinedItemPathException e)
        {
            return false; // this attribute is not part of model
        }
    }

    /**
     * Get the apprenticeship period description
     * @return the apprenticeship period
     */
    public RichText getApprenticeshipPeriod()
    {
        try
        {
            return getValue(APPRENTICESHIP_PERIOD);
        }
        catch (UndefinedItemPathException e)
        {
            return null; // this attribute is not part of model
        }
    }
    
    /**
     * Get the available apprenticeship contracts
     * @return the apprenticeship contracts
     */
    public String[] getApprenticeshipContract()
    {
        try
        {
            return ContentDataHelper.getContentIdsArrayFromMultipleContentData(this, APPRENTICESHIP_CONTRACT);
        }
        catch (UndefinedItemPathException e)
        {
            return ArrayUtils.EMPTY_STRING_ARRAY; // this attribute is not part of model
        }
    }

    /**
     * Get the international education
     * @return the international education
     */
    public String[] getInternationalEducation()
    {
        try
        {
            return ContentDataHelper.getContentIdsArrayFromMultipleContentData(this, INTERNATIONAL_EDUCATION);
        }
        catch (UndefinedItemPathException e)
        {
            return ArrayUtils.EMPTY_STRING_ARRAY; // this attribute is not part of model
        }
    }

    /**
     * Get the international dimension
     * @return the international dimension
     */
    public RichText getInternationalDimension()
    {
        try
        {
            return getValue(INTERNATIONAL_DIMENSION);
        }
        catch (UndefinedItemPathException e)
        {
            return null; // this attribute is not part of model
        }
    }
    
    /**
     * Get the geocode latitude and longitude
     * @return the geocode
     */
    public Geocode getGeocode()
    {
        try
        {
            return getValue(GEOCODE);
        }
        catch (UndefinedItemPathException e)
        {
            return null; // this attribute is not part of model
        }
    }
    
    /**
     * Get the other partners
     * @return the other partners
     */
    public RichText getOtherPartners()
    {
        try
        {
            return getValue(OTHER_PARTNERS);
        }
        catch (UndefinedItemPathException e)
        {
            return null; // this attribute is not part of model
        }
    }
    
    /**
     * Get the campus
     * @return the campus
     */
    public String[] getCampus()
    {
        try
        {
            return ContentDataHelper.getContentIdsArrayFromMultipleContentData(this, CAMPUS);
        }
        catch (UndefinedItemPathException e)
        {
            return ArrayUtils.EMPTY_STRING_ARRAY; // this attribute is not part of model
        }
    }

    /**
     * Get the foreign places
     * @return the foreign places
     */
    public String[] getForeignPlace()
    {
        try
        {
            return ContentDataHelper.getContentIdsArrayFromMultipleContentData(this, FOREIGN_PLACE);
        }
        catch (UndefinedItemPathException e)
        {
            return ArrayUtils.EMPTY_STRING_ARRAY; // this attribute is not part of model
        }
    }
    
    /**
     * Get the inscription
     * @return the inscription
     */
    public RichText getInscription()
    {
        try
        {
            return getValue(INSCRIPTION);
        }
        catch (UndefinedItemPathException e)
        {
            return null; // this attribute is not part of model
        }
    }
    
    /**
     * Get the further study programs
     * @return the further study programs
     */
    public String[] getFurtherStudyPrograms()
    {
        try
        {
            return ContentDataHelper.getContentIdsArrayFromMultipleContentData(this, FURTHER_STUDY_PROGRAMS);
        }
        catch (UndefinedItemPathException e)
        {
            return ArrayUtils.EMPTY_STRING_ARRAY; // this attribute is not part of model
        }
    }
    
    // --------------------------------------------------------------------------------------//
    // Methods of CourseListContainer
    // --------------------------------------------------------------------------------------//
    @Override
    public List<CourseList> getCourseLists()
    {
        return Arrays.stream(getValue(CHILD_PROGRAM_PARTS, false, new ContentValue[0]))
                .map(ContentValue::getContentIfExists)
                .flatMap(Optional::stream)
                // This cast is not checked because an exception must be thrown if the retrieved content is not a program part
                // TODO: change this behavior to throw our own exception and not a CassCastException
                .map(ProgramPart.class::cast)
                // Program parts that are not course lists are simply ignored
                .filter(CourseList.class::isInstance)
                .map(CourseList.class::cast)
                .collect(Collectors.toList());
    }
    
    @Override
    public boolean containsCourseList(String clId)
    {
        return ArrayUtils.contains(ContentDataHelper.getContentIdsArrayFromMultipleContentData(this, CHILD_PROGRAM_PARTS), clId);
    }

    @Override
    public boolean hasCourseLists()
    {
        return !getCourseLists().isEmpty();
    }
    
    // --------------------------------------------------------------------------------------//
    // CDMfr
    // --------------------------------------------------------------------------------------//
    @Override
    protected String getCDMType()
    {
        return "PR";
    }
    
    /**
     * Returns the surrounding tag name in the CDM-fr representation.
     * @return the surrounding tag name
     */
    public abstract String getCDMTagName();
}
