/*
 *  Copyright 2019 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.odf.catalog;

import java.util.Map;
import java.util.Optional;
import java.util.stream.Stream;

import org.apache.commons.lang3.StringUtils;

import org.ametys.cms.contenttype.ContentAttributeDefinition;
import org.ametys.cms.data.ContentValue;
import org.ametys.cms.repository.Content;
import org.ametys.plugins.repository.data.holder.ModifiableModelAwareDataHolder;
import org.ametys.plugins.repository.data.holder.group.ModifiableModelAwareComposite;
import org.ametys.plugins.repository.data.holder.group.ModifiableModelAwareRepeater;
import org.ametys.plugins.repository.data.holder.group.ModifiableModelAwareRepeaterEntry;
import org.ametys.plugins.repository.model.CompositeDefinition;
import org.ametys.plugins.repository.model.RepeaterDefinition;
import org.ametys.runtime.model.ModelItem;
import org.ametys.runtime.plugin.component.AbstractLogEnabled;

/**
 * The abstract class to copy content attribute of program item type.
 */
public abstract class AbstractProgramItemAttributeCopyUpdater extends AbstractLogEnabled implements CopyCatalogUpdater
{
    /**
     * Update the program item attribute with the value of the new catalog.
     * @param dataHolder The data holder
     * @param definitionPath The definition path
     * @param copiedContents The copied contents for the referenced attribute
     */
    protected void _updateContentAttribute(ModifiableModelAwareDataHolder dataHolder, String definitionPath, Map<Content, Content> copiedContents)
    {
        String[] pathSegments = StringUtils.split(definitionPath, ModelItem.ITEM_PATH_SEPARATOR);
        String attributeName = pathSegments[0];
        ModelItem definition = dataHolder.getDefinition(attributeName);
        if (pathSegments.length == 1 && definition instanceof ContentAttributeDefinition)
        {
            if (((ContentAttributeDefinition) definition).isMultiple())
            {
                _updateMultipleContentAttribute(dataHolder, attributeName, copiedContents);
            }
            else
            {
                _updateSingleContentAttribute(dataHolder, attributeName, copiedContents);
            }
        }
        else if (definition instanceof RepeaterDefinition)
        {
            ModifiableModelAwareRepeater repeater = dataHolder.getRepeater(attributeName);
            if (repeater != null)
            {
                String childDefinitionPath = StringUtils.join(pathSegments, ModelItem.ITEM_PATH_SEPARATOR, 1, pathSegments.length);
                for (ModifiableModelAwareRepeaterEntry entry : repeater.getEntries())
                {
                    _updateContentAttribute(entry, childDefinitionPath, copiedContents);
                }
            }
        }
        else if (definition instanceof CompositeDefinition)
        {
            ModifiableModelAwareComposite composite = dataHolder.getComposite(attributeName);
            if (composite != null)
            {
                String childDefinitionPath = StringUtils.join(pathSegments, ModelItem.ITEM_PATH_SEPARATOR, 1, pathSegments.length);
                _updateContentAttribute(composite, childDefinitionPath, copiedContents);
            }
        }
        else
        {
            // This method shouldn't be called with anything else than a final content attribute
            throw new IllegalArgumentException("The path '" + definitionPath + "' on data holder '" + definition.getPath() + "' is not a content attribute on content type '" + definition.getModel().getId() + "'.");
        }
    }

    /**
     * Update the single attribute with the content value.
     * @param dataHolder The data holder
     * @param attributeName The attribute name
     * @param copiedContents The copied contents for the referenced attribute
     */
    protected void _updateSingleContentAttribute(ModifiableModelAwareDataHolder dataHolder, String attributeName, Map<Content, Content> copiedContents)
    {
        Content newContent = Optional.of(dataHolder)
            // Get the attribute value
            .map(dh -> dh.<ContentValue>getValue(attributeName))
            // Get the content
            .flatMap(ContentValue::getContentIfExists)
            // Keep himself if not found in the copied contents map
            .map(content -> copiedContents.getOrDefault(content, content))
            // Get the retrieved value or null
            .orElse(null);

        // Set the attribute with the new content
        // If null, the attribute value will be empty
        dataHolder.setValue(attributeName, newContent);
    }
    
    /**
     * Update the multiple attribute with the program item value.
     * @param dataHolder The data holder
     * @param attributeName The attribute name
     * @param copiedContents The copied contents for the referenced attribute
     */
    protected void _updateMultipleContentAttribute(ModifiableModelAwareDataHolder dataHolder, String attributeName, Map<Content, Content> copiedContents)
    {
        Content[] newContents = Optional.of(dataHolder)
            // Get the attribute values
            .map(dh -> dh.<ContentValue[]>getValue(attributeName))
            // Build a stream from the ContentValue array
            .map(Stream::of)
            // Or build an empty stream if there is no values
            .orElseGet(Stream::empty)
            // For each element, get the content
            .map(ContentValue::getContentIfExists)
            .flatMap(Optional::stream)
            // Keep himself if not found in the copied contents map
            .map(content -> copiedContents.getOrDefault(content, content))
            // Collect into an array
            .toArray(Content[]::new);
        
        // Set the attribute with the new contents list
        dataHolder.setValue(attributeName, newContents);
    }
}
