/*
 *  Copyright 2025 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.odf.rights;

import java.util.HashSet;
import java.util.List;
import java.util.Objects;
import java.util.Set;
import java.util.stream.Collectors;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;

import org.ametys.cms.content.ContentHelper;
import org.ametys.cms.repository.Content;
import org.ametys.core.right.AccessController;
import org.ametys.core.right.RightsException;
import org.ametys.odf.ODFHelper;
import org.ametys.odf.ProgramItem;
import org.ametys.odf.data.EducationalPath;
import org.ametys.odf.orgunit.OrgUnit;
import org.ametys.odf.rights.ODFRightHelper.ContextualizedContent;
import org.ametys.odf.rights.ODFRightHelper.ContextualizedPermissionContext;
import org.ametys.odf.tree.ODFContentsTreeHelper;
import org.ametys.plugins.core.impl.right.AbstractHierarchicalWithPermissionContextAccessController;
import org.ametys.plugins.repository.AmetysObjectResolver;
import org.ametys.runtime.i18n.I18nizableText;

/**
 * {@link AccessController} for a ODF {@link ContextualizedContent} based on stored ACL
 */
public class ODFContextualizedContentHierarchicalAccessController extends AbstractHierarchicalWithPermissionContextAccessController<ContextualizedContent, ODFRightHelper.ContextualizedPermissionContext>
{
    /** the odf context category */
    public static final I18nizableText ODF_CONTEXT_CATEGORY = new I18nizableText("plugin.odf", "PLUGINS_ODF_RIGHT_ASSIGNMENT_CONTEXT_CONTENTS_LABEL");
    /** The helper for root content */
    protected ODFHelper _odfHelper;
    /** The helper for contents */
    protected ContentHelper _contentHelper;
    /** Ametys Object Resolver */
    protected AmetysObjectResolver _resolver;
    /** The right helper for ODF contents */
    protected ODFRightHelper _odfRightHelper;
    /** The ODF contents tree helper */
    protected ODFContentsTreeHelper _odfContentTreeHelper;
    
    @Override
    public void service(ServiceManager manager) throws ServiceException
    {
        super.service(manager);
        _odfContentTreeHelper = (ODFContentsTreeHelper) manager.lookup(ODFContentsTreeHelper.ROLE);
        _odfHelper = (ODFHelper) manager.lookup(ODFHelper.ROLE);
        _odfRightHelper = (ODFRightHelper) manager.lookup(ODFRightHelper.ROLE);
        _contentHelper = (ContentHelper) manager.lookup(ContentHelper.ROLE);
        _resolver = (AmetysObjectResolver) manager.lookup(AmetysObjectResolver.ROLE);
    }
    
    public boolean supports(Object object)
    {
        return object instanceof ContextualizedContent;
    }
    
    @Override
    protected Set<ContextualizedContent> _getParents(ContextualizedContent object, ContextualizedPermissionContext permissionCtx)
    {
        Content content = object.content();
        
        if (content instanceof ProgramItem programItem)
        {
            EducationalPath educationalPath = permissionCtx.getEducationalPath();
            if (educationalPath == null)
            {
                return Set.of();
            }
            
            List<ProgramItem> programItemsInPath = educationalPath.getProgramItems(_resolver);
            
            ProgramItem parent = programItemsInPath.getLast();
            
            // Update educational path in permission context
            permissionCtx.withEducationalPath(_removeLast(educationalPath));
            
            Set<ContextualizedContent> parents = new HashSet<>();
            parents.add(new ContextualizedContent((Content) parent, permissionCtx.getEducationalPath()));
            
            // Add orgunits 
            List<String> ouIds = programItem.getOrgUnits();
            parents.addAll(ouIds.stream()
                    .filter(Objects::nonNull)
                    .filter(_resolver::hasAmetysObjectForId)
                    .map(_resolver::resolveById)
                    .map(OrgUnit.class::cast)
                    .map(ou -> new ContextualizedContent(ou, null))
                    .collect(Collectors.toSet()));
            
            return Set.of(new ContextualizedContent((Content) parent, permissionCtx.getEducationalPath()));
        }
        else if (content instanceof OrgUnit ou)
        {
            OrgUnit parentOrgUnit = ou.getParentOrgUnit();
            if (parentOrgUnit != null)
            {
                return Set.of(new ContextualizedContent(parentOrgUnit, null));
            }
        }
        
        return Set.of();
    }
    
    @Override
    protected ContextualizedPermissionContext _getPermissionContext(ContextualizedContent contextualizedContent)
    {
        EducationalPath educationalPath = contextualizedContent.path();
        Content initialContent = contextualizedContent.content();
        
        // Remove initial content from education path
        List<ProgramItem> programItemsInPath = educationalPath.getProgramItems(_resolver);
        if (programItemsInPath.getLast().getId().equals(initialContent.getId()))
        {
            educationalPath = _removeLast(educationalPath);
        }
        return new ContextualizedPermissionContext(initialContent, educationalPath);
    }
    
    private EducationalPath _removeLast(EducationalPath educationalPath)
    {
        List<ProgramItem> programItemsInPath = educationalPath.getProgramItems(_resolver);
        List<ProgramItem> subList = programItemsInPath.subList(0, programItemsInPath.size() - 1);
        if (!subList.isEmpty())
        {
            return EducationalPath.of(subList.toArray(ProgramItem[]::new));
        }
        return null;
    }
    
    @Override
    protected Object _convertContext(Object initialContext)
    {
        if (initialContext instanceof ContextualizedContent contextualizedContent)
        {
            return contextualizedContent.content();
        }
        
        return super._convertContext(initialContext);
    }
    
    @Override
    protected Set< ? extends Object> _convertWorkspaceToRootRightContexts(Set<Object> workspacesContexts)
    {
        return null;
    }

    public I18nizableText getObjectLabel(Object object)
    {
        if (object instanceof ContextualizedContent contextualizedContent)
        {
            return ODFContentHierarchicalAccessController.getContentObjectLabel(contextualizedContent.content(), _odfContentTreeHelper);
        }
        throw new RightsException("Unsupported context: " + object.toString());
    }

    public I18nizableText getObjectCategory(Object object)
    {
        return ODFContentHierarchicalAccessController.ODF_CONTEXT_CATEGORY;
    }
}
