/*
 *  Copyright 2021 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.odf.schedulable;

import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Random;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.cocoon.components.ContextHelper;
import org.apache.cocoon.components.source.impl.SitemapSource;
import org.apache.commons.io.FileUtils;
import org.apache.commons.lang.StringUtils;
import org.apache.commons.lang3.exception.ExceptionUtils;
import org.apache.excalibur.source.SourceResolver;
import org.apache.excalibur.source.SourceUtil;
import org.quartz.JobDataMap;
import org.quartz.JobExecutionContext;

import org.ametys.cms.schedule.AbstractSendingMailSchedulable;
import org.ametys.core.schedule.progression.ContainerProgressionTracker;
import org.ametys.core.ui.mail.StandardMailBodyHelper;
import org.ametys.core.util.JSONUtils;
import org.ametys.odf.ODFHelper;
import org.ametys.odf.catalog.Catalog;
import org.ametys.odf.catalog.CatalogsManager;
import org.ametys.odf.enumeration.OdfReferenceTableHelper;
import org.ametys.odf.orgunit.OrgUnit;
import org.ametys.plugins.core.schedule.Scheduler;
import org.ametys.plugins.repository.AmetysObjectResolver;
import org.ametys.plugins.repository.UnknownAmetysObjectException;
import org.ametys.runtime.config.Config;
import org.ametys.runtime.i18n.I18nizableText;
import org.ametys.runtime.model.ElementDefinition;
import org.ametys.runtime.util.AmetysHomeHelper;

/**
 * Schedulable to export the ODF catalog as PDF
 */
public class CatalogPDFExportSchedulable extends AbstractSendingMailSchedulable
{
    /** The key for the catalog */
    public static final String JOBDATAMAP_CATALOG_KEY = "catalog";
    /** The key for the lang */
    public static final String JOBDATAMAP_LANG_KEY = "lang";
    /** The key for the orgunits */
    public static final String JOBDATAMAP_ORGUNIT_KEY = "orgunit";
    /** The key for the degrees */
    public static final String JOBDATAMAP_DEGREE_KEY = "degree";
    /** The key for the query'id */
    public static final String JOBDATAMAP_QUERY_KEY = "queryId";
    /** The key for the mode */
    public static final String JOBDATAMAP_MODE_KEY = "mode";
    /** The key for including subprograms */
    public static final String JOBDATAMAP_INCLUDE_SUBPROGRAMS = "includeSubPrograms";
    /** Mode when catalog is generated from a query */
    public static final String MODE_QUERY = "QUERY";

    /** Map key where the generated filename is stored */
    protected static final String _CATALOG_FILENAME = "catalogFilename";
    
    /** The Ametys object resolver. */
    protected AmetysObjectResolver _resolver;
    /** The ODF reference table helper. */
    protected OdfReferenceTableHelper _odfRefTableHelper;
    /** The avalon source resolver. */
    protected SourceResolver _sourceResolver;

    /** The catalog directory. */
    protected File _catalogRootDirectory;
    /** The JSON utils */
    protected JSONUtils _jsonUtils;
    /** The catalog manager */
    protected CatalogsManager _catalogsManager;
    
    @Override
    public void service(ServiceManager manager) throws ServiceException
    {
        super.service(manager);
        _resolver = (AmetysObjectResolver) manager.lookup(AmetysObjectResolver.ROLE);
        _odfRefTableHelper = (OdfReferenceTableHelper) manager.lookup(OdfReferenceTableHelper.ROLE);
        _sourceResolver = (SourceResolver) manager.lookup(SourceResolver.ROLE);
        _jsonUtils = (JSONUtils) manager.lookup(JSONUtils.ROLE);
        _catalogsManager = (CatalogsManager) manager.lookup(CatalogsManager.ROLE);
    }
    
    @Override
    public void initialize() throws Exception
    {
        super.initialize();
        _catalogRootDirectory = new File(AmetysHomeHelper.getAmetysHomeData(), "odf/catalog");
    }
    
    @Override
    protected void _doExecute(JobExecutionContext context, ContainerProgressionTracker progressionTracker) throws Exception
    {
        SitemapSource source = null;
        File pdfTmpFile = null;
        try
        {
            JobDataMap jobDataMap = context.getJobDetail().getJobDataMap();
            String catalog = jobDataMap.getString(Scheduler.PARAM_VALUES_PREFIX + JOBDATAMAP_CATALOG_KEY);
            String lang = jobDataMap.getString(Scheduler.PARAM_VALUES_PREFIX + JOBDATAMAP_LANG_KEY);
            
            FileUtils.forceMkdir(_catalogRootDirectory);
            
            File catalogDir = new File (_catalogRootDirectory, catalog);
            if (!catalogDir.exists())
            {
                catalogDir.mkdir();
            }
            
            File langDir = new File (catalogDir, lang);
            if (!langDir.exists())
            {
                langDir.mkdir();
            }
            
            String catalogFilename;
            
            // Resolve the export to the appropriate pdf url.
            Map<String, Object> params = new HashMap<>();
            
            String mode = jobDataMap.getString(Scheduler.PARAM_VALUES_PREFIX + JOBDATAMAP_MODE_KEY);
            params.put(JOBDATAMAP_MODE_KEY, mode);
            
            if (MODE_QUERY.equals(mode))
            {
                String queryId = jobDataMap.getString(Scheduler.PARAM_VALUES_PREFIX + JOBDATAMAP_QUERY_KEY);
                
                if (StringUtils.isEmpty(queryId))
                {
                    throw new IllegalArgumentException("Id of query is missing to generate PDF catalog from query");
                }
                params.put(JOBDATAMAP_QUERY_KEY, queryId);
                
                catalogFilename = _getCatalogFilename(queryId, null, null);
            }
            else
            {
                // Org units
                Object[] orgunits = _jsonUtils.convertJsonToArray(jobDataMap.getString(Scheduler.PARAM_VALUES_PREFIX + JOBDATAMAP_ORGUNIT_KEY));
                if (orgunits.length > 0)
                {
                    params.put(JOBDATAMAP_ORGUNIT_KEY, orgunits);
                }
                
                // Degrees
                Object[] degrees =  _jsonUtils.convertJsonToArray(jobDataMap.getString(Scheduler.PARAM_VALUES_PREFIX + JOBDATAMAP_DEGREE_KEY));
                if (degrees.length > 0)
                {
                    params.put(JOBDATAMAP_DEGREE_KEY, degrees);
                }
                
                catalogFilename = _getCatalogFilename(null, orgunits, degrees);
            }
            
            // Include subprograms
            boolean includeSubprograms = jobDataMap.getBoolean(Scheduler.PARAM_VALUES_PREFIX + JOBDATAMAP_INCLUDE_SUBPROGRAMS);
            params.put(JOBDATAMAP_INCLUDE_SUBPROGRAMS, includeSubprograms);
            
            // Set the attribute to force the switch to live data
            ContextHelper.getRequest(_context).setAttribute(ODFHelper.REQUEST_ATTRIBUTE_VALID_LABEL, true);
            source = (SitemapSource) _sourceResolver.resolveURI("cocoon://_plugins/odf/programs/" + catalog + "/" + lang + "/catalog.pdf", null, params);
            
            // Save the pdf into a temporary file.
            String tmpFilename = catalogFilename + "-" + new Random().nextInt() + ".tmp.pdf";
            pdfTmpFile = new File(langDir, tmpFilename);
            
            try (
                OutputStream pdfTmpOs = new FileOutputStream(pdfTmpFile);
                InputStream sourceIs = source.getInputStream()
            )
            {
                SourceUtil.copy(sourceIs, pdfTmpOs);
            }
            
            // If all went well until now, rename the temporary file
            File catalogFile = new File(langDir, catalogFilename + ".pdf");
            if (catalogFile.exists())
            {
                catalogFile.delete();
            }
            
            context.put(_CATALOG_FILENAME, catalogFile.getName());
            
            if (!pdfTmpFile.renameTo(catalogFile))
            {
                throw new IOException("Fail to rename catalog.tmp.pdf to catalog.pdf");
            }
        }
        finally
        {
            if (pdfTmpFile != null)
            {
                FileUtils.deleteQuietly(pdfTmpFile);
            }
            
            if (source != null)
            {
                _sourceResolver.release(source);
            }
        }
        
    }
    
    /**
     * Get the catalog PDF file name from configuration
     * @param queryId The id of query to execute. <code>null</code> when export is not based on a query
     * @param orgunits The restricted orgunits. <code>null</code> when export is based on a query
     * @param degrees The restricted degrees. <code>null</code> when export is based on a query
     * @return the computed catalog file name
     */
    protected String _getCatalogFilename(String queryId, Object[] orgunits, Object[] degrees)
    {
        List<String> filenamePrefix = new ArrayList<>();
        
        filenamePrefix.add("catalog");
        
        if (StringUtils.isNotEmpty(queryId))
        {
            filenamePrefix.add(StringUtils.substringAfter(queryId, "query://"));
        }
        else
        {
            if (orgunits != null && orgunits.length > 0)
            {
                Arrays.stream(orgunits)
                    .map(String.class::cast)
                    .map(this::_resolveSilently)
                    .filter(Objects::nonNull)
                    .map(OrgUnit::getUAICode)
                    .filter(StringUtils::isNotEmpty)
                    .forEach(filenamePrefix::add);
            }
            
            // Degrees
            if (degrees != null && degrees.length > 0)
            {
                Arrays.stream(degrees)
                        .map(String.class::cast)
                        .map(_odfRefTableHelper::getItemCode)
                        .filter(StringUtils::isNotEmpty)
                        .forEach(filenamePrefix::add);
            }
        }
        return StringUtils.join(filenamePrefix, "-");
    }
    
    private OrgUnit _resolveSilently(String ouId)
    {
        try
        {
            return _resolver.resolveById(ouId);
        }
        catch (UnknownAmetysObjectException e)
        {
            getLogger().warn("Can't find orgunit with id {}", ouId);
            return null;
        }
    }
    
    @Override
    public Map<String, ElementDefinition> getParameters()
    {
        // Remove unsupported widgets if necessary
        return ODFSchedulableHelper.cleanUnsupportedWidgets(ContextHelper.getRequest(_context), super.getParameters());
    }
    
    @Override
    protected I18nizableText _getSuccessMailSubject(JobExecutionContext context) throws Exception
    {
        return new I18nizableText("plugin.odf", "PLUGINS_ODF_PDF_EXPORT_MAIL_SUBJECT");
    }
    
    @Override
    protected boolean _isMailBodyInHTML(JobExecutionContext context) throws Exception
    {
        return true;
    }
    
    @Override
    protected String _getSuccessMailBody(JobExecutionContext context, String language) throws Exception
    {
        JobDataMap jobDataMap = context.getJobDetail().getJobDataMap();
        String catalogName = jobDataMap.getString(Scheduler.PARAM_VALUES_PREFIX + JOBDATAMAP_CATALOG_KEY);
        String lang = jobDataMap.getString(Scheduler.PARAM_VALUES_PREFIX + JOBDATAMAP_LANG_KEY);
        
        String catalogTitle = _getCatalogTitle(context);
        
        String downloadLink = StringUtils.removeEndIgnoreCase(Config.getInstance().getValue("cms.url"), "index.html");
        downloadLink += (downloadLink.endsWith("/") ? "" : "/") + "plugins/odf/download/" + catalogName + "/" + lang + "/" + context.get(_CATALOG_FILENAME);

        try
        {
            return StandardMailBodyHelper.newHTMLBody()
                    .withTitle(new I18nizableText("plugin.odf", "PLUGINS_ODF_PDF_EXPORT_MAIL_SUBJECT"))
                    .withMessage(new I18nizableText("plugin.odf", "PLUGINS_ODF_PDF_EXPORT_MAIL_BODY_SUCCESS", List.of(catalogTitle, downloadLink)))
                    .withLink(downloadLink, new I18nizableText("plugin.odf", "PLUGINS_ODF_PDF_EXPORT_MAIL_BODY_DOWNLOAD_LINK"))
                    .withLanguage(language)
                    .build();
        }
        catch (IOException e)
        {
            getLogger().warn("Failed to build HTML email body for PDF export result. Fallback to no wrapped email", e);
            return _i18nUtils.translate(new I18nizableText("plugin.odf", "PLUGINS_ODF_PDF_EXPORT_MAIL_BODY_SUCCESS", List.of(catalogTitle, downloadLink)), language);
        }
    }

    @Override
    protected I18nizableText _getErrorMailSubject(JobExecutionContext context) throws Exception
    {
        return new I18nizableText("plugin.odf", "PLUGINS_ODF_PDF_EXPORT_MAIL_SUBJECT");
    }

    @Override
    protected String _getErrorMailBody(JobExecutionContext context, String language, Throwable throwable) throws Exception
    {
        try
        {
            String catalogTitle = _getCatalogTitle(context);
            String error = ExceptionUtils.getStackTrace(throwable);
            
            return StandardMailBodyHelper.newHTMLBody()
                    .withTitle(new I18nizableText("plugin.odf", "PLUGINS_ODF_PDF_EXPORT_MAIL_SUBJECT"))
                    .withMessage(new I18nizableText("plugin.odf", "PLUGINS_ODF_PDF_EXPORT_MAIL_BODY_FAILURE", List.of(catalogTitle)))
                    .withDetails(null, error, true)
                    .withLanguage(language)
                    .build();
        }
        catch (IOException e)
        {
            getLogger().warn("Failed to build HTML email body for PDF export result. Fallback to no wrapped email", e);
            return _i18nUtils.translate(new I18nizableText("plugin.odf", "PLUGINS_ODF_PDF_EXPORT_MAIL_BODY_FAILURE"), language);
        }
    }
    
    private String _getCatalogTitle(JobExecutionContext context)
    {
        JobDataMap jobDataMap = context.getJobDetail().getJobDataMap();
        String catalogName = jobDataMap.getString(Scheduler.PARAM_VALUES_PREFIX + JOBDATAMAP_CATALOG_KEY);
        
        Catalog catalog = _catalogsManager.getCatalog(catalogName);
        return catalog.getTitle();
    }
}
