/*
 *  Copyright 2025 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.odf.schedulable;

import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;

import org.apache.cocoon.environment.Request;

import org.ametys.runtime.model.DefaultElementDefinition;
import org.ametys.runtime.model.ElementDefinition;
import org.ametys.runtime.workspace.WorkspaceMatcher;

/**
 * Helper for ODF scheduable
 */
public final class ODFSchedulableHelper
{
    private static final List<String> __UNSUPPORTED_WIDGETS_FOR_ADMIN = List.of("edition.select-content", "edition.select-orgunit", "widget.select-in-content-attribute-values", "edition.select-referencetable-content");
    
    private ODFSchedulableHelper() 
    {
        // Empty
    }
    
    /**
     * Get the parameters for job execution taking into account the supported widgets in the current context
     * @param request The request
     * @param initialParameters The initial parameters of scheduable
     * @return the parameters 
     */
    @SuppressWarnings("unchecked")
    public static Map<String, ElementDefinition> cleanUnsupportedWidgets(Request request, Map<String, ElementDefinition> initialParameters)
    {
        // FIXME CMS-9980
        if (_isAdminContext(request))
        {
            // Remove unsupported widgets in admin context
            Map<String, ElementDefinition> copiedParameters = new HashMap<>();
            
            initialParameters.entrySet().stream().forEach(e -> {
                ElementDefinition def = e.getValue();
                String name = e.getKey();
                
                if (def.getWidget() != null && __UNSUPPORTED_WIDGETS_FOR_ADMIN.contains(def.getWidget()))
                {
                    DefaultElementDefinition newDef = new DefaultElementDefinition(def);
                    newDef.setWidget(null);
                    newDef.setWidgetParameters(Map.of());
                    copiedParameters.put(name, newDef);
                }
                else
                {
                    copiedParameters.put(name, def);
                }
            });
            
            return copiedParameters;
        }
        else
        {
            return initialParameters;
        }
    }
    
    private static boolean _isAdminContext(Request request)
    {
        return Optional.ofNullable(request)
                .map(r -> r.getAttribute(WorkspaceMatcher.WORKSPACE_NAME))
                .map("admin"::equals)
                .orElse(false);
    }
}
