/*
 *  Copyright 2025 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */

/**
 * This class provides a widget to select one or more macro skills from a program.<br>
 * This embeds a drop down list with querying on title of macro skills and type-ahead support.<br> 
 * Allow content creation using #cfg-allowCreation.<br>
 * 
 * This widget is used to select macro skills in a program edit.
 */
Ext.define('Ametys.odf.widget.SelectMacroSkillsFromProgram', {
    extend: 'Ametys.odf.widget.SelectCatalogAwareContent',
    
    constructor: function (config)
    {
        config.transversal = config.transversal == "true";
        
        // Get the root content path (needed for creation)
        Ametys.data.ServerComm.callMethod({
            role: "org.ametys.odf.skill.ODFSkillsHelper",
            methodName: "getOdfRootContentPath",
            callback: {
                handler: this._setRootContentPathCb,
                scope: this
            },
            errorMessage:true,
            waitMessage: false
        });
        
        if (config.transversal)
        {
            this._parentProgram = this._noValueOptionId;
        }
        else
        {
            this._parentProgram = config.contentInfo.contentId;
            
            // Get the catalog of the current content (needed for creation)
            Ametys.data.ServerComm.callMethod({
                role: "org.ametys.odf.catalog.CatalogsManager",
                methodName: "getContentCatalog",
                parameters: [this._parentProgram],
                callback: {
                    handler: this._setContentCatalogCb,
                    scope: this
                },
                errorMessage:true,
                waitMessage: false
            });
        }
        
        this.callParent(arguments);
    },
    
    /**
     * Set the current content catalog.
     * @param {String} rootContentPath The rootContentPath
     * @private
     */
    _setRootContentPathCb: function(rootContentPath)
    {
        this._rootContentPath = rootContentPath;
    },
    /**
     * Set the current content catalog.
     * @param {String} catalogName The catalog name
     * @private
     */
    _setContentCatalogCb: function(catalogName)
    {
        this._contentCatalog = catalogName;
    },
    
    // FIXME ODF-4025 [APC] Improve the skills widgets, stop this hack and support it in the helper directly
    _initSearchFn: function(form, searchModel)
    {
        this.callParent(arguments);
        
        var programFd = form.getForm().findField('reference-parentProgram-eq');
        programFd.setValue(this._parentProgram);
        programFd.disable();
    },
    
    _getAdditionalCreationParameters: function()
    {
        return {
            forceMode: 'disabled',
            forceValues: {
                catalog: this._contentCatalog,
            },
            rootContentPath: this._rootContentPath,
            additionalWorkflowParameters: {
                'org.ametys.odf.workflow.AbstractCreateODFContentFunction$catalog': this._contentCatalog,
                'org.ametys.odf.skill.workflow.SkillEditionFunction$transversal': this.transversal,
            }
        };
    },
    
    _getAdditionalLoadValues: function()
    {
        var values = this.callParent(arguments);

        // Only load the macro skills for the program (or transversal) by setting the value of parentProgram field
        values['reference-parentProgram-eq'] = this._parentProgram;
        
        return values;
    },
    
    _openCreatedContent: function(contentId)
    {
        // Nothing
    }
})