/*
 *  Copyright 2025 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.odf.skill.export.csv;

import java.io.IOException;
import java.util.Locale;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.cocoon.ProcessingException;
import org.apache.cocoon.environment.ObjectModelHelper;
import org.apache.cocoon.environment.Request;
import org.apache.cocoon.xml.AttributesImpl;
import org.apache.cocoon.xml.XMLUtils;
import org.xml.sax.SAXException;

import org.ametys.cms.repository.Content;
import org.ametys.cms.search.cocoon.SearchAction;
import org.ametys.cms.search.cocoon.SearchGenerator;
import org.ametys.odf.skill.ODFSkillsHelper;
import org.ametys.odf.skill.workflow.SkillEditionFunction;
import org.ametys.runtime.model.ModelViewItem;
import org.ametys.runtime.model.View;
import org.ametys.runtime.model.ViewHelper;
import org.ametys.runtime.model.ViewItem;
import org.ametys.runtime.model.ViewItemAccessor;
import org.ametys.runtime.model.ViewItemContainer;

/**
 * Generator to export skills.
 */
public class SkillsExportGenerator extends SearchGenerator
{
    /** The odf skills helper */
    protected ODFSkillsHelper _skillsHelper;
    
    @Override
    public void service(ServiceManager smanager) throws ServiceException
    {
        super.service(smanager);
        _skillsHelper = (ODFSkillsHelper) smanager.lookup(ODFSkillsHelper.ROLE);
    }
    
    @Override
    public void generate() throws IOException, SAXException, ProcessingException
    {
        contentHandler.startDocument();
        XMLUtils.startElement(contentHandler, "export");

        _saxColumns();
        
        Request request = ObjectModelHelper.getRequest(objectModel);
        Locale defaultLocale = (Locale) request.getAttribute(SearchAction.SEARCH_LOCALE);
        
        String catalog = request.getParameter("catalog");
        
        View view = _cTypeHelper.getView("csv-export", new String[] {SkillEditionFunction.MICRO_SKILL_TYPE}, new String[0]);
        
        _saxSkills(catalog, view, defaultLocale);
        
        XMLUtils.endElement(contentHandler, "export");
        contentHandler.endDocument();
    }
    
    private void _saxColumns() throws SAXException
    {
        XMLUtils.startElement(contentHandler, "columns");
        
        View view = _cTypeHelper.getView("csv-export", new String[] {SkillEditionFunction.MICRO_SKILL_TYPE}, new String[0]);
        
        _saxColumns(view);
        
        XMLUtils.endElement(contentHandler, "columns");
    }
    
    private void _saxColumns(ViewItemAccessor viewItemAccessor) throws SAXException
    {
        for (ViewItem viewItem : viewItemAccessor.getViewItems())
        {
            if (viewItem instanceof ViewItemAccessor itemAccessor && !itemAccessor.getViewItems().isEmpty())
            {
                _saxColumns(itemAccessor);
            }
            else if (viewItem instanceof ModelViewItem modelViewItem)
            {
                AttributesImpl attrs = new AttributesImpl();
                attrs.addCDATAAttribute("id", ViewHelper.getModelViewItemPath(modelViewItem));
                attrs.addCDATAAttribute("type", modelViewItem.getDefinition().getType().getId());
                
                XMLUtils.startElement(contentHandler, "column", attrs);
                
                modelViewItem.getLabel().toSAX(contentHandler);
                
                XMLUtils.endElement(contentHandler, "column");
            }
        }
    }

    /**
     * Sax the rules
     * @param catalog The catalog
     * @param view The export view
     * @param defaultLocale The default locale
     * @throws SAXException if an error occurred
     */
    protected void _saxSkills(String catalog, ViewItemContainer view, Locale defaultLocale) throws SAXException
    {
        XMLUtils.startElement(contentHandler, "contents");
        
        for (Content microSkill : _skillsHelper.getMicroSkills(catalog))
        {
            try
            {
                saxContent(microSkill, view, defaultLocale);
            }
            catch (Exception e)
            {
                getLogger().warn("The skill '" + microSkill.getTitle() + "' [" + microSkill.getId() + "] could not be exported to CSV.", e);
            }
        }
        
        XMLUtils.endElement(contentHandler, "contents");
    }
}
