/*
 *  Copyright 2021 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.odf.validator;

import java.util.Map;
import java.util.Optional;

import org.ametys.cms.contenttype.AttributeDefinition;
import org.ametys.cms.contenttype.validation.AbstractContentValidator;
import org.ametys.cms.repository.Content;
import org.ametys.odf.courselist.CourseList;
import org.ametys.odf.courselist.CourseList.ChoiceType;
import org.ametys.plugins.repository.data.holder.impl.DataHolderHelper;
import org.ametys.plugins.repository.data.holder.values.SynchronizationContext;
import org.ametys.plugins.repository.data.holder.values.UntouchedValue;
import org.ametys.runtime.i18n.I18nizableText;
import org.ametys.runtime.model.View;
import org.ametys.runtime.parameter.ValidationResult;

/**
 * Validator for {@link CourseList} to check if the min value is correctly fill when choice type is choosen.
 */
public class CourseListChoiceValidator extends AbstractContentValidator
{
    @Override
    public ValidationResult validate(Content content)
    {
        assert content instanceof CourseList;
        
        ValidationResult result = new ValidationResult();
        
        CourseList courseList = (CourseList) content;
        if (courseList.getType() == ChoiceType.CHOICE)
        {
            if (courseList.getMinNumberOfCourses() < 1)
            {
                result.addError(new I18nizableText("plugin.odf", "PLUGINS_ODF_COURSELIST_VALIDATOR_CHOICE_MIN_ERROR"));
            }
        }
        
        return result;
    }

    @Override
    public ValidationResult validate(Content content, Map<String, Object> values, View view)
    {
        assert content instanceof CourseList;
        
        ValidationResult result = new ValidationResult();
        
        // Validation du formulaire uniquement si "min" n'est pas en lecture seule
        AttributeDefinition minDefinition = (AttributeDefinition) content.getDefinition(CourseList.MIN_COURSES);
        if (values.containsKey(CourseList.MIN_COURSES) && !(values.get(CourseList.MIN_COURSES) instanceof UntouchedValue) && minDefinition.canWrite(content))
        {
            ChoiceType choice = Optional.ofNullable(values)
                .map(v -> _getValue(content, v, CourseList.CHOICE_TYPE))
                .map(String.class::cast)
                .map(ChoiceType::valueOf)
                .orElse(null);
            
            if (choice == ChoiceType.CHOICE)
            {
                Long minCourses = Optional.ofNullable(values)
                    .map(v -> v.get(CourseList.MIN_COURSES))
                    .map(v -> DataHolderHelper.getValueFromSynchronizableValue(v, content, minDefinition, Optional.of(CourseList.MIN_COURSES), SynchronizationContext.newInstance()))
                    .map(Long.class::cast)
                    .orElse(-1L);
                
                if (minCourses < 1)
                {
                    result.addError(new I18nizableText("plugin.odf", "PLUGINS_ODF_COURSELIST_VALIDATOR_CHOICE_MIN_ERROR"));
                }
            }
        }
        
        return result;
    }
    
    private Object _getValue(Content content, Map<String, Object> values, String attributePath)
    {
        AttributeDefinition definition = (AttributeDefinition) content.getDefinition(attributePath);
        return DataHolderHelper.getValueFromSynchronizableValue(values.get(attributePath), content, definition, Optional.of(attributePath), SynchronizationContext.newInstance());
    }
}
