/*
 *  Copyright 2014 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.odf.export.indesign;

import java.io.IOException;
import java.net.MalformedURLException;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.cocoon.ProcessingException;
import org.apache.cocoon.components.source.impl.SitemapSource;
import org.apache.cocoon.environment.ObjectModelHelper;
import org.apache.cocoon.environment.Request;
import org.apache.cocoon.generation.ServiceableGenerator;
import org.apache.cocoon.xml.AttributesImpl;
import org.apache.cocoon.xml.XMLUtils;
import org.apache.commons.lang.StringUtils;
import org.xml.sax.SAXException;

import org.ametys.core.util.IgnoreRootHandler;
import org.ametys.core.util.URIUtils;
import org.ametys.odf.catalog.Catalog;
import org.ametys.odf.catalog.CatalogsManager;
import org.ametys.odf.program.Program;
import org.ametys.plugins.repository.AmetysObjectIterable;
import org.ametys.plugins.repository.UnknownAmetysObjectException;
import org.ametys.runtime.config.Config;

/**
 * Generate the SAX events corresponding to the transformation of a catalogue to a XML file for InDesign
 */
public class IndesignCatalogueGenerator extends ServiceableGenerator
{
    /** The catalogs manager */
    private CatalogsManager _catalogsManager;
    
    @Override
    public void service(ServiceManager sManager) throws ServiceException
    {
        super.service(sManager);
        _catalogsManager = (CatalogsManager) sManager.lookup(CatalogsManager.ROLE);
    }
    
    public void generate() throws IOException, SAXException, ProcessingException
    {
        Request request = ObjectModelHelper.getRequest(objectModel);
        
        String catalogName = parameters.getParameter("catalog", null);
        String xslt = request.getParameter("xslt");
        
        contentHandler.startDocument();
        XMLUtils.startElement(contentHandler, "programs");
        
        Catalog catalog = _catalogsManager.getCatalog(catalogName);
        if (catalog == null)
        {
            throw new IllegalArgumentException ("Failed to generate the XML for InDesign of unknown catalog '" + catalogName + "'");
        }
        
        XMLUtils.createElement(contentHandler, "catalog", catalog.getTitle());
        
        // Retrieve language
        String lang = request.getParameter("lang");
        if (StringUtils.isEmpty(lang))
        {
            lang = Config.getInstance().getValue("odf.programs.lang");
        }
            
        _saxPrograms(catalog.getName(), lang, xslt);
        
        XMLUtils.endElement(contentHandler, "programs");
        contentHandler.endDocument();
    }
    
    /**
     * SAX the programs for the given catalogue and language
     * @param catalogName the catalog's name
     * @param lang the programs' language
     * @param xslt the xslt used to transform a given program
     * @throws MalformedURLException if the url is malformed
     * @throws IOException if an in / out type error occurs
     * @throws SAXException if an exception occurs while saxing 
     */
    private void _saxPrograms(String catalogName, String lang, String xslt) throws MalformedURLException, IOException, SAXException
    {
        AmetysObjectIterable<Program> programs = _catalogsManager.getPrograms(catalogName, lang);
        
        for (Program program : programs)
        {
            String programId = program.getId();
            
            AttributesImpl attrs = new AttributesImpl();
            attrs.addCDATAAttribute("id", program.getId());
            attrs.addCDATAAttribute("name", program.getName());
            attrs.addCDATAAttribute("title", program.getTitle());
            
            XMLUtils.startElement(contentHandler, "program", attrs);
            
            SitemapSource src = null;      
            try
            {
                src = (SitemapSource) resolver.resolveURI(_getUri(programId, xslt));
                src.toSAX(new IgnoreRootHandler(contentHandler));
            }
            catch (UnknownAmetysObjectException e)
            {
                // The content may be archived
            }
            finally
            {
                resolver.release(src);
            }
            
            XMLUtils.endElement(contentHandler, "program");
        }
    }
    
    /**
     * Get the URI corresponding to the pipeline
     * @param programId the id of the program
     * @param xslt the name of the xslt
     * @return the URI 
     */
    protected String _getUri(String programId, String xslt)
    {
        return "cocoon://_plugins/odf/_indesign/program.xml?id=" + URIUtils.encodeParameter(programId) + "&xslt=" + xslt;
    }
}
