/*
 *  Copyright 2025 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.odf.observation.skill;

import java.util.Arrays;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;

import org.ametys.cms.ObservationConstants;
import org.ametys.cms.data.ContentValue;
import org.ametys.cms.repository.Content;
import org.ametys.cms.trash.element.TrashElementDAO;
import org.ametys.core.observation.Event;
import org.ametys.core.observation.ObservationManager;
import org.ametys.core.user.CurrentUserProvider;
import org.ametys.plugins.repository.trash.TrashElement;

/**
 * Observer to delete skills that became orphans after the deletion of their parent.
 * When the deleted content is a Program, delete the linked macro skills that would become orphans
 *    The deletion of macro skills will trigger the deletion of its micro skills
 * When the deleted content is a Macro skill, delete the linked micro skills that would become orphans
 */
public class DeleteContentSkillStep2Observer extends AbstractSkillsStepObserver
{
    private CurrentUserProvider _currentUserProvider;
    private ObservationManager _observationManager;
    private TrashElementDAO _trashElementDAO;

    @Override
    public void service(ServiceManager manager) throws ServiceException
    {
        super.service(manager);
        _currentUserProvider = (CurrentUserProvider) manager.lookup(CurrentUserProvider.ROLE);
        _observationManager = (ObservationManager) manager.lookup(ObservationManager.ROLE);
        _trashElementDAO = (TrashElementDAO) manager.lookup(org.ametys.plugins.repository.trash.TrashElementDAO.ROLE);
    }
    
    @Override
    protected String getSupportedEventId()
    {
        return ObservationConstants.EVENT_CONTENT_DELETED;
    }
    
    public void observe(Event event, Map<String, Object> transientVars) throws Exception
    {
        String contentId = (String) event.getArguments().get(ObservationConstants.ARGS_CONTENT_ID);
        // When the content is modified, we need to check if the skills are still linked to the content
        ContentValue[] previousSkills = _getRequestAttribute("previousSkills", contentId);
        // Remove the orphan skills
        if (previousSkills != null && previousSkills.length > 0)
        {
            // Check if the original parent was trashed or deleted
            TrashElement trashElement = _trashElementDAO.find(contentId);
            
            if (trashElement != null)
            {
                // The parent is deleted so all previous values need to be deleted
                List<String> skillsToDelete = Arrays.stream(previousSkills)
                        .map(ContentValue::getContentId)
                        .toList();
                
                // Trash skills
                _contentDAO.trashContents(skillsToDelete, true);
                
                for (String deletedChild : skillsToDelete)
                {
                    TrashElement trashChild = _trashElementDAO.find(deletedChild);
                    if (trashChild != null)
                    {
                        trashChild.setHidden(true);
                        trashChild.saveChanges();
                        
                        // Notify observers
                        Map<String, Object> eventParams = new HashMap<>();
                        eventParams.put(org.ametys.cms.ObservationConstants.ARGS_TRASH_ELEMENT_ID, trashChild.getId());
                        eventParams.put(org.ametys.cms.ObservationConstants.ARGS_AMETYS_OBJECT_ID, contentId);
                        _observationManager.notify(new Event(org.ametys.cms.ObservationConstants.EVENT_TRASH_UPDATED, _currentUserProvider.getUser(), eventParams));
                    }
                }
                
                trashElement.addLinkedObjects(skillsToDelete.toArray(String[]::new));
                trashElement.saveChanges();
                
                // Notify observers
                Map<String, Object> eventParams = new HashMap<>();
                eventParams.put(org.ametys.cms.ObservationConstants.ARGS_TRASH_ELEMENT_ID, trashElement.getId());
                eventParams.put(org.ametys.cms.ObservationConstants.ARGS_AMETYS_OBJECT_ID, trashElement.getAmetysObjectId());
                _observationManager.notify(new Event(org.ametys.cms.ObservationConstants.EVENT_TRASH_UPDATED, _currentUserProvider.getUser(), eventParams));
            }
            else
            {
                List<Content> skillsToDelete = Arrays.stream(previousSkills)
                        .map(ContentValue::getContentIfExists)
                        .filter(Optional::isPresent)
                        .map(Optional::get)
                        .map(Content.class::cast)
                        .toList();
                
                if (!skillsToDelete.isEmpty())
                {
                    // Remove the orphan skills
                    _contentDAO.forceDeleteContentsObj(skillsToDelete, null);
                }
            }
        }
    }
}
