/*
 *  Copyright 2011 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.odf;

import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.Reader;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.apache.avalon.framework.logger.Logger;
import org.apache.commons.io.IOUtils;
import org.apache.http.HttpEntity;
import org.apache.http.NameValuePair;
import org.apache.http.client.config.RequestConfig;
import org.apache.http.client.entity.UrlEncodedFormEntity;
import org.apache.http.client.methods.CloseableHttpResponse;
import org.apache.http.client.methods.HttpPost;
import org.apache.http.entity.InputStreamEntity;
import org.apache.http.impl.client.CloseableHttpClient;
import org.apache.http.impl.client.HttpClientBuilder;
import org.apache.http.message.BasicNameValuePair;

import org.ametys.core.util.StringUtils;
import org.ametys.runtime.config.Config;

/**
 * Helper for calling WS on a remote server
 *
 */
public final class CallWSHelper
{
    private CallWSHelper ()
    {
        // Empty
    }
    
    /**
     * Request the given url on remote server
     * @param url the url to be called.
     * @param parameters the HTTP parameters
     * @param logger The logger for traces
     * @return the list of server's url which failed. If empty all WS succeed. Returns <code>null</code> if a global error occurred.
     */
    public static Set<String> callWS (String url, Map<String, String> parameters, Logger logger)
    {
        Set<String> failedUrl = new HashSet<>();
        
        String serverConfig = Config.getInstance().getValue("odf.publish.server.url");
        
        Collection<String> serverUrls = StringUtils.stringToCollection(serverConfig);
        
        try
        {
            for (String serverUrl : serverUrls)
            {
            
                String wsURL = serverUrl + (serverUrl.endsWith("/") ? "" : "/") + url;
                
                try (CloseableHttpClient httpClient = _getHttpClient())
                {
                    try
                    {
                        // Prepare a request object
                        HttpPost postRequest = new HttpPost(wsURL);
                        
                        // HTTP parameters
                        List<NameValuePair> params = new ArrayList<>();
                        for (String paramName : parameters.keySet())
                        {
                            params.add(new BasicNameValuePair(paramName, parameters.get(paramName)));
                        }
                        postRequest.setEntity(new UrlEncodedFormEntity(params, "UTF-8"));
                        
                        // Execute the request
                        try (CloseableHttpResponse httpResponse = httpClient.execute(postRequest))
                        {
                            if (httpResponse.getStatusLine().getStatusCode() == 200)
                            {
                                HttpEntity responseEntity = httpResponse.getEntity();
                                if (responseEntity != null)
                                {
                                    try (Reader reader = new InputStreamReader(responseEntity.getContent(), "UTF-8"))
                                    {
                                        // Nothing to do
                                    }
                                }
                            }
                            else
                            {
                                logger.error("Unable to publish CDM-fr with request: " + wsURL + ", response: " + httpResponse.getStatusLine());
                                failedUrl.add(serverUrl);
                            }
                        }
                    }
                    catch (Exception e)
                    {
                        logger.error("Unable to publish CDM-fr with request: " + wsURL, e);
                        failedUrl.add(serverUrl);
                    }
                }
            }
        }
        catch (IOException e)
        {
            logger.error("Unable to publish CDM-fr", e);
            return null;
        }
        
        return failedUrl;
    }
    
    /**
     * Request the given url on remote server
     * @param url the url to be called.
     * @param is The input stream for HTTP entity
     * @param logger The logger for traces
     * @return the list of server's url which failed. If empty all WS succeed. Returns <code>null</code> if a global error occurred.
     */
    public static Set<String> callWS (String url, InputStream is, Logger logger)
    {
        Set<String> failedUrl = new HashSet<>();
        
        String serverConfig = Config.getInstance().getValue("odf.publish.server.url");
        
        Collection<String> serverUrls = StringUtils.stringToCollection(serverConfig);
        
        try
        {
            byte[] bytes = IOUtils.toByteArray(is);
            
            for (String serverUrl : serverUrls)
            {
                String wsURL = serverUrl + (serverUrl.endsWith("/") ? "" : "/") + url;
                
                try (CloseableHttpClient httpclient = _getHttpClient())
                {
                    try
                    {
                        // Prepare a request object
                        HttpPost postRequest = new HttpPost(wsURL);
                        postRequest.addHeader("Content-Type", "text/xml");
                        
                        ByteArrayInputStream bIs = new ByteArrayInputStream(bytes);
                        HttpEntity entity = new InputStreamEntity(bIs, bytes.length);
                        postRequest.setEntity(entity);
                        
                        // Execute the request
                        try (CloseableHttpResponse httpResponse = httpclient.execute(postRequest))
                        {
                            if (httpResponse.getStatusLine().getStatusCode() == 200)
                            {
                                HttpEntity responseEntity = httpResponse.getEntity();
                                if (responseEntity != null)
                                {
                                    try (Reader reader = new InputStreamReader(responseEntity.getContent(), "UTF-8"))
                                    {
                                        // Nothing to do
                                    }
                                }
                            }
                            else
                            {
                                logger.error("Unable to publish CDM-fr with request: " + wsURL + ", response: " + httpResponse.getStatusLine());
                                failedUrl.add(serverUrl);
                            }
                        }
                        
                        bIs.reset();
                    }
                    catch (Exception e)
                    {
                        logger.error("Unable to publish CDM-fr with request: " + wsURL, e);
                        failedUrl.add(serverUrl);
                    }
                }
            }
        }
        catch (IOException e)
        {
            logger.error("Unable to publish CDM-fr", e);
            return null;
        }
        
        return failedUrl;
    }
    
    private static CloseableHttpClient _getHttpClient ()
    {
        RequestConfig requestConfig = RequestConfig.custom().build();
        return HttpClientBuilder.create()
                .setDefaultRequestConfig(requestConfig)
                .useSystemProperties()
                .build();
    }
}
