/*
 *  Copyright 2025 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.odf.restriction;

import java.util.LinkedHashMap;
import java.util.Map;

import org.apache.avalon.framework.configuration.Configurable;
import org.apache.avalon.framework.configuration.Configuration;
import org.apache.avalon.framework.configuration.ConfigurationException;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;

import org.ametys.cms.contenttype.ContentTypeExtensionPoint;
import org.ametys.odf.EducationalPathHelper;
import org.ametys.plugins.repository.model.RepeaterDefinition;
import org.ametys.runtime.i18n.I18nizableText;
import org.ametys.runtime.model.Enumerator;
import org.ametys.runtime.model.ModelItem;
import org.ametys.runtime.plugin.component.DeferredServiceable;

/**
 * List the educational paths attributes
 */
public class ByEducationalPathAttributesEnumerator implements Enumerator<String>, Configurable, DeferredServiceable
{
    /** The contenttype to consider */
    protected String _contentType;
    /** The content type extension point */
    protected ContentTypeExtensionPoint _contentTypeExtensionPoint;
    /** The educational path helper */
    protected EducationalPathHelper _educationalPathHelper;

    public void configure(Configuration configuration) throws ConfigurationException
    {
        _contentType = configuration.getChild("content-type").getValue();
        if (_contentTypeExtensionPoint != null // Avoid exception in safe mode
                && !_contentTypeExtensionPoint.hasExtension(_contentType))
        {
            throw new ConfigurationException("The content type '" + _contentType + "' is not registered in the content type extension point.");
        }
    }
    
    public void deferredService(ServiceManager manager) throws ServiceException
    {
        if (manager.hasService(ContentTypeExtensionPoint.ROLE)) // Safe mode compatible
        {
            _contentTypeExtensionPoint = (ContentTypeExtensionPoint) manager.lookup(ContentTypeExtensionPoint.ROLE);
            _educationalPathHelper = (EducationalPathHelper) manager.lookup(EducationalPathHelper.ROLE);
        }
    }
    
    public Map<String, I18nizableText> getEntries() throws Exception
    {
        Map<String, I18nizableText> entries = new LinkedHashMap<>();
        
        if (_contentTypeExtensionPoint != null)
        {
            for (ModelItem modelItem : _contentTypeExtensionPoint.getExtension(_contentType).getModelItems())
            {
                if (modelItem instanceof RepeaterDefinition repeaterDef)
                {
                    if (_educationalPathHelper.isRepeaterWithEducationalPath(repeaterDef))
                    {
                        entries.put(repeaterDef.getName(), repeaterDef.getLabel());
                    }
                }
            }
        }
        
        return entries;
    }

}
