/*
 *  Copyright 2025 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.odf.catalog;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.stream.Stream;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;

import org.ametys.cms.ObservationConstants;
import org.ametys.cms.data.ContentValue;
import org.ametys.cms.repository.Content;
import org.ametys.cms.repository.ContentQueryHelper;
import org.ametys.cms.repository.ContentTypeExpression;
import org.ametys.cms.repository.DefaultContent;
import org.ametys.cms.repository.ModifiableContent;
import org.ametys.cms.repository.ModifiableDefaultContent;
import org.ametys.cms.repository.WorkflowAwareContent;
import org.ametys.cms.workflow.ContentWorkflowHelper;
import org.ametys.cms.workflow.EditContentFunction;
import org.ametys.core.observation.Event;
import org.ametys.core.observation.ObservationManager;
import org.ametys.core.user.CurrentUserProvider;
import org.ametys.odf.ODFHelper;
import org.ametys.odf.course.Course;
import org.ametys.odf.program.Program;
import org.ametys.odf.program.ProgramFactory;
import org.ametys.odf.skill.ODFSkillsHelper;
import org.ametys.odf.skill.workflow.SkillEditionFunction;
import org.ametys.plugins.repository.AmetysObjectResolver;
import org.ametys.plugins.repository.AmetysRepositoryException;
import org.ametys.plugins.repository.ModifiableAmetysObject;
import org.ametys.plugins.repository.ModifiableTraversableAmetysObject;
import org.ametys.plugins.repository.RemovableAmetysObject;
import org.ametys.plugins.repository.data.holder.group.ModifiableModelAwareRepeater;
import org.ametys.plugins.repository.data.holder.group.ModifiableModelAwareRepeaterEntry;
import org.ametys.plugins.repository.jcr.NameHelper;
import org.ametys.plugins.repository.lock.LockableAmetysObject;
import org.ametys.plugins.repository.query.expression.AndExpression;
import org.ametys.plugins.repository.query.expression.Expression;
import org.ametys.plugins.repository.query.expression.Expression.Operator;
import org.ametys.plugins.repository.query.expression.StringExpression;
import org.ametys.plugins.workflow.AbstractWorkflowComponent;
import org.ametys.plugins.workflow.component.CheckRightsCondition;
import org.ametys.runtime.plugin.component.AbstractLogEnabled;

import com.opensymphony.workflow.WorkflowException;

/**
 * Copy updater to update the micro skills on a macro skill and the macro skills on a {@link Program} and (@link Course}.
 */
public class SkillsCopyUpdater extends AbstractLogEnabled implements CopyCatalogUpdater, Serviceable
{
    private static final List<String> __SKILLS_IGNORED_ATTRIBUTES = List.of("parentMacroSkill", "microSkills", "parentProgram", "catalog");
    
    /** The ametys object resolver */
    protected AmetysObjectResolver _resolver;
    /** The observation manager */
    protected ObservationManager _observationManager;
    /** The current user provider */
    protected CurrentUserProvider _currentUserProvider;
    /** The content workflow helper */
    protected ContentWorkflowHelper _contentWorkflowHelper;
    /** The ODF helper */
    protected ODFHelper _odfHelper;
    
    public void service(ServiceManager manager) throws ServiceException
    {
        _resolver = (AmetysObjectResolver) manager.lookup(AmetysObjectResolver.ROLE);
        _observationManager = (ObservationManager) manager.lookup(ObservationManager.ROLE);
        _currentUserProvider = (CurrentUserProvider) manager.lookup(CurrentUserProvider.ROLE);
        _contentWorkflowHelper = (ContentWorkflowHelper) manager.lookup(ContentWorkflowHelper.ROLE);
        _odfHelper = (ODFHelper) manager.lookup(ODFHelper.ROLE);
    }
    
    public void updateContents(String initialCatalogName, String newCatalogName, Map<Content, Content> copiedContents, Content targetParentContent)
    {
        // Do Nothing
    }
    
    public List<Content> getAdditionalContents(String catalogName)
    {
        List<Content> results = new ArrayList<>();
        
        results.addAll(_getContents(catalogName, SkillEditionFunction.MICRO_SKILL_TYPE));
        results.addAll(_getContents(catalogName, SkillEditionFunction.MACRO_SKILL_TYPE));
        
        return results;
    }
    
    public void copyAdditionalContents(String initialCatalogName, String newCatalogName, Map<Content, Content> copiedContents)
    {
        // Only copy the skills if they are enabled
        if (!ODFSkillsHelper.isSkillsEnabled())
        {
            return;
        }
        
        // Get the skills of the catalog to copy
        List<DefaultContent> microSkillsToCopy = _getContents(initialCatalogName, SkillEditionFunction.MICRO_SKILL_TYPE);
        List<DefaultContent> macroSkillsToCopy = _getContents(initialCatalogName, SkillEditionFunction.MACRO_SKILL_TYPE);
        
        // Copy the micro skills in the new catalog
        Map<String, Content> copiedMicroSkills = _copyMicroSkills(microSkillsToCopy, newCatalogName);
        // Copy the macro skills in the new catalog and update the links to the copied micro skills
        Map<String, Content> copiedMacroSkills = _copyMacroSkills(macroSkillsToCopy, newCatalogName, copiedMicroSkills);

        // Update the links to the copied macro skills in programs and skills affectations in courses
        _updateContentsAfterSkillsCreation(newCatalogName, copiedContents, copiedMacroSkills, copiedMicroSkills);
    }
    
    private Map<String, Content> _copyMacroSkills(List<DefaultContent> skills, String newCatalogName, Map<String, Content> copiedMicroSkills)
    {
        return _copySkills(skills, SkillEditionFunction.MACRO_SKILL_TYPE, newCatalogName, copiedMicroSkills);
    }
    
    private Map<String, Content> _copyMicroSkills(List<DefaultContent> skills, String newCatalogName)
    {
        return _copySkills(skills, SkillEditionFunction.MICRO_SKILL_TYPE, newCatalogName, Map.of());
    }
    
    private Map<String, Content> _copySkills(List<DefaultContent> skills, String contentType, String newCatalogName, Map<String, Content> copiedMicroSkills)
    {
        Map<String, Content> copiedSkills = new HashMap<>();
        for (DefaultContent skill : skills)
        {
            // Log if the targeted catalog already contains the skill
            if (_skillExists(skill, contentType, newCatalogName))
            {
                getLogger().info("A skill already exists with the same code, catalog and language [{}, {}, {}]", skill.getValue("code"), newCatalogName, skill.getLanguage());
            }
            // Copy the skill in the targeted catalog
            else
            {
                try
                {
                    ModifiableContent newSkill = _createSkill(skill, newCatalogName);
                    
                    // If the skill is a macroSkill and has microSkills values, update the linkes to point to the new ones
                    if (contentType.equals(SkillEditionFunction.MACRO_SKILL_TYPE) && skill.hasValue("microSkills"))
                    {
                        // If the content is a macroSkill and has microSkills, link the new ones that where previously copied
                        Map<String, Object> values = new HashMap<>();
                        ContentValue[] previousMicroSkills = skill.getValue("microSkills");
                        List<Content> microSkills = Arrays.asList(previousMicroSkills)
                                .stream()
                                .filter(Objects::nonNull)
                                .map(microSkill -> copiedMicroSkills.get(microSkill.getContentId()))
                                .toList();
                        
                        values.put("microSkills", microSkills);
                        try
                        {
                            _contentWorkflowHelper.editContent((ModifiableDefaultContent) newSkill, values, 2);
                        }
                        catch (Exception e)
                        {
                            // Log and rollback
                            getLogger().error("Impossible to update skill '{}' ({}) while creating the catalog {}", newSkill.getTitle(), newSkill.getId(), newCatalogName);
                            
                            _deleteContent(newSkill);
                        }
                    }
                    
                    // If the skill could be created, add it to the copied skills
                    if (newSkill != null)
                    {
                        copiedSkills.put(skill.getId(), newSkill);
                    }
                }
                catch (AmetysRepositoryException e)
                {
                    getLogger().error("Impossible to create the skill '{}' ({}) while creating the catalog {}", skill.getTitle(), skill.getId(), newCatalogName, e);
                }
            }
        }
        
        return copiedSkills;
    }
    
    private ModifiableContent _createSkill(DefaultContent skill, String newCatalogName) throws AmetysRepositoryException
    {
        // Create the skill in the new catalog
        ModifiableContent newSkill = skill.copyTo((ModifiableTraversableAmetysObject) skill.getParent(), NameHelper.filterName(skill.getTitle()));
        
        // Remove the attributes that need to be updated
        for (String data : newSkill.getDataNames())
        {
            // Remove the attribute that can't be copied
            if (__SKILLS_IGNORED_ATTRIBUTES.contains(data))
            {
                newSkill.removeValue(data);
            }
        }
        // Set the new catalog
        newSkill.setValue("catalog", newCatalogName);
        newSkill.saveChanges();
        
        return newSkill;
    }
    
    private void _updateContentsAfterSkillsCreation(String newCatalogName, Map<Content, Content> copiedContents, Map<String, Content> copiedMacroSkills, Map<String, Content> copiedMicroSkills)
    {
        // For every copied program, update its links from the original macro skills to the copied macro skills
        for (Content copiedContent : copiedContents.values())
        {
            try
            {
                if (copiedContent instanceof Program program)
                {
                    _updateProgram(program, copiedMacroSkills, copiedMicroSkills);
                }
                else if (copiedContent instanceof Course course)
                {
                    _updateCourse(course, copiedMicroSkills);
                }
            }
            catch (Exception e)
            {
                getLogger().error("An error occurred while copying the program '{}' in the new catalog '{}'", copiedContent.getId(), newCatalogName, e);
            }
        }
    }
    
    private void _updateProgram(Program program, Map<String, Content> copiedMacroSkills, Map<String, Content> copiedMicroSkills) throws WorkflowException
    {
        ContentValue[] programSkills = program.getValue(Program.SKILLS);
        // Remove the skills without triggering the observer that would delete the skills
        program.removeValue(Program.SKILLS);
        program.saveChanges();

        // If the program has skills, update the links to target the copied macro skills
        if (programSkills != null)
        {
            Content[] skills = _getCopiedSkills(programSkills, copiedMacroSkills);
            
            for (Content skill : skills)
            {
                ModifiableContent skillModifiable = (ModifiableContent) skill;
                // Set the parent program of the copied skills
                skillModifiable.synchronizeValues(Map.of("parentProgram", program.getId()));
                skillModifiable.saveChanges();
                
                // To update the version and indexation
                _editContent(skillModifiable);
            }
            
            program.saveChanges();
        }
        
        ContentValue[] programTransversalSkills = program.getValue(Program.TRANSVERSAL_SKILLS);
        // If the program has transversal skills linked, update the link to target the copied macro skills
        if (programTransversalSkills != null)
        {
            Content[] transversalSkills = _getCopiedSkills(programTransversalSkills, copiedMacroSkills);
            program.setValue(Program.TRANSVERSAL_SKILLS, transversalSkills);
            program.saveChanges();
        }
        
        ContentValue[] programBlockingMicroSkills = program.getValue(Program.BLOCKING_SKILLS);
        // If the program has blocking skills linked, update the link to target the copied micro skills
        if (programBlockingMicroSkills != null)
        {
            Content[] blockingMicroSkills = _getCopiedSkills(programBlockingMicroSkills, copiedMicroSkills);
            program.setValue(Program.BLOCKING_SKILLS, blockingMicroSkills);
            program.saveChanges();
        }
    }
    
    private void _updateCourse(Course course, Map<String, Content> copiedMicroSkills)
    {
        String catalog = course.getCatalog();
        
        // Get the micro skills of the course by program
        List<? extends ModifiableModelAwareRepeaterEntry> microSkillsByProgramEntries = Optional.ofNullable(course.getRepeater(Course.ACQUIRED_MICRO_SKILLS))
                .map(ModifiableModelAwareRepeater::getEntries)
                .orElse(List.of());
        
        for (ModifiableModelAwareRepeaterEntry entry : microSkillsByProgramEntries)
        {
            ContentValue[] microSkills = entry.getValue(Course.ACQUIRED_MICRO_SKILLS_SKILLS, false, new ContentValue[0]);
            Content[] newMicroSkills = _getCopiedSkills(microSkills, copiedMicroSkills);
            entry.setValue(Course.ACQUIRED_MICRO_SKILLS_SKILLS, newMicroSkills);
            
            String programId = Optional.ofNullable(entry.<ContentValue>getValue(Course.ACQUIRED_MICRO_SKILLS_PROGRAM))
                    .flatMap(ContentValue::getContentIfExists)
                    .map(Program.class::cast)
                    .map(p -> _getCopiedProgram(p, catalog))
                    .map(Content::getId)
                    .orElse(null);
            entry.setValue(Course.ACQUIRED_MICRO_SKILLS_PROGRAM, programId);
        }
        course.saveChanges();
    }
    
    private Program _getCopiedProgram(Program program, String newCatalog)
    {
        return _odfHelper.getODFContent(ProgramFactory.PROGRAM_CONTENT_TYPE, program.getCode(), newCatalog, program.getLanguage());
    }
    
    private Content[] _getCopiedSkills(ContentValue[] originalSkills, Map<String, Content> copiedSkills)
    {
        return Arrays.asList(originalSkills)
                     .stream()
                     // Keep the former skill if the copy is not found
                     .map(originalSkill -> copiedSkills.getOrDefault(originalSkill.getContentId(), originalSkill.getContent()))
                     .toArray(Content[]::new);
    }
    
    private boolean _skillExists(Content skill, String contentType, String newCatalogName)
    {
        return _getContents(newCatalogName, contentType, skill).findAny().isPresent();
    }
    
    private <T extends Content> List<T> _getContents(String catalogName, String contentType)
    {
        return this.<T>_getContents(catalogName, contentType, null).toList();
    }
    
    private <T extends Content> Stream<T> _getContents(String catalogName, String contentType, Content skill)
    {
        List<Expression> exprs = new ArrayList<>();
        exprs.add(new ContentTypeExpression(Operator.EQ, contentType));
        exprs.add(new StringExpression("catalog", Operator.EQ, catalogName));
        if (skill != null)
        {
            exprs.add(new StringExpression("code", Operator.EQ, skill.getValue("code")));
        }
        Expression expression = new AndExpression(exprs.toArray(Expression[]::new));
        
        String query = ContentQueryHelper.getContentXPathQuery(expression);
        return _resolver.<T>query(query).stream();
    }
    
    private void _deleteContent(Content content)
    {
        Map<String, Object> eventParams = new HashMap<>();
        eventParams.put(ObservationConstants.ARGS_CONTENT, content);
        eventParams.put(ObservationConstants.ARGS_CONTENT_NAME, content.getName());
        eventParams.put(ObservationConstants.ARGS_CONTENT_ID, content.getId());
        ModifiableAmetysObject parent = content.getParent();
        
        _observationManager.notify(new Event(ObservationConstants.EVENT_CONTENT_DELETING, _currentUserProvider.getUser(), eventParams));
        
        // Remove the content.
        LockableAmetysObject lockedContent = (LockableAmetysObject) content;
        if (lockedContent.isLocked())
        {
            lockedContent.unlock();
        }
        
        ((RemovableAmetysObject) content).remove();
        
        parent.saveChanges();
        
        _observationManager.notify(new Event(ObservationConstants.EVENT_CONTENT_DELETED, _currentUserProvider.getUser(), eventParams));
    }
    
    private void _editContent(Content content) throws AmetysRepositoryException, WorkflowException
    {
        Map<String, Object> parameters = new HashMap<>();
        parameters.put(EditContentFunction.QUIT, true);

        Map<String, Object> inputs = new HashMap<>();
        inputs.put(AbstractWorkflowComponent.CONTEXT_PARAMETERS_KEY, parameters);
        // Force because we don't want to control user rights with this operation
        inputs.put(CheckRightsCondition.FORCE, true);
        
        _contentWorkflowHelper.doAction((WorkflowAwareContent) content, 222, inputs);
    }
}
