/*
 *  Copyright 2023 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.odf.cdmfr;

import static org.ametys.cms.data.type.ModelItemTypeConstants.CONTENT_ELEMENT_TYPE_ID;
import static org.ametys.cms.data.type.ModelItemTypeConstants.FILE_ELEMENT_TYPE_ID;
import static org.ametys.cms.data.type.ModelItemTypeConstants.REFERENCE_ELEMENT_TYPE_ID;
import static org.ametys.cms.data.type.ModelItemTypeConstants.RICH_TEXT_ELEMENT_TYPE_ID;
import static org.ametys.plugins.repository.data.type.ModelItemTypeConstants.COMPOSITE_TYPE_ID;
import static org.ametys.plugins.repository.data.type.ModelItemTypeConstants.REPEATER_TYPE_ID;
import static org.ametys.runtime.model.type.ModelItemTypeConstants.BOOLEAN_TYPE_ID;
import static org.ametys.runtime.model.type.ModelItemTypeConstants.DATE_TYPE_ID;
import static org.ametys.runtime.model.type.ModelItemTypeConstants.DOUBLE_TYPE_ID;
import static org.ametys.runtime.model.type.ModelItemTypeConstants.LONG_TYPE_ID;
import static org.ametys.runtime.model.type.ModelItemTypeConstants.STRING_TYPE_ID;

import java.time.LocalDate;
import java.util.HashMap;
import java.util.Map;
import java.util.Set;

import org.apache.avalon.framework.configuration.Configurable;
import org.apache.avalon.framework.configuration.Configuration;
import org.apache.avalon.framework.configuration.ConfigurationException;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;
import org.apache.cocoon.xml.AttributesImpl;
import org.apache.cocoon.xml.XMLUtils;
import org.apache.excalibur.source.SourceResolver;
import org.xml.sax.ContentHandler;
import org.xml.sax.SAXException;

import org.ametys.cms.contenttype.ContentTypesHelper;
import org.ametys.cms.data.ContentValue;
import org.ametys.cms.data.File;
import org.ametys.cms.data.Reference;
import org.ametys.cms.data.RichText;
import org.ametys.cms.repository.Content;
import org.ametys.odf.course.Course;
import org.ametys.odf.enumeration.OdfReferenceTableHelper;
import org.ametys.odf.orgunit.OrgUnit;
import org.ametys.odf.person.Person;
import org.ametys.odf.program.AbstractProgram;
import org.ametys.odf.program.Container;
import org.ametys.odf.program.Program;
import org.ametys.odf.program.ProgramFactory;
import org.ametys.odf.program.SubProgram;
import org.ametys.plugins.repository.data.holder.ModelAwareDataHolder;
import org.ametys.plugins.repository.data.holder.group.ModelAwareComposite;
import org.ametys.plugins.repository.data.holder.group.ModelAwareRepeater;
import org.ametys.plugins.repository.data.holder.group.ModelAwareRepeaterEntry;
import org.ametys.plugins.repository.model.RepositoryDataContext;
import org.ametys.runtime.i18n.I18nizableText;
import org.ametys.runtime.model.ElementDefinition;
import org.ametys.runtime.model.Enumerator;
import org.ametys.runtime.model.ModelItem;
import org.ametys.runtime.model.ModelItemGroup;
import org.ametys.runtime.model.View;
import org.ametys.runtime.model.type.DataContext;
import org.ametys.runtime.model.type.ModelItemType;

/**
 * Simple {@link CDMfrExtension} generating CDM-fr Ametys extension for each configured attribute.<br>
 * Each attribute should be configured like:<br><br>
 * <code>&lt;attribute name="attributeName" tag="tagName"></code><br><br>
 * and will be output with the following syntax:<br><br>
 * <code>&lt;ametys-cdm:tagName>value&lt;/ametys-cdm:tagName></code><br><br>
 * The tag name is optional, defaulting to the attribute name.<br>
 * It the attribute is of type "content", then the "cdm" view is also exported inside the attribute's XML tag if it exists.
 */
public class GenericCDMfrExtension extends AbstractCDMfrExtension implements Configurable, Serviceable
{
    private SourceResolver _sourceResolver;
    private OdfReferenceTableHelper _refTableHelper;
    private ContentTypesHelper _contentTypesHelper;
    
    private Map<String, String> _abstractPrograms = new HashMap<>();
    private Map<String, String> _programs = new HashMap<>();
    private Map<String, String> _subPrograms = new HashMap<>();
    private Map<String, String> _containers = new HashMap<>();
    private Map<String, String> _courses = new HashMap<>();
    private Map<String, String> _orgUnits = new HashMap<>();
    private Map<String, String> _persons = new HashMap<>();
    
    public void service(ServiceManager manager) throws ServiceException
    {
        _sourceResolver = (SourceResolver) manager.lookup(SourceResolver.ROLE);
        _refTableHelper = (OdfReferenceTableHelper) manager.lookup(OdfReferenceTableHelper.ROLE);
        _contentTypesHelper = (ContentTypesHelper) manager.lookup(ContentTypesHelper.ROLE);
    }
    
    public void configure(Configuration configuration) throws ConfigurationException
    {
        _abstractPrograms = _configure(configuration, "abstractProgram");
        _programs = _configure(configuration, "program");
        _subPrograms = _configure(configuration, "subProgram");
        _containers = _configure(configuration, "container");
        _courses = _configure(configuration, "course");
        _orgUnits = _configure(configuration, "orgUnit");
        _persons = _configure(configuration, "person");
    }
    
    private Map<String, String> _configure(Configuration configuration, String attributeSet) throws ConfigurationException
    {
        Map<String, String> attributeMap = new HashMap<>();
        
        for (Configuration conf : configuration.getChild(attributeSet, true).getChildren("attribute"))
        {
            String name = conf.getAttribute("name");
            String tag = conf.getAttribute("tag", name);
            
            attributeMap.put(name, tag);
        }
        
        return attributeMap;
    }
    
    public void abstractProgram2CDM(ContentHandler contentHandler, AbstractProgram<? extends ProgramFactory> program, Set<String> persons, Set<String> orgUnits) throws SAXException
    {
        _attributes2CDM(contentHandler, program, _abstractPrograms);
    }

    public void program2CDM(ContentHandler contentHandler, Program program, Set<String> persons, Set<String> orgUnits) throws SAXException
    {
        _attributes2CDM(contentHandler, program, _programs);
    }
    
    public void subProgram2CDM(ContentHandler contentHandler, SubProgram subProgram, Set<String> persons, Set<String> orgUnits) throws SAXException
    {
        _attributes2CDM(contentHandler, subProgram, _subPrograms);
    }

    public void course2CDM(ContentHandler contentHandler, Course course, Set<String> persons, Set<String> orgUnits) throws SAXException
    {
        _attributes2CDM(contentHandler, course, _courses);
    }

    public void orgunit2CDM(ContentHandler contentHandler, OrgUnit orgunit) throws SAXException
    {
        _attributes2CDM(contentHandler, orgunit, _orgUnits);
    }

    public void person2CDM(ContentHandler contentHandler, Person person) throws SAXException
    {
        _attributes2CDM(contentHandler, person, _persons);
    }

    public void container2CDM(ContentHandler contentHandler, Container container, Set<String> persons, Set<String> orgUnits) throws SAXException
    {
        _attributes2CDM(contentHandler, container, _containers);
    }
    
    private void _attributes2CDM(ContentHandler contentHandler, Content content, Map<String, String> attributes) throws SAXException
    {
        for (Map.Entry<String, String> entry : attributes.entrySet())
        {
            _attribute2CDM(contentHandler, content, content, null, entry.getKey(), entry.getValue());
        }
    }

    private void _attribute2CDM(ContentHandler contentHandler, ModelAwareDataHolder dataHolder, Content initialContent, String path, String attributeName, String tagName) throws SAXException
    {
        Object value = dataHolder.getValue(attributeName, false, null);

        if (value == null)
        {
            return;
        }
        
        String dataPath = path == null ? attributeName : path + ModelItem.ITEM_PATH_SEPARATOR + attributeName;

        ModelItem modelItem = dataHolder.getDefinition(attributeName);
        ModelItemType type = modelItem.getType();
        String typeId = type.getId();
        
        if (modelItem instanceof ElementDefinition definition)
        {
            Object[] values = definition.isMultiple() ? (Object[]) value : new Object[] {value};
            
            switch (typeId)
            {
                case BOOLEAN_TYPE_ID:
                case STRING_TYPE_ID:
                case DOUBLE_TYPE_ID:
                case LONG_TYPE_ID:
                    _simple2CDM(contentHandler, values, tagName, definition);
                    break;
                case REFERENCE_ELEMENT_TYPE_ID:
                    _reference2CDM(contentHandler, values, tagName);
                    break;
                case DATE_TYPE_ID:
                    _date2CDM(contentHandler, values, tagName);
                    break;
                case RICH_TEXT_ELEMENT_TYPE_ID:
                    _richText2CDM(contentHandler, values, tagName, initialContent, dataPath);
                    break;
                case CONTENT_ELEMENT_TYPE_ID:
                    _content2CDM(contentHandler, values, tagName, initialContent);
                    break;
                case FILE_ELEMENT_TYPE_ID:
                    _file2CDM(contentHandler, values, tagName, initialContent, dataPath);
                    break;
                default:
                    // Ignore it
                    break;
            }
        }
        else if (typeId.equals(COMPOSITE_TYPE_ID))
        {
            ModelAwareComposite composite = (ModelAwareComposite) value;
            ModelItemGroup group = (ModelItemGroup) modelItem;
            
            _composite2CDM(contentHandler, composite, group, initialContent, dataPath, tagName);
        }
        else if (typeId.equals(REPEATER_TYPE_ID))
        {
            ModelAwareRepeater repeater = (ModelAwareRepeater) value;
            ModelItemGroup group = (ModelItemGroup) modelItem;
            
            _repeater2CDM(contentHandler, repeater, group, initialContent, dataPath, tagName);
        }
    }
    
    private void _simple2CDM(ContentHandler contentHandler, Object[] values, String tagName, ElementDefinition definition) throws SAXException
    {
        Enumerator enumerator = definition.getEnumerator();
        
        for (Object v : values)
        {
            if (enumerator == null)
            {
                XMLUtils.createElement(contentHandler, CDMFRTagsConstants.NAMESPACE_AMETYS_CDM + tagName, v.toString());
            }
            else
            {
                try
                {
                    @SuppressWarnings("unchecked")
                    I18nizableText label = enumerator.getEntry(v);
                    
                    AttributesImpl atts = new AttributesImpl();
                    atts.addCDATAAttribute("value", v.toString());
                    
                    XMLUtils.startElement(contentHandler, CDMFRTagsConstants.NAMESPACE_AMETYS_CDM + tagName, atts);
                    label.toSAX(contentHandler);
                    XMLUtils.endElement(contentHandler, CDMFRTagsConstants.NAMESPACE_AMETYS_CDM + tagName);
                }
                catch (Exception e)
                {
                    throw new SAXException("Cannot retrieve enumerated label for value " + v + " for attribute " + definition.getName(), e);
                }
            }
        }
    }
    
    private void _reference2CDM(ContentHandler contentHandler, Object[] values, String tagName) throws SAXException
    {
        for (Object v : values)
        {
            Reference ref = (Reference) v;

            AttributesImpl attrs = new AttributesImpl();
            attrs.addCDATAAttribute("type", ref.getType());
            XMLUtils.createElement(contentHandler, CDMFRTagsConstants.NAMESPACE_AMETYS_CDM + tagName, attrs, ref.getValue());
        }
    }
    
    private void _date2CDM(ContentHandler contentHandler, Object[] values, String tagName) throws SAXException
    {
        for (Object v : values)
        {
            CDMHelper.date2CDM(contentHandler, CDMFRTagsConstants.NAMESPACE_AMETYS_CDM + tagName, (LocalDate) v);
        }
    }
    
    private void _richText2CDM(ContentHandler contentHandler, Object[] values, String tagName, Content content, String dataPath) throws SAXException
    {
        for (Object v : values)
        {
            DataContext context = RepositoryDataContext.newInstance()
                                                       .withObject(content)
                                                       .withDataPath(dataPath);
            
            CDMHelper.richText2CDM(contentHandler, CDMFRTagsConstants.NAMESPACE_AMETYS_CDM + tagName, (RichText) v, context, _sourceResolver);
        }
    }
    
    private void _content2CDM(ContentHandler contentHandler, Object[] values, String tagName, Content initialContent) throws SAXException
    {
        for (Object v : values)
        {
            Content content = ((ContentValue) v).getContentIfExists().orElse(null);
            
            if (content != null)
            {
                if (_refTableHelper.isTableReferenceEntry(content))
                {
                    AttributesImpl attrs = new AttributesImpl();
                    attrs.addCDATAAttribute("id", content.getId());
                    attrs.addCDATAAttribute("code", _refTableHelper.getItemCDMfrValue(content.getId(), true));
                    attrs.addCDATAAttribute("title", _refTableHelper.getItemLabel(content.getId(), initialContent.getLanguage()));
                    
                    XMLUtils.startElement(contentHandler, CDMFRTagsConstants.NAMESPACE_AMETYS_CDM + tagName, attrs);
                    _contentView2CDM(contentHandler, content);
                    XMLUtils.endElement(contentHandler, CDMFRTagsConstants.NAMESPACE_AMETYS_CDM + tagName);
                }
                else
                {
                    AttributesImpl attrs = new AttributesImpl();
                    attrs.addCDATAAttribute("id", content.getId());
                    attrs.addCDATAAttribute("title", content.getTitle());
                    
                    XMLUtils.startElement(contentHandler, CDMFRTagsConstants.NAMESPACE_AMETYS_CDM + tagName, attrs);
                    _contentView2CDM(contentHandler, content);
                    XMLUtils.endElement(contentHandler, CDMFRTagsConstants.NAMESPACE_AMETYS_CDM + tagName);
                }
            }
        }
    }
    
    private void _contentView2CDM(ContentHandler contentHandler, Content content) throws SAXException
    {
        DataContext context = RepositoryDataContext.newInstance()
                                                   .withObject(content);

        View view = _contentTypesHelper.getView("cdm", content);
        
        if (view != null)
        {
            content.dataToSAX(contentHandler, view, context);
        }
    }
    
    private void _file2CDM(ContentHandler contentHandler, Object[] values, String tagName, Content content, String dataPath) throws SAXException
    {
        for (Object v : values)
        {
            File file = (File) v;
            XMLUtils.createElement(contentHandler, CDMFRTagsConstants.NAMESPACE_AMETYS_CDM + tagName, _getFileAbsoluteUrl(content, file, dataPath));
        }
    }
    
    private void _composite2CDM(ContentHandler contentHandler, ModelAwareComposite composite, ModelItemGroup group, Content initialContent, String dataPath, String tagName) throws SAXException
    {
        XMLUtils.startElement(contentHandler, CDMFRTagsConstants.NAMESPACE_AMETYS_CDM + tagName);
        
        for (ModelItem subAttribute : group.getChildren())
        {
            _attribute2CDM(contentHandler, composite, initialContent, dataPath, subAttribute.getName(), subAttribute.getName());
        }
        
        XMLUtils.endElement(contentHandler, CDMFRTagsConstants.NAMESPACE_AMETYS_CDM + tagName);
    }
    
    private void _repeater2CDM(ContentHandler contentHandler, ModelAwareRepeater repeater, ModelItemGroup group, Content initialContent, String dataPath, String tagName) throws SAXException
    {
        
        for (ModelAwareRepeaterEntry entry : repeater.getEntries())
        {
            XMLUtils.startElement(contentHandler, CDMFRTagsConstants.NAMESPACE_AMETYS_CDM + tagName);
            
            for (ModelItem subAttribute : group.getChildren())
            {
                _attribute2CDM(contentHandler, entry, initialContent, dataPath + "[" + entry.getPosition() + "]", subAttribute.getName(), subAttribute.getName());
            }
            
            XMLUtils.endElement(contentHandler, CDMFRTagsConstants.NAMESPACE_AMETYS_CDM + tagName);
        }
    }
}
