/*
 *  Copyright 2022 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.odf.init;

import java.io.IOException;
import java.net.URI;
import java.nio.file.FileSystems;
import java.nio.file.Files;
import java.nio.file.Path;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.apache.avalon.framework.configuration.Configurable;
import org.apache.avalon.framework.configuration.Configuration;
import org.apache.avalon.framework.configuration.ConfigurationException;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;
import org.apache.excalibur.source.SourceResolver;
import org.apache.excalibur.source.TraversableSource;

import org.ametys.runtime.plugin.component.AbstractLogEnabled;
import org.ametys.runtime.plugin.component.PluginAware;

/**
 * ODF reference table data description to import on initialization.
 * This implementation only works outside a JAR file.
 */
public class OdfRefTableDataAsFolder extends AbstractLogEnabled implements OdfRefTableData, PluginAware, Serviceable, Configurable
{
    /** The source resolver component */
    protected SourceResolver _srcResolver;
    
    /** The plugin name */
    protected String _pluginName;
    /** The extension id */
    protected String _id;

    /** List of data files to import */
    protected String _dataFolder;
    /** The priority */
    protected Integer _priority;
    
    public void setPluginInfo(String pluginName, String featureName, String id)
    {
        _pluginName = pluginName;
        _id = id;
    }
    
    public void service(ServiceManager manager) throws ServiceException
    {
        _srcResolver = (SourceResolver) manager.lookup(SourceResolver.ROLE);
    }
    
    public void configure(Configuration configuration) throws ConfigurationException
    {
        _priority = configuration.getChild("priority").getValueAsInteger(50);
        _dataFolder = configuration.getChild("files").getAttribute("folder", "plugin:" + _pluginName + "://ref-data");
    }
    
    public int getPriority()
    {
        return _priority;
    }
    
    public Map<String, String> getDataToImport()
    {
        Map<String, String> dataToImport = new HashMap<>();
        
        for (String dataFile : getDataFiles())
        {
            String contentType = "odf-enumeration." + dataFile.substring(dataFile.lastIndexOf(FileSystems.getDefault().getSeparator()) + 1, dataFile.lastIndexOf("."));
            dataToImport.put(contentType, dataFile);
        }
        
        if (getLogger().isInfoEnabled())
        {
            getLogger().info("Data files found in extension '{}': {}", _id, dataToImport.toString());
        }
        
        return dataToImport;
    }
    
    /**
     * Get the list of {@link Path} including data files to import.
     * @return The list of files to import
     */
    protected List<String> getDataFiles()
    {
        TraversableSource refDataSource = null;
        try
        {
            refDataSource = (TraversableSource) _srcResolver.resolveURI(_dataFolder);
            if (refDataSource.exists() && refDataSource.isCollection())
            {
                Path refDataPath = Path.of(URI.create(refDataSource.getURI()));
                return Files.list(refDataPath)
                        .map(Path::toString)
                        .toList();
            }
        }
        catch (IOException e)
        {
            getLogger().error("Problem while getting ODF reference table data to import for extension {}.", _id, e);
        }
        finally
        {
            if (refDataSource != null)
            {
                _srcResolver.release(refDataSource);
            }
        }
        
        return List.of();
    }
}
