/*
 *  Copyright 2010 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.odf.program.generators;

import java.io.IOException;
import java.util.HashMap;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Set;

import org.apache.avalon.framework.activity.Initializable;
import org.apache.avalon.framework.parameters.Parameters;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.cocoon.ProcessingException;
import org.apache.cocoon.components.source.impl.SitemapSource;
import org.apache.cocoon.environment.ObjectModelHelper;
import org.apache.cocoon.environment.Request;
import org.apache.cocoon.generation.Generator;
import org.apache.cocoon.xml.AttributesImpl;
import org.apache.cocoon.xml.SaxBuffer;
import org.apache.cocoon.xml.XMLUtils;
import org.apache.commons.lang.StringUtils;
import org.apache.commons.lang3.LocaleUtils;
import org.apache.commons.lang3.tuple.Triple;
import org.apache.excalibur.source.SourceResolver;
import org.xml.sax.ContentHandler;
import org.xml.sax.SAXException;

import org.ametys.cms.CmsConstants;
import org.ametys.cms.content.ContentHelper;
import org.ametys.cms.contenttype.ContentAttributeDefinition;
import org.ametys.cms.contenttype.ContentType;
import org.ametys.cms.data.ContentDataHelper;
import org.ametys.cms.data.ContentValue;
import org.ametys.cms.repository.Content;
import org.ametys.core.cache.AbstractCacheManager;
import org.ametys.core.cache.Cache;
import org.ametys.core.util.DateUtils;
import org.ametys.core.util.IgnoreRootHandler;
import org.ametys.odf.NoLiveVersionException;
import org.ametys.odf.ODFHelper;
import org.ametys.odf.course.Course;
import org.ametys.odf.courselist.CourseList;
import org.ametys.odf.courselist.CourseList.ChoiceType;
import org.ametys.odf.coursepart.CoursePart;
import org.ametys.odf.orgunit.OrgUnit;
import org.ametys.odf.orgunit.OrgUnitFactory;
import org.ametys.odf.person.Person;
import org.ametys.odf.person.PersonFactory;
import org.ametys.odf.program.AbstractProgram;
import org.ametys.odf.program.Container;
import org.ametys.odf.program.Program;
import org.ametys.odf.program.ProgramPart;
import org.ametys.odf.program.SubProgram;
import org.ametys.odf.program.TraversableProgramPart;
import org.ametys.odf.translation.TranslationHelper;
import org.ametys.plugins.repository.UnknownAmetysObjectException;
import org.ametys.plugins.repository.data.holder.ModelAwareDataHolder;
import org.ametys.plugins.repository.data.holder.group.ModelAwareComposite;
import org.ametys.plugins.repository.data.holder.group.ModelAwareRepeater;
import org.ametys.plugins.repository.data.holder.group.ModelAwareRepeaterEntry;
import org.ametys.plugins.repository.jcr.DefaultAmetysObject;
import org.ametys.plugins.repository.model.RepeaterDefinition;
import org.ametys.runtime.i18n.I18nizableText;
import org.ametys.runtime.model.ModelItem;
import org.ametys.runtime.model.ModelItemContainer;
import org.ametys.runtime.model.View;
import org.ametys.runtime.model.type.DataContext;

/**
 * {@link Generator} for rendering raw content data for a {@link Program}.
 */
public class ProgramContentGenerator extends ODFContentGenerator implements Initializable
{
    private static final String __CACHE_ID = ProgramContentGenerator.class.getName() + "$linkedContents";
    
    /** The source resolver */
    protected SourceResolver _srcResolver;
    
    /** The ODF helper */
    protected ODFHelper _odfHelper;
    
    /** The content helper */
    protected ContentHelper _contentHelper;
    
    private AbstractCacheManager _cacheManager;
    private Cache<Triple<String, String, String>, SaxBuffer> _cache;

    @Override
    public void service(ServiceManager serviceManager) throws ServiceException
    {
        super.service(serviceManager);
        _srcResolver = (SourceResolver) serviceManager.lookup(SourceResolver.ROLE);
        _odfHelper = (ODFHelper) serviceManager.lookup(ODFHelper.ROLE);
        _cacheManager = (AbstractCacheManager) serviceManager.lookup(AbstractCacheManager.ROLE);
        _contentHelper = (ContentHelper) serviceManager.lookup(ContentHelper.ROLE);
    }
    
    public void initialize() throws Exception
    {
        if (!_cacheManager.hasCache(__CACHE_ID))
        {
            _cacheManager.createRequestCache(__CACHE_ID,
                                      new I18nizableText("plugin.odf", "PLUGINS_ODF_CACHE_PROGRAMCONTENTGENERATOR_LABEL"),
                                      new I18nizableText("plugin.odf", "PLUGINS_ODF_CACHE_PROGRAMCONTENTGENERATOR_DESCRIPTION"),
                                      false);
        }
    }
    
    @Override
    public void setup(org.apache.cocoon.environment.SourceResolver res, Map objModel, String src, Parameters par) throws ProcessingException, SAXException, IOException
    {
        super.setup(res, objModel, src, par);
        _cache = _cacheManager.get(__CACHE_ID);
    }
    
    @Override
    public void recycle()
    {
        super.recycle();
        _cache = null;
    }
    
    @Override
    protected void _saxOtherData(Content content, Locale defaultLocale) throws SAXException, ProcessingException, IOException
    {
        super._saxOtherData(content, defaultLocale);
        
        boolean isEdition = parameters.getParameterAsBoolean("isEdition", false);
        if (!isEdition)
        {
            if (content instanceof AbstractProgram)
            {
                AbstractProgram program = (AbstractProgram) content;
                
                // Contacts
                saxPersons(program);
                
                if (_odfHelper.useLegacyProgramStructure(program))
                {
                    // Child containers, subprograms or course lists
                    saxChildProgramPart(program, defaultLocale);
                }
                
                // OrgUnits
                saxOrgUnits(program);
                
                // Translations
                saxTranslation(program);
            }
        }
        
        Request request = ObjectModelHelper.getRequest(objectModel);
        request.setAttribute(Content.class.getName(), content);
    }
    
    /**
     * SAX the referenced {@link Person}s
     * @param content The content to analyze
     * @throws SAXException if an error occurs while saxing
     */
    protected void saxPersons(Content content) throws SAXException
    {
        saxLinkedContents(content, "persons", PersonFactory.PERSON_CONTENT_TYPE, "abstract");
    }
    
    /**
     * SAX the referenced {@link OrgUnit}s
     * @param content The content to analyze
     * @throws SAXException if an error occurs while saxing
     */
    protected void saxOrgUnits(Content content) throws SAXException
    {
        saxLinkedContents(content, "orgunits", OrgUnitFactory.ORGUNIT_CONTENT_TYPE, "link");
    }
    
    /**
     * SAX the referenced {@link ProgramPart}s
     * @param program The program or subprogram
     * @param defaultLocale The default locale
     * @throws SAXException if an error occurs while saxing
     * @throws IOException if an error occurs
     * @throws ProcessingException if an error occurs
     */
    protected void saxChildProgramPart(AbstractProgram program, Locale defaultLocale) throws SAXException, ProcessingException, IOException
    {
        ContentValue[] children = program.getValue(TraversableProgramPart.CHILD_PROGRAM_PARTS, false, new ContentValue[0]);
        for (ContentValue child : children)
        {
            try
            {
                Content content = child.getContent();
                _odfHelper.switchToLiveVersionIfNeeded((DefaultAmetysObject) content);
                
                if (content instanceof SubProgram)
                {
                    saxSubProgram((SubProgram) content, program.getContextPath());
                }
                else if (content instanceof Container)
                {
                    saxContainer ((Container) content, program.getContextPath(), defaultLocale);
                }
                else if (content instanceof CourseList)
                {
                    saxCourseList((CourseList) content, program.getContextPath(), defaultLocale);
                }
            }
            catch (UnknownAmetysObjectException | NoLiveVersionException e)
            {
                // The content can reference a non-existing child: ignore the exception.
                // Or the content has no live version: ignore too
            }
        }
    }
    
    /**
     * SAX the existing translation
     * @param content The content
     * @throws SAXException if an error occurs while saxing
     */
    protected void saxTranslation(Content content) throws SAXException
    {
        Map<String, String> translations = TranslationHelper.getTranslations(content);
        if (!translations.isEmpty())
        {
            saxTranslations(translations);
        }
    }
    
    /**
     * SAX the referenced content types.
     * @param content The content to analyze
     * @param tagName The root tagName
     * @param linkedContentType The content type to search
     * @param viewName The view to parse the found contents
     * @throws SAXException if an error occurs while saxing
     */
    protected void saxLinkedContents(Content content, String tagName, String linkedContentType, String viewName) throws SAXException
    {
        Set<String> contentIds = getLinkedContents(content, linkedContentType);
        
        XMLUtils.startElement(contentHandler, tagName);
        for (String id : contentIds)
        {
            if (StringUtils.isNotEmpty(id))
            {
                try
                {
                    Content subContent = _resolver.resolveById(id);
                    saxContent(subContent, contentHandler, viewName);
                }
                catch (IOException e)
                {
                    throw new SAXException(e);
                }
                catch (UnknownAmetysObjectException e)
                {
                    // The program can reference a non-existing person: ignore the exception.
                }
            }
        }
        
        XMLUtils.endElement(contentHandler, tagName);
    }
    
    /**
     * Get the linked contents of the defined content type.
     * @param content The content to analyze
     * @param linkedContentType The content type to search
     * @return A {@link Set} of content ids
     */
    protected Set<String> getLinkedContents(Content content, String linkedContentType)
    {
        Set<String> contentIds = new LinkedHashSet<>();
        
        for (String cTypeId : content.getTypes())
        {
            ContentType cType = _contentTypeExtensionPoint.getExtension(cTypeId);
            contentIds.addAll(_getContentIds(content, cType, linkedContentType));
        }
        
        return contentIds;
    }
    
    private Set<String> _getContentIds(ModelAwareDataHolder dataHolder, ModelItemContainer modelItemContainer, String contentType)
    {
        Set<String> contentIds = new LinkedHashSet<>();
        
        for (ModelItem modelItem : modelItemContainer.getModelItems())
        {
            if (modelItem instanceof ContentAttributeDefinition)
            {
                if (contentType.equals(((ContentAttributeDefinition) modelItem).getContentTypeId()))
                {
                    String modelItemName = modelItem.getName();
                    if (dataHolder.isMultiple(modelItemName))
                    {
                        List<String> values = ContentDataHelper.getContentIdsListFromMultipleContentData(dataHolder, modelItemName);
                        contentIds.addAll(values);
                    }
                    else
                    {
                        contentIds.add(ContentDataHelper.getContentIdFromContentData(dataHolder, modelItemName));
                    }
                }
            }
            else if (modelItem instanceof ModelItemContainer)
            {

                if (dataHolder.hasValue(modelItem.getName()))
                {
                    if (modelItem instanceof RepeaterDefinition)
                    {
                        ModelAwareRepeater repeater = dataHolder.getRepeater(modelItem.getName());
                        for (ModelAwareRepeaterEntry entry : repeater.getEntries())
                        {
                            contentIds.addAll(_getContentIds(entry, (ModelItemContainer) modelItem, contentType));
                        }
                    }
                    else
                    {
                        ModelAwareComposite composite = dataHolder.getComposite(modelItem.getName());
                        contentIds.addAll(_getContentIds(composite, (ModelItemContainer) modelItem, contentType));
                    }
                }
            }
        }
        
        return contentIds;
    }
    
    /**
     * SAX a container
     * @param container the container to SAX
     * @param parentPath the parent path
     * @param defaultLocale The default locale
     * @throws SAXException if an error occurs
     * @throws IOException if an error occurs
     * @throws ProcessingException if an error occurs
     */
    protected void saxContainer(Container container, String parentPath, Locale defaultLocale) throws SAXException, ProcessingException, IOException
    {
        AttributesImpl attrs = new AttributesImpl();
        attrs.addCDATAAttribute("title", container.getTitle());
        attrs.addCDATAAttribute("id", container.getId());
        _addAttrIfNotEmpty(attrs, "code", container.getCode());
        _addAttrIfNotEmpty(attrs, "nature", container.getNature());
        double ects = container.getEcts();
        if (ects > 0)
        {
            attrs.addCDATAAttribute("ects", String.valueOf(ects));
        }
        
        XMLUtils.startElement(contentHandler, "container", attrs);
        
        // Children
        ContentValue[] children = container.getValue(TraversableProgramPart.CHILD_PROGRAM_PARTS, false, new ContentValue[0]);
        for (ContentValue child : children)
        {
            try
            {
                Content content = child.getContent();
                if (content instanceof SubProgram)
                {
                    saxSubProgram((SubProgram) content, parentPath);
                }
                else if (content instanceof Container)
                {
                    saxContainer ((Container) content, parentPath, defaultLocale);
                }
                else if (content instanceof CourseList)
                {
                    saxCourseList((CourseList) content, parentPath, defaultLocale);
                }
            }
            catch (UnknownAmetysObjectException e)
            {
                // The content can reference a non-existing child (in live for example): ignore the exception.
            }
        }
        
        XMLUtils.endElement(contentHandler, "container");
    }
    
    /**
     * SAX a sub program
     * @param subProgram the sub program to SAX
     * @param parentPath the parent path
     * @throws SAXException if an error occurs
     */
    protected void saxSubProgram(SubProgram subProgram, String parentPath) throws SAXException
    {
        AttributesImpl attrs = new AttributesImpl();
        attrs.addCDATAAttribute("title", subProgram.getTitle());
        attrs.addCDATAAttribute("id", subProgram.getId());
        _addAttrIfNotEmpty(attrs, "code", subProgram.getCode());
        _addAttrIfNotEmpty(attrs, "ects", subProgram.getEcts());
        
        if (parentPath != null)
        {
            attrs.addCDATAAttribute("path", parentPath + "/" + subProgram.getName());
        }
        XMLUtils.startElement(contentHandler, "subprogram", attrs);
        
        try
        {
            // SAX the "parcours" view of a subprogram
            saxContent(subProgram, contentHandler, "parcours", "xml", false, true);
        }
        catch (IOException e)
        {
            throw new SAXException(e);
        }
        
        XMLUtils.endElement(contentHandler, "subprogram");
    }
    
    /**
     * SAX a course list
     * @param courseList The course list to SAX
     * @param parentPath the parent path
     * @param defaultLocale The default locale
     * @throws SAXException if an error occurs
     * @throws ProcessingException if an error occurs
     */
    protected void saxCourseList(CourseList courseList, String parentPath, Locale defaultLocale) throws SAXException, ProcessingException
    {
        AttributesImpl attrs = new AttributesImpl();
        attrs.addCDATAAttribute("title", courseList.getTitle());
        attrs.addCDATAAttribute("id", courseList.getId());
        _addAttrIfNotEmpty(attrs, "code", courseList.getCode());
        
        ChoiceType type = courseList.getType();
        if (type != null)
        {
            attrs.addCDATAAttribute("type", type.toString());
        }
        
        if (ChoiceType.CHOICE.equals(type))
        {
            attrs.addCDATAAttribute("min", String.valueOf(courseList.getMinNumberOfCourses()));
            attrs.addCDATAAttribute("max", String.valueOf(courseList.getMaxNumberOfCourses()));
        }
        
        XMLUtils.startElement(contentHandler, "courseList", attrs);
        
        for (Course course : courseList.getCourses())
        {
            try
            {
                saxCourse(course, parentPath, defaultLocale);
            }
            catch (UnknownAmetysObjectException e)
            {
                // The content can reference a non-existing course (in live for example): ignore the exception.
            }
        }
        XMLUtils.endElement(contentHandler, "courseList");
    }
    
    /**
     * SAX a course
     * @param course the course to SAX
     * @param parentPath the parent path
     * @param defaultLocale The default locale
     * @throws SAXException if an error occurs
     * @throws ProcessingException if an error occurs
     */
    protected void saxCourse(Course course, String parentPath, Locale defaultLocale) throws SAXException, ProcessingException
    {
        AttributesImpl attrs = new AttributesImpl();
        attrs.addCDATAAttribute("title", course.getTitle());
        attrs.addCDATAAttribute("code", course.getCode());
        attrs.addCDATAAttribute("id", course.getId());
        attrs.addCDATAAttribute("name", course.getName());
        
        if (parentPath != null)
        {
            attrs.addCDATAAttribute("path", parentPath + "/" + course.getName() + "-" + course.getCode());
        }
        
        View view = _cTypesHelper.getView("courseList", course.getTypes(), course.getMixinTypes());
        
        XMLUtils.startElement(contentHandler, "course", attrs);
        
        if (view != null)
        {
            XMLUtils.startElement(contentHandler, "metadata");
            course.dataToSAX(contentHandler, view, DataContext.newInstance().withEmptyValues(false));
            XMLUtils.endElement(contentHandler, "metadata");
        }
        
        // Course list
        for (CourseList childCl : course.getCourseLists())
        {
            saxCourseList(childCl, parentPath, defaultLocale);
        }
        
        // Course parts
        for (CoursePart coursePart : course.getCourseParts())
        {
            saxCoursePart(coursePart, defaultLocale);
        }
        
        XMLUtils.endElement(contentHandler, "course");
    }
    
    /**
     * SAX the HTML content of a {@link Content}
     * @param content the content
     * @param handler the {@link ContentHandler} to send SAX events to.
     * @param viewName the view name
     * @throws SAXException If an error occurred saxing the content
     * @throws IOException If an error occurred resolving the content
     */
    protected void saxContent(Content content, ContentHandler handler, String viewName) throws SAXException, IOException
    {
        String format = parameters.getParameter("output-format", "html");
        if (StringUtils.isEmpty(format))
        {
            format = "html";
        }
        
        saxContent(content, handler, viewName, format, true, false);
    }
    
    /**
     * SAX a {@link Content} to given format
     * @param content the content
     * @param handler the {@link ContentHandler} to send SAX events to.
     * @param viewName the view name
     * @param format the output format
     * @param withContentRoot true to wrap content stream into a root content tag
     * @param ignoreChildren true to not SAX sub contents
     * @throws SAXException If an error occurred saxing the content
     * @throws IOException If an error occurred resolving the content
     */
    protected void saxContent(Content content, ContentHandler handler, String viewName, String format, boolean withContentRoot, boolean ignoreChildren) throws SAXException, IOException
    {
        Triple<String, String, String> cacheKey = Triple.of(content.getId(), viewName, format);
        SaxBuffer buffer = _cache.get(cacheKey);
        
        if (buffer != null)
        {
            buffer.toSAX(handler);
            return;
        }
        
        buffer = new SaxBuffer();

        Request request = ObjectModelHelper.getRequest(objectModel);
        
        SitemapSource src = null;
        try
        {
            Map<String, String> params = new HashMap<>();
            if (ignoreChildren)
            {
                params.put("ignoreChildren", "true");
            }
            
            if (request.getAttribute(ODFHelper.REQUEST_ATTRIBUTE_VALID_LABEL) != null)
            {
                params.put("versionLabel", CmsConstants.LIVE_LABEL);
            }
            
            String uri = _contentHelper.getContentHtmlViewUrl(content, viewName, params);
            src = (SitemapSource) _srcResolver.resolveURI(uri);
            
            if (withContentRoot)
            {
                AttributesImpl attrs = new AttributesImpl();
                attrs.addCDATAAttribute("id", content.getId());
                attrs.addCDATAAttribute("name", content.getName());
                attrs.addCDATAAttribute("title", content.getTitle());
                attrs.addCDATAAttribute("lastModifiedAt", DateUtils.zonedDateTimeToString(content.getLastModified()));
                
                XMLUtils.startElement(buffer, "content", attrs);
            }
            
            src.toSAX(new IgnoreRootHandler(buffer));
            
            if (withContentRoot)
            {
                XMLUtils.endElement(buffer, "content");
            }
            
            _cache.put(cacheKey, buffer);
            buffer.toSAX(handler);
        }
        catch (UnknownAmetysObjectException e)
        {
            // The content may be archived
        }
        finally
        {
            _srcResolver.release(src);
        }
    }
    
    /**
     * Add an attribute if its not null or empty.
     * @param attrs The attributes
     * @param attrName The attribute name
     * @param attrValue The attribute value
     */
    protected void _addAttrIfNotEmpty(AttributesImpl attrs, String attrName, String attrValue)
    {
        if (StringUtils.isNotEmpty(attrValue))
        {
            attrs.addCDATAAttribute(attrName, attrValue);
        }
    }
    
    /**
     * SAX a course part
     * @param coursePart The course part to SAX
     * @param defaultLocale The default locale
     * @throws SAXException if an error occurs
     * @throws ProcessingException if an error occurs
     */
    protected void saxCoursePart(CoursePart coursePart, Locale defaultLocale) throws SAXException, ProcessingException
    {
        AttributesImpl attrs = new AttributesImpl();
        attrs.addCDATAAttribute("id", coursePart.getId());
        attrs.addCDATAAttribute("title", coursePart.getTitle());
        _addAttrIfNotEmpty(attrs, "code", coursePart.getCode());

        XMLUtils.startElement(contentHandler, "coursePart", attrs);
        View view = _cTypesHelper.getView("sax", coursePart.getTypes(), coursePart.getMixinTypes());
        coursePart.dataToSAX(contentHandler, view, DataContext.newInstance().withEmptyValues(false).withLocale(LocaleUtils.toLocale(coursePart.getLanguage())));
        XMLUtils.endElement(contentHandler, "coursePart");
    }
}
