/*
 *  Copyright 2019 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.odf.xslt;

import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

import org.apache.commons.lang3.StringUtils;
import org.w3c.dom.Node;

import org.ametys.core.util.dom.AbstractWrappingAmetysElement;
import org.ametys.core.util.dom.AmetysAttribute;
import org.ametys.odf.enumeration.OdfReferenceTableEntry;

/**
 * DOM layer for {@link OdfReferenceTableEntry}.
 */
public class OdfReferenceTableEntryElement extends AbstractWrappingAmetysElement<OdfReferenceTableEntry>
{
    private String _lang;
    
    /**
     * Constructor.
     * @param entry the underlying {@link OdfReferenceTableEntry}.
     * @param lang the language
     */
    public OdfReferenceTableEntryElement(OdfReferenceTableEntry entry, String lang)
    {
        super(entry);
        _lang = lang;
    }
    
    /**
     * Constructor.
     * @param entry the underlying {@link OdfReferenceTableEntry}.
     * @param lang the language
     * @param parent the parent {@link OdfReferenceTableElement}.
     */
    public OdfReferenceTableEntryElement(OdfReferenceTableEntry entry, String lang, OdfReferenceTableElement parent)
    {
        super(entry, parent);
        _lang = lang;
    }
    
    @Override
    public String getTagName()
    {
        return "item";
    }
    
    @Override
    protected Map<String, AmetysAttribute> _lookupAttributes()
    {
        Map<String, AmetysAttribute> result = new HashMap<>();
        
        result.put("title", new AmetysAttribute("title", "title", null, _object.getLabel(_lang), this));
        result.put("code", new AmetysAttribute("code", "code", null, _object.getCode(), this));
        result.put("id", new AmetysAttribute("id", "id", null, _object.getId(), this));
        result.put("order", new AmetysAttribute("order", "order", null, _object.getOrder().toString(), this));
        result.put("cdmValue", new AmetysAttribute("cdmValue", "cdmValue", null, _object.getCdmValue(), this));
        result.put("archived", new AmetysAttribute("archived", "archived", null, _object.isArchived().toString(), this));
        if (_object.getContent().hasDefinition("shortLabel"))
        {
            result.put("shortLabel", new AmetysAttribute("shortLabel", "shortLabel", null, _object.getContent().getValue("shortLabel", false, StringUtils.EMPTY), this));
        }
        
        return result;
    }
    
    @Override
    public Node getNextSibling()
    {
        if (_parent == null)
        {
            return null;
        }
        
        List<OdfReferenceTableEntry> children = ((OdfReferenceTableElement) _parent).getChildren();
        
        boolean isNext = false;
        OdfReferenceTableEntry nextSibling = null;
        Iterator<OdfReferenceTableEntry> it = children.iterator();
        
        while (nextSibling == null && it.hasNext())
        {
            OdfReferenceTableEntry child = it.next();
            
            if (isNext)
            {
                nextSibling = child;
            }
            else if (_object.getId().equals(child.getId()))
            {
                isNext = true;
            }
        }
        
        return nextSibling == null ? null : new OdfReferenceTableEntryElement(nextSibling, _lang, (OdfReferenceTableElement) _parent);
    }
}
