/*
 *  Copyright 2017 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.odf;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.stream.Collectors;

import org.apache.avalon.framework.component.Component;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;

import org.ametys.cms.repository.Content;
import org.ametys.cms.repository.ModifiableContent;
import org.ametys.cms.repository.WorkflowAwareContent;
import org.ametys.cms.workflow.AbstractContentWorkflowComponent;
import org.ametys.cms.workflow.ContentWorkflowHelper;
import org.ametys.core.util.JSONUtils;
import org.ametys.odf.course.Course;
import org.ametys.odf.course.CourseFactory;
import org.ametys.odf.courselist.CourseList;
import org.ametys.odf.courselist.CourseList.ChoiceType;
import org.ametys.odf.courselist.CourseListFactory;
import org.ametys.odf.coursepart.CoursePart;
import org.ametys.odf.coursepart.CoursePartFactory;
import org.ametys.odf.enumeration.OdfReferenceTableEntry;
import org.ametys.odf.enumeration.OdfReferenceTableHelper;
import org.ametys.odf.orgunit.OrgUnit;
import org.ametys.odf.orgunit.OrgUnitFactory;
import org.ametys.odf.person.Person;
import org.ametys.odf.person.PersonFactory;
import org.ametys.odf.program.AbstractProgram;
import org.ametys.odf.program.Container;
import org.ametys.odf.program.ContainerFactory;
import org.ametys.odf.program.Program;
import org.ametys.odf.program.ProgramFactory;
import org.ametys.odf.program.SubProgram;
import org.ametys.odf.program.SubProgramFactory;
import org.ametys.odf.program.TraversableProgramPart;
import org.ametys.odf.workflow.AbstractCreateODFContentFunction;
import org.ametys.odf.workflow.CreateCoursePartFunction;
import org.ametys.plugins.repository.AmetysRepositoryException;
import org.ametys.plugins.workflow.AbstractWorkflowComponent;

import com.opensymphony.workflow.WorkflowException;

/**
 * Helper to create and edit ODF contents during tests.
 */
public class ODFContentHelper implements Serviceable, Component
{
    /** Avalon Role */
    public static final String ROLE = ODFContentHelper.class.getName();
    
    /** Default org unit code UAI */
    public static final String DEFAULT_ORG_UNIT_CODE_UAI = "UAI-DEF-1234";
    /** Default program domain */
    public static final String DEFAULT_PROGRAM_DOMAIN = "ALL";
    /** Default program degree */
    public static final String DEFAULT_PROGRAM_DEGREE = "XA";
    /** Default container period */
    public static final String DEFAULT_CONTAINER_PERIOD = "an";
    /** Default container nature */
    public static final String DEFAULT_CONTAINER_NATURE = "annee";
    /** Default course list choice type */
    public static final ChoiceType DEFAULT_COURSE_LIST_CHOICE_TYPE = ChoiceType.MANDATORY;
    /** Default course nature */
    public static final String DEFAULT_COURSE_NATURE = "UE";
    /** Default course part nature */
    public static final String DEFAULT_COURSE_PART_NATURE = "TD";
    /** Default course part number of hours */
    public static final Double DEFAULT_COURSE_PART_NB_HOURS = 1.0D;
    
    /** Name of the data to set the type of the program part child */
    public static final String PROGRAM_PART_TYPE = "programPartType";
    
    /**
     * Type of the program part child
     */
    public enum ProgramPartType
    {
        /** Type for {@link SubProgram} children */
        SUB_PROGRAM,
        /** Type for {@link Container} children */
        CONTAINER,
        /** Type for {@link CourseList} children */
        COURSE_LIST
    }

    /** Content workflow helper */
    protected ContentWorkflowHelper _contentWorkflowHelper;
    /** ODF Reference table helper */
    protected OdfReferenceTableHelper _odfReferenceTableHelper;
    /** JSON utils */
    protected JSONUtils _jsonUtils;
    
    private Map<String, String> _programDomains = new HashMap<>();
    private Map<String, String> _programDegrees = new HashMap<>();
    private Map<String, String> _containerPeriods = new HashMap<>();
    private Map<String, String> _containerNatures = new HashMap<>();
    private Map<String, String> _courseNatures = new HashMap<>();
    private Map<String, String> _coursePartNatures = new HashMap<>();
    
    public void service(ServiceManager manager) throws ServiceException
    {
        _contentWorkflowHelper = (ContentWorkflowHelper) manager.lookup(ContentWorkflowHelper.ROLE);
        _odfReferenceTableHelper = (OdfReferenceTableHelper) manager.lookup(OdfReferenceTableHelper.ROLE);
        _jsonUtils = (JSONUtils) manager.lookup(JSONUtils.ROLE);
    }
    
    /**
     * Utility class to create an ODF content
     * @param <T> Type of the ODF content that will be built by this builder
     * @param <B> Type of the implentation itself
     */
    public abstract class AbstractODFContentBuidler<T, B extends AbstractODFContentBuidler<T, B>>
    {
        /** The content's title */
        protected String _title;
        /** The calog's name */
        protected String _catalogName;
        /** The default calog's name */
        protected String _defaultCatalogName;
        /** The content's data */
        protected Map<String, Object> _data = new HashMap<>();
        
        /**
         * Add data to the builder from a JSON string
         * @param data the data to add to the builder
         * @return the builder
         */
        @SuppressWarnings("unchecked")
        public B fromJSON(String data)
        {
            _data = _jsonUtils.convertJsonToMap(data);
            return (B) this;
        }
        
        /**
         * Add data to the builder from a {@link Map}
         * @param data the data to add to the builder
         * To set composite values, put the name of the composite as key and a {@link Map} with named data as value.
         *      To set values into repeater entries, put the name of repeater as key and a {@link List} of {@link Map} for each entry
         *      Example: values = Map.of("singleStringData", "String value",
         *                               "composite", Map.of("multipleLongDataInComposite", List.of(15L, 18L),
         *                                                   "repeaterInComposite", List.of(
         *                                                              Map.of("singleStringDataInRepeater", "String value in first entry",
         *                                                                      "singleLongDataInRepeater", 1L),
         *                                                              Map.of("singleStringDataInRepeater", "String value in second entry",
         *                                                                      "singleLongDataInRepeater", 2L))));
         * @return the builder
         */
        @SuppressWarnings("unchecked")
        public B fromJavaMap(Map<String, Object> data)
        {
            _data = data;
            return (B) this;
        }
        
        /**
         * Add the default catalog name to the content
         * @param defaultCatalogName the default catalog's name
         * @return the content builder
         */
        @SuppressWarnings("unchecked")
        public B withDefaultCatalog(String defaultCatalogName)
        {
            _defaultCatalogName = defaultCatalogName;
            return (B) this;
        }
        
        /**
         * Retrieves the values to set to the content
         * For each content type, the given data need to be converted.
         * For example, an reference table entry will be given via its code, but to create the content, we need the content id of the entry
         * @param data the data
         * @return the content values
         * @throws AmetysRepositoryException if an error occurs
         * @throws WorkflowException if an error occurs
         */
        protected Map<String, Object> _getContentValues(Map<String, Object> data) throws AmetysRepositoryException, WorkflowException
        {
            return data;
        }
        
        /**
         * Retrieves the default values to use if no such data appears in the actual content values
         * @return the default values
         */
        protected Map<String, Object> _getDefaultValues()
        {
            return Map.of();
        }
        
        /**
         * Retrieves the default title of content
         * @return the default title
         */
        protected abstract String _getDefaultTitle();
        
        /**
         * Retrieves the content type identifier of the content to create
         * @return the content type identifier
         */
        protected abstract String _getContentType();
        
        /**
         * Retrieves the name of the workflow to use for the content creation
         * @return the name of the workflow
         */
        protected abstract String _getWorkflowName();
        
        /**
         * Prepare the workflow inputs to create the content
         * @return the prepared workflow inputs
         */
        protected Map<String, Object> _getWorkflowInputs()
        {
            Map<String, Object> inputs = new HashMap<>();
            if (_catalogName != null)
            {
                inputs.put(AbstractCreateODFContentFunction.CONTENT_CATALOG_KEY, _catalogName);
            }
            return inputs;
        }
        
        /**
         * Build the ODF content
         * @return the built content
         * @throws AmetysRepositoryException if an error occurs
         * @throws WorkflowException if an error occurs
         */
        public T build() throws AmetysRepositoryException, WorkflowException
        {
            _title = _data.containsKey(Content.ATTRIBUTE_TITLE) ? (String) _data.get(Content.ATTRIBUTE_TITLE) : _getDefaultTitle();
            _catalogName = _data.containsKey(ProgramItem.CATALOG) ? (String) _data.get(ProgramItem.CATALOG) : _defaultCatalogName;
            
            Map<String, Object> filteredData = _data.keySet()
                    .stream()
                    .filter(key -> !Content.ATTRIBUTE_TITLE.equals(key))
                    .filter(key -> !ProgramItem.CATALOG.equals(key))
                    .collect(Collectors.toMap(key -> key, key -> _data.get(key)));
            
            Map<String, Object> contentValues = _getContentValues(filteredData);
            Map<String, Object> defaultValues = _getDefaultValues();

            for (Map.Entry<String, Object> entry : defaultValues.entrySet())
            {
                if (!contentValues.containsKey(entry.getKey()))
                {
                    contentValues.put(entry.getKey(), entry.getValue());
                }
            }
            
            return createContent(_title, _getContentType(), _getWorkflowName(), contentValues, _getWorkflowInputs());
        }
    }
    
    /**
     * Utility class to create an {@link OrgUnit}
     */
    public class OrgUnitBuilder extends AbstractODFContentBuidler<OrgUnit, OrgUnitBuilder>
    {
        @Override
        protected Map<String, Object> _getContentValues(Map<String, Object> data) throws AmetysRepositoryException, WorkflowException
        {
            Map<String, Object> contentValues = new HashMap<>();
            
            for (Map.Entry<String, Object> entry : data.entrySet())
            {
                switch (entry.getKey())
                {
                    case OrgUnit.CHILD_ORGUNITS:
                        List<String> childrenIds = _getChildrenIds(data, _catalogName);
                        contentValues.put(OrgUnit.CHILD_ORGUNITS, childrenIds);
                        break;
                    default:
                        contentValues.put(entry.getKey(), entry.getValue());
                        break;
                }
            }
            
            return contentValues;
        }
        
        private List<String> _getChildrenIds(Map<String, Object> data, String defaultCatalog) throws AmetysRepositoryException, WorkflowException
        {
            List<String> childrenIds = new ArrayList<>();
            @SuppressWarnings("unchecked")
            List<Object> dataChildren = (List<Object>) data.get(OrgUnit.CHILD_ORGUNITS);
            
            for (Object child : dataChildren)
            {
                if (child instanceof String)
                {
                    childrenIds.add((String) child);
                }
                else if (child instanceof OrgUnit)
                {
                    childrenIds.add(((OrgUnit) child).getId());
                }
                else
                {
                    @SuppressWarnings("unchecked")
                    Map<String, Object> dataChild = (Map<String, Object>) child;
                    
                    OrgUnit orgUnit = new OrgUnitBuilder()
                            .withDefaultCatalog(defaultCatalog)
                            .fromJavaMap(dataChild)
                            .build();
                    
                    childrenIds.add(orgUnit.getId());
                }
            }
            
            return childrenIds;
        }
        
        @Override
        protected Map<String, Object> _getDefaultValues()
        {
            return Map.of(OrgUnit.CODE_UAI, DEFAULT_ORG_UNIT_CODE_UAI);
        }
        
        @Override
        protected String _getContentType()
        {
            return OrgUnitFactory.ORGUNIT_CONTENT_TYPE;
        }
        
        @Override
        protected String _getDefaultTitle()
        {
            return "OrgUnit";
        }
        
        @Override
        protected String _getWorkflowName()
        {
            return "orgunit";
        }
    }
    
    /**
     * Utility class to create a {@link Program}
     */
    public class ProgramBuilder extends AbstractODFContentBuidler<Program, ProgramBuilder>
    {
        /** The content's default org units */
        protected List<String> _defaultOrgUnitIds = new ArrayList<>();
        
        @SuppressWarnings("unchecked")
        @Override
        protected Map<String, Object> _getContentValues(Map<String, Object> data) throws AmetysRepositoryException, WorkflowException
        {
            Map<String, Object> contentValues = new HashMap<>();
            
            List<String> orgUnitIds = data.containsKey(ProgramItem.ORG_UNITS_REFERENCES) ? _getOrgUnitIds(data, ProgramItem.ORG_UNITS_REFERENCES) : _defaultOrgUnitIds;
            contentValues.put(ProgramItem.ORG_UNITS_REFERENCES, orgUnitIds);
            
            for (Map.Entry<String, Object> entry : data.entrySet())
            {
                switch (entry.getKey())
                {
                    case ProgramItem.ORG_UNITS_REFERENCES:
                        // Those data have already been processed => Ignore
                        break;
                    case AbstractProgram.DEGREE:
                        contentValues.put(AbstractProgram.DEGREE, _getDegreeContentId((String) entry.getValue()));
                        break;
                    case AbstractProgram.DOMAIN:
                        contentValues.put(AbstractProgram.DOMAIN, _getDomainContentIds((List<String>) entry.getValue()));
                        break;
                    case TraversableProgramPart.CHILD_PROGRAM_PARTS:
                        List<String> childrenIds = _getAbstractProgramPartChildren(data, _title, ProgramPartType.SUB_PROGRAM, _catalogName, orgUnitIds);
                        contentValues.put(TraversableProgramPart.CHILD_PROGRAM_PARTS, childrenIds);
                        break;
                    default:
                        contentValues.put(entry.getKey(), entry.getValue());
                        break;
                }
            }
            
            return contentValues;
        }
        
        @Override
        protected Map<String, Object> _getDefaultValues()
        {
            return Map.of(AbstractProgram.DEGREE, _getDegreeContentId(DEFAULT_PROGRAM_DEGREE),
                    AbstractProgram.DOMAIN, _getDomainContentIds(List.of(DEFAULT_PROGRAM_DOMAIN)));
        }
        
        private String _getDegreeContentId(String degree)
        {
            if (!_programDegrees.containsKey(degree))
            {
                _programDegrees.put(degree, _odfReferenceTableHelper.getItemFromCode(OdfReferenceTableHelper.DEGREE, degree).getId());
            }
            return _programDegrees.get(degree);
        }
        
        private List<String> _getDomainContentIds(List<String> domains)
        {
            List<String> domainIds = new ArrayList<>();
            
            for (String currentDomain : domains)
            {
                if (!_programDomains.containsKey(currentDomain))
                {
                    _programDomains.put(currentDomain, _odfReferenceTableHelper.getItemFromCode(OdfReferenceTableHelper.DOMAIN, currentDomain).getId());
                }
                domainIds.add(_programDomains.get(currentDomain));
            }
            
            return domainIds;
        }
        
        /**
         * Add default org units to the program
         * @param defaultOrgUnits the default org units
         * @return the program builder
         */
        public ProgramBuilder withDefaultOrgUnits(OrgUnit... defaultOrgUnits)
        {
            _defaultOrgUnitIds = Arrays.stream(defaultOrgUnits)
                    .map(OrgUnit::getId)
                    .collect(Collectors.toList());
            return this;
        }
        
        @Override
        protected String _getContentType()
        {
            return ProgramFactory.PROGRAM_CONTENT_TYPE;
        }
        
        @Override
        protected String _getDefaultTitle()
        {
            return "Program";
        }
        
        @Override
        protected String _getWorkflowName()
        {
            return "program";
        }
    }
    
    /**
     * Utility class to create a {@link SubProgram}
     */
    public class SubProgramBuilder extends AbstractODFContentBuidler<SubProgram, SubProgramBuilder>
    {
        /** The content's default org units */
        protected List<String> _defaultOrgUnitIds = new ArrayList<>();
        
        @Override
        protected Map<String, Object> _getContentValues(Map<String, Object> data) throws AmetysRepositoryException, WorkflowException
        {
            Map<String, Object> contentValues = new HashMap<>();
            
            List<String> orgUnitIds = data.containsKey(ProgramItem.ORG_UNITS_REFERENCES) ? _getOrgUnitIds(data, ProgramItem.ORG_UNITS_REFERENCES) : _defaultOrgUnitIds;
            contentValues.put(ProgramItem.ORG_UNITS_REFERENCES, orgUnitIds);
            
            for (Map.Entry<String, Object> entry : data.entrySet())
            {
                switch (entry.getKey())
                {
                    case ProgramItem.ORG_UNITS_REFERENCES:
                        // Those data have already been processed => Ignore
                        break;
                    case PROGRAM_PART_TYPE:
                        // Do nothing, this is a technical data to choose the builder while building the parent's children
                        break;
                    case TraversableProgramPart.CHILD_PROGRAM_PARTS:
                        List<String> childrenIds = _getAbstractProgramPartChildren(data, _title, ProgramPartType.CONTAINER, _catalogName, orgUnitIds);
                        contentValues.put(TraversableProgramPart.CHILD_PROGRAM_PARTS, childrenIds);
                        break;
                    default:
                        contentValues.put(entry.getKey(), entry.getValue());
                        break;
                }
            }
            
            return contentValues;
        }

        /**
         * Add default org units to the sub program
         * @param defaultOrgUnits the default org units
         * @return the sub program builder
         */
        public SubProgramBuilder withDefaultOrgUnits(OrgUnit... defaultOrgUnits)
        {
            return withDefaultOrgUnits(Arrays.stream(defaultOrgUnits)
                    .map(OrgUnit::getId)
                    .collect(Collectors.toList()));
        }
        
        /**
         * Add default org units to the sub program
         * @param defaultOrgUnitIds the default org units identifiers
         * @return the sub program builder
         */
        public SubProgramBuilder withDefaultOrgUnits(List<String> defaultOrgUnitIds)
        {
            _defaultOrgUnitIds = defaultOrgUnitIds;
            return this;
        }
        
        @Override
        protected String _getContentType()
        {
            return SubProgramFactory.SUBPROGRAM_CONTENT_TYPE;
        }
        
        @Override
        protected String _getDefaultTitle()
        {
            return "SubProgram";
        }
        
        @Override
        protected String _getWorkflowName()
        {
            return "subprogram";
        }
    }
    
    /**
     * Utility class to create a {@link Container}
     */
    public class ContainerBuilder extends AbstractODFContentBuidler<Container, ContainerBuilder>
    {
        /** The content's default org units */
        protected List<String> _defaultOrgUnitIds = new ArrayList<>();
        
        @Override
        protected Map<String, Object> _getContentValues(Map<String, Object> data) throws AmetysRepositoryException, WorkflowException
        {
            Map<String, Object> contentValues = new HashMap<>();
            
            // Manage org units
            List<String> defaultOrgUnitIds = new ArrayList<>();
            if (data.containsKey(ProgramItem.ORG_UNITS_REFERENCES))
            {
                Object orgUnit = data.get(ProgramItem.ORG_UNITS_REFERENCES);
                String orgUnitId = (orgUnit instanceof OrgUnit) ? ((OrgUnit) orgUnit).getId() : (String) orgUnit;
                contentValues.put(ProgramItem.ORG_UNITS_REFERENCES, List.of(orgUnitId));
                
                // If the given data contains an org unit, it is the new default value
                defaultOrgUnitIds.add(orgUnitId);
            }
            else if (!_defaultOrgUnitIds.isEmpty())
            {
                // If the given data contains no org unit, use the first default value
                contentValues.put(ProgramItem.ORG_UNITS_REFERENCES, _defaultOrgUnitIds);
                defaultOrgUnitIds.addAll(defaultOrgUnitIds);
            }
            
            for (Map.Entry<String, Object> entry : data.entrySet())
            {
                switch (entry.getKey())
                {
                    case ProgramItem.ORG_UNITS_REFERENCES:
                        // Those data have already been processed => Ignore
                        break;
                    case PROGRAM_PART_TYPE:
                        // Do nothing, this is a technical data to choose the builder while building the parent's children
                        break;
                    case Container.PERIOD:
                        contentValues.put(Container.PERIOD, _getPeriodContentId((String) entry.getValue()));
                        break;
                    case Container.NATURE:
                        contentValues.put(Container.NATURE, _getNatureContentId((String) entry.getValue()));
                        break;
                    case TraversableProgramPart.CHILD_PROGRAM_PARTS:
                        List<String> childrenIds = _getAbstractProgramPartChildren(data, _title, ProgramPartType.COURSE_LIST, _catalogName, defaultOrgUnitIds);
                        contentValues.put(TraversableProgramPart.CHILD_PROGRAM_PARTS, childrenIds);
                        break;
                    default:
                        contentValues.put(entry.getKey(), entry.getValue());
                        break;
                }
            }
            
            return contentValues;
        }
        
        @Override
        protected Map<String, Object> _getDefaultValues()
        {
            return Map.of(Container.PERIOD, _getPeriodContentId(DEFAULT_CONTAINER_PERIOD),
                    Container.NATURE, _getNatureContentId(DEFAULT_CONTAINER_NATURE));
        }
        
        private String _getPeriodContentId(String period)
        {
            if (!_containerPeriods.containsKey(period))
            {
                _containerPeriods.put(period, _odfReferenceTableHelper.getItemFromCode(OdfReferenceTableHelper.PERIOD, period).getId());
            }
            return _containerPeriods.get(period);
        }
        
        private String _getNatureContentId(String nature)
        {
            if (!_containerNatures.containsKey(nature))
            {
                _containerNatures.put(nature, _odfReferenceTableHelper.getItemFromCode(OdfReferenceTableHelper.CONTAINER_NATURE, nature).getId());
            }
            return _containerNatures.get(nature);
        }

        /**
         * Add default org units to the container
         * @param defaultOrgUnits the default org units
         * @return the container builder
         */
        public ContainerBuilder withDefaultOrgUnits(OrgUnit... defaultOrgUnits)
        {
            return withDefaultOrgUnits(Arrays.stream(defaultOrgUnits)
                    .map(OrgUnit::getId)
                    .collect(Collectors.toList()));
        }
        
        /**
         * Add default org units to the container
         * @param defaultOrgUnitIds the default org units identifiers
         * @return the container builder
         */
        public ContainerBuilder withDefaultOrgUnits(List<String> defaultOrgUnitIds)
        {
            _defaultOrgUnitIds = defaultOrgUnitIds;
            return this;
        }
        
        @Override
        protected String _getContentType()
        {
            return ContainerFactory.CONTAINER_CONTENT_TYPE;
        }
        
        @Override
        protected String _getDefaultTitle()
        {
            return "Container";
        }
        
        @Override
        protected String _getWorkflowName()
        {
            return "container";
        }
    }
    
    /**
     * Utility class to create a {@link CourseList}
     */
    public class CourseListBuilder extends AbstractODFContentBuidler<CourseList, CourseListBuilder>
    {
        /** The content's default org units */
        protected List<String> _defaultOrgUnitIds = new ArrayList<>();
        
        @Override
        protected Map<String, Object> _getContentValues(Map<String, Object> data) throws AmetysRepositoryException, WorkflowException
        {
            Map<String, Object> contentValues = new HashMap<>();
            
            for (Map.Entry<String, Object> entry : data.entrySet())
            {
                switch (entry.getKey())
                {
                    case PROGRAM_PART_TYPE:
                        // Do nothing, this is a technical data to choose the builder while building the parent's children
                        break;
                    case CourseList.CHOICE_TYPE:
                        String choiceType = data.get(CourseList.CHOICE_TYPE).toString();
                        contentValues.put(CourseList.CHOICE_TYPE, choiceType);
                        break;
                    case CourseList.CHILD_COURSES:
                        List<String> childrenIds = _getChildrenIds(data, _catalogName, _defaultOrgUnitIds);
                        contentValues.put(CourseList.CHILD_COURSES, childrenIds);
                        break;
                    default:
                        contentValues.put(entry.getKey(), entry.getValue());
                        break;
                }
            }
            
            return contentValues;
        }
        
        @Override
        protected Map<String, Object> _getDefaultValues()
        {
            return Map.of(CourseList.CHOICE_TYPE, DEFAULT_COURSE_LIST_CHOICE_TYPE.toString());
        }
        
        private List<String> _getChildrenIds(Map<String, Object> data, String defaultCatalog, List<String> defaultOrgUnitIds) throws AmetysRepositoryException, WorkflowException
        {
            List<String> childrenIds = new ArrayList<>();
            @SuppressWarnings("unchecked")
            List<Object> dataChildren = (List<Object>) data.get(CourseList.CHILD_COURSES);
            
            for (Object child : dataChildren)
            {
                if (child instanceof String)
                {
                    childrenIds.add((String) child);
                }
                else if (child instanceof Course)
                {
                    childrenIds.add(((Course) child).getId());
                }
                else
                {
                    @SuppressWarnings("unchecked")
                    Map<String, Object> dataChild = (Map<String, Object>) child;
                    
                    Course course = new CourseBuilder()
                            .withDefaultCatalog(defaultCatalog)
                            .withDefaultOrgUnits(defaultOrgUnitIds)
                            .fromJavaMap(dataChild)
                            .build();
                    
                    childrenIds.add(course.getId());
                }
            }
            
            return childrenIds;
        }
        
        /**
         * Add default org units to the course list
         * @param defaultOrgUnits the default org units
         * @return the course list builder
         */
        public CourseListBuilder withDefaultOrgUnits(OrgUnit... defaultOrgUnits)
        {
            return withDefaultOrgUnits(Arrays.stream(defaultOrgUnits)
                    .map(OrgUnit::getId)
                    .collect(Collectors.toList()));
        }
        
        /**
         * Add default org units to the course list
         * @param defaultOrgUnitIds the default org units identifiers
         * @return the course list builder
         */
        public CourseListBuilder withDefaultOrgUnits(List<String> defaultOrgUnitIds)
        {
            _defaultOrgUnitIds = defaultOrgUnitIds;
            return this;
        }
        
        @Override
        protected String _getContentType()
        {
            return CourseListFactory.COURSE_LIST_CONTENT_TYPE;
        }
        
        @Override
        protected String _getDefaultTitle()
        {
            return "CourseList";
        }
        
        @Override
        protected String _getWorkflowName()
        {
            return "courselist";
        }
    }
    
    /**
     * Utility class to create a {@link Course}
     */
    public class CourseBuilder extends AbstractODFContentBuidler<Course, CourseBuilder>
    {
        /** The content's default org units */
        protected List<String> _defaultOrgUnitIds = new ArrayList<>();
        
        @Override
        protected Map<String, Object> _getContentValues(Map<String, Object> data) throws AmetysRepositoryException, WorkflowException
        {
            Map<String, Object> contentValues = new HashMap<>();
            
            List<String> orgUnitIds = data.containsKey(ProgramItem.ORG_UNITS_REFERENCES) ? _getOrgUnitIds(data, ProgramItem.ORG_UNITS_REFERENCES) : _defaultOrgUnitIds;
            contentValues.put(ProgramItem.ORG_UNITS_REFERENCES, orgUnitIds);
            
            for (Map.Entry<String, Object> entry : data.entrySet())
            {
                switch (entry.getKey())
                {
                    case ProgramItem.ORG_UNITS_REFERENCES:
                        // Those data have already been processed => Ignore
                        break;
                    case Course.COURSE_TYPE:
                        contentValues.put(Course.COURSE_TYPE, _getNatureContentId((String) entry.getValue()));
                        break;
                    case Course.CHILD_COURSE_LISTS:
                        List<String> courseListIds = _getCourseLists(data, _catalogName, orgUnitIds);
                        contentValues.put(Course.CHILD_COURSE_LISTS, courseListIds);
                        break;
                    case Course.CHILD_COURSE_PARTS:
                        List<String> coursePartIds = _getCourseParts(data, _catalogName);
                        contentValues.put(Course.CHILD_COURSE_PARTS, coursePartIds);
                        break;
                    default:
                        contentValues.put(entry.getKey(), entry.getValue());
                        break;
                }
            }
            
            return contentValues;
        }
        
        @Override
        protected Map<String, Object> _getDefaultValues()
        {
            return Map.of(Course.COURSE_TYPE, _getNatureContentId(DEFAULT_COURSE_NATURE));
        }
        
        private String _getNatureContentId(String nature)
        {
            if (!_courseNatures.containsKey(nature))
            {
                _courseNatures.put(nature, _odfReferenceTableHelper.getItemFromCode(OdfReferenceTableHelper.COURSE_NATURE, nature).getId());
            }
            return _courseNatures.get(nature);
        }
        
        private List<String> _getCourseLists(Map<String, Object> data, String defaultCatalog, List<String> defaultOrgUnitIds) throws AmetysRepositoryException, WorkflowException
        {
            List<String> courseListIds = new ArrayList<>();
            @SuppressWarnings("unchecked")
            List<Object> dataChildren = (List<Object>) data.get(Course.CHILD_COURSE_LISTS);
            
            for (Object child : dataChildren)
            {
                if (child instanceof String)
                {
                    courseListIds.add((String) child);
                }
                else if (child instanceof CourseList)
                {
                    courseListIds.add(((CourseList) child).getId());
                }
                else
                {
                    @SuppressWarnings("unchecked")
                    Map<String, Object> dataChild = (Map<String, Object>) child;
                    
                    CourseList courseList = new CourseListBuilder()
                            .withDefaultCatalog(defaultCatalog)
                            .withDefaultOrgUnits(defaultOrgUnitIds)
                            .fromJavaMap(dataChild)
                            .build();
                    
                    courseListIds.add(courseList.getId());
                }
            }
            
            return courseListIds;
        }
        
        private List<String> _getCourseParts(Map<String, Object> data, String defaultCatalog) throws AmetysRepositoryException, WorkflowException
        {
            List<String> coursePartIds = new ArrayList<>();
            @SuppressWarnings("unchecked")
            List<Object> dataChildren = (List<Object>) data.get(Course.CHILD_COURSE_PARTS);
            
            for (Object child : dataChildren)
            {
                if (child instanceof String)
                {
                    coursePartIds.add((String) child);
                }
                else if (child instanceof CoursePart)
                {
                    coursePartIds.add(((CoursePart) child).getId());
                }
                else
                {
                    @SuppressWarnings("unchecked")
                    Map<String, Object> dataChild = (Map<String, Object>) child;
                    
                    CoursePart coursePart = new CoursePartBuilder()
                            .withDefaultCatalog(defaultCatalog)
                            .fromJavaMap(dataChild)
                            .build();
                    
                    coursePartIds.add(coursePart.getId());
                }
            }
            
            return coursePartIds;
        }
        
        /**
         * Add default org units to the course
         * @param defaultOrgUnits the default org units
         * @return the course builder
         */
        public CourseBuilder withDefaultOrgUnits(OrgUnit... defaultOrgUnits)
        {
            return withDefaultOrgUnits(Arrays.stream(defaultOrgUnits)
                    .map(OrgUnit::getId)
                    .collect(Collectors.toList()));
        }
        
        /**
         * Add default org units to the course
         * @param defaultOrgUnitIds the default org units identifiers
         * @return the course builder
         */
        public CourseBuilder withDefaultOrgUnits(List<String> defaultOrgUnitIds)
        {
            _defaultOrgUnitIds = defaultOrgUnitIds;
            return this;
        }
        
        @Override
        protected String _getContentType()
        {
            return CourseFactory.COURSE_CONTENT_TYPE;
        }
        
        @Override
        protected String _getDefaultTitle()
        {
            return "Course";
        }
        
        @Override
        protected String _getWorkflowName()
        {
            return "course";
        }
    }
    
    /**
     * Utility class to create a {@link CoursePart}
     */
    public class CoursePartBuilder extends AbstractODFContentBuidler<CoursePart, CoursePartBuilder>
    {
        /** The course part's nature */
        protected String _nature;
        /** The number of hours in this course part's */
        protected Double _nbHours;
        
        @Override
        protected Map<String, Object> _getDefaultValues()
        {
            return Map.of(CoursePart.NATURE, _getNatureContentId(DEFAULT_COURSE_PART_NATURE),
                    CoursePart.NB_HOURS, DEFAULT_COURSE_PART_NB_HOURS);
        }
        
        private String _getNatureContentId(String nature)
        {
            if (!_coursePartNatures.containsKey(nature))
            {
                OdfReferenceTableEntry itemFromCode = _odfReferenceTableHelper.getItemFromCode(OdfReferenceTableHelper.ENSEIGNEMENT_NATURE, nature);
                _coursePartNatures.put(nature, itemFromCode != null ? itemFromCode.getId() : nature);
            }
            return _coursePartNatures.get(nature);
        }
        
        @Override
        protected String _getContentType()
        {
            return CoursePartFactory.COURSE_PART_CONTENT_TYPE;
        }
        
        @Override
        protected String _getDefaultTitle()
        {
            return "CoursePart";
        }
        
        @Override
        protected String _getWorkflowName()
        {
            return "course-part";
        }
        
        @Override
        protected Map<String, Object> _getWorkflowInputs()
        {
            Map<String, Object> inputs = super._getWorkflowInputs();
            Optional.of(CoursePart.COURSE_HOLDER)
                .map(_data::get)
                .map(Course.class::cast)
                .map(Course::getId)
                .ifPresent(id -> inputs.put(CreateCoursePartFunction.COURSE_HOLDER_KEY, id));
            return inputs;
        }
    }
    
    /**
     * Utility class to create a {@link ODFTableRefBuilder}
     */
    public class ODFTableRefBuilder extends AbstractODFContentBuidler<ModifiableContent, ODFTableRefBuilder>
    {
        private String _contentType;
        
        @Override
        protected String _getContentType()
        {
            return _contentType;
        }
        
        @Override
        protected String _getDefaultTitle()
        {
            return "TableRef";
        }
        
        @Override
        protected String _getWorkflowName()
        {
            return "reference-table";
        } 
        
        /**
         * Add the actual content type to the content
         * @param contentType the content type
         * @return the content builder
         */
        public ODFTableRefBuilder withContentType(String contentType)
        {
            this._contentType = contentType;
            return this;
        }
    }

    /**
     * Utility class to create a {@link Person}
     */
    public class PersonBuilder extends AbstractODFContentBuidler<Person, PersonBuilder>
    {
        @Override
        protected String _getContentType()
        {
            return PersonFactory.PERSON_CONTENT_TYPE;
        }
        
        @Override
        protected String _getDefaultTitle()
        {
            return "Person";
        }
        
        @Override
        protected String _getWorkflowName()
        {
            return "person";
        }
    }
    
    private List<String> _getOrgUnitIds(Map<String, Object> data, String key)
    {
        List<String> orgUnitIds = new ArrayList<>();
        
        @SuppressWarnings("unchecked")
        List<Object> orgUnits = (List<Object>) data.get(key);
        for (Object orgUnit : orgUnits)
        {
            String orgUnitId = (orgUnit instanceof OrgUnit) ? ((OrgUnit) orgUnit).getId() : (String) orgUnit;
            orgUnitIds.add(orgUnitId);
        }
        
        return orgUnitIds;
    }
    
    private List<String> _getAbstractProgramPartChildren(Map<String, Object> data, String contentTitle, ProgramPartType defaultProgramPartType, String defaultCatalog, List<String> defaultOrgUnits) throws AmetysRepositoryException, WorkflowException
    {
        List<String> children = new ArrayList<>();
        
        @SuppressWarnings("unchecked")
        List<Object> dataChildren = (List<Object>) data.get(TraversableProgramPart.CHILD_PROGRAM_PARTS);
        for (Object child : dataChildren)
        {
            if (child instanceof String)
            {
                children.add((String) child);
            }
            else if (child instanceof Content)
            {
                children.add(((Content) child).getId());
            }
            else
            {
                @SuppressWarnings("unchecked")
                Map<String, Object> dataChild = (Map<String, Object>) child;
                
                ProgramPartType programPartType = _getProgramPartType(dataChild, contentTitle, defaultProgramPartType);
                if (ProgramPartType.SUB_PROGRAM.equals(programPartType))
                {
                    SubProgram subProgram = new SubProgramBuilder()
                            .withDefaultCatalog(defaultCatalog)
                            .withDefaultOrgUnits(defaultOrgUnits)
                            .fromJavaMap(dataChild)
                            .build();
                    
                    children.add(subProgram.getId());
                }
                else if (ProgramPartType.CONTAINER.equals(programPartType))
                {
                    Container container = new ContainerBuilder()
                            .withDefaultCatalog(defaultCatalog)
                            .withDefaultOrgUnits(defaultOrgUnits)
                            .fromJavaMap(dataChild)
                            .build();
                    
                    children.add(container.getId());
                }
                else if (ProgramPartType.COURSE_LIST.equals(programPartType))
                {
                    CourseList courseList = new CourseListBuilder()
                            .withDefaultCatalog(defaultCatalog)
                            .withDefaultOrgUnits(defaultOrgUnits)
                            .fromJavaMap(dataChild)
                            .build();
                    children.add(courseList.getId());
                }
                else
                {
                    throw new IllegalArgumentException("the '" + PROGRAM_PART_TYPE + "' in the given child data of the content '" + contentTitle + "' is invalid. The given value is '" + programPartType + "' but should be one of the following: '" + ProgramPartType.SUB_PROGRAM + "', '" + ProgramPartType.CONTAINER + "' or '" + ProgramPartType.COURSE_LIST + "'.");
                }
            }
        }
        
        return children;
    }
    
    private ProgramPartType _getProgramPartType(Map<String, Object> data, String contentTitle, ProgramPartType defaultProgramPartType)
    {
        if (data.containsKey(PROGRAM_PART_TYPE))
        {
            if (data.get(PROGRAM_PART_TYPE) instanceof String)
            {
                return ProgramPartType.valueOf((String) data.get(PROGRAM_PART_TYPE));
            }
            else if (data.get(PROGRAM_PART_TYPE) instanceof ProgramPartType)
            {
                return (ProgramPartType) data.get(PROGRAM_PART_TYPE);
            }
            else
            {
                throw new IllegalArgumentException("the '" + PROGRAM_PART_TYPE + "' in the given data of the content '" + contentTitle + "' is invalid.");
            }
        }
        else
        {
            return defaultProgramPartType;
        }
    }
    
    /**
     * Create an ODF content.
     * @param <T> Type of the created ODF content
     * @param title Title of the content
     * @param contentType Content type
     * @param workflowName Workflow name
     * @param values Values to set
     * @param inputs The workflow inputs
     * @return The created content
     * @throws AmetysRepositoryException if an error occurs
     * @throws WorkflowException if an error occurs
     */
    public <T extends WorkflowAwareContent> T createContent(String title, String contentType, String workflowName, Map<String, Object> values, Map<String, Object> inputs) throws AmetysRepositoryException, WorkflowException
    {
        Map<String, Object> results = _contentWorkflowHelper.createContent(workflowName, 1, title, title, new String[] {contentType}, null, "fr", inputs);
        
        @SuppressWarnings("unchecked")
        T content = (T) results.get(Content.class.getName());
        editContent(content, values);
        
        return content;
    }
    
    /**
     * Edit an ODF content.
     * @param content Content to edit
     * @param values Values to set.
     *      To set composite values, put the name of the composite as key and a {@link Map} with named data as value.
     *      To set values into repeater entries, put the name of repeater as key and a {@link List} of {@link Map} for each entry
     *      Example: values = Map.of("singleStringData", "String value",
     *                               "composite", Map.of("multipleLongDataInComposite", List.of(15L, 18L),
     *                                                   "repeaterInComposite", List.of(
     *                                                              Map.of("singleStringDataInRepeater", "String value in first entry",
     *                                                                      "singleLongDataInRepeater", 1L),
     *                                                              Map.of("singleStringDataInRepeater", "String value in second entry",
     *                                                                      "singleLongDataInRepeater", 2L))));
     * @throws AmetysRepositoryException if an error occurs
     * @throws WorkflowException if an error occurs
     */
    public void editContent(WorkflowAwareContent content, Map<String, Object> values) throws AmetysRepositoryException, WorkflowException
    {
        _contentWorkflowHelper.editContent(content, values, 2);
    }
    
    /**
     * Validate the given ODF contents.
     * @param contents Contents to validate
     * @throws AmetysRepositoryException if an error occurs
     * @throws WorkflowException if an error occurs
     */
    public void validateContents(WorkflowAwareContent... contents) throws AmetysRepositoryException, WorkflowException
    {
        for (WorkflowAwareContent content : contents)
        {
            Map<String, Object> inputs = new HashMap<>(); 
            inputs.put(AbstractContentWorkflowComponent.CONTENT_KEY, content); 
            inputs.put(AbstractWorkflowComponent.RESULT_MAP_KEY, Map.of()); 
            
            _contentWorkflowHelper.doAction(content, 4, inputs);
        }
    }
}
