/*
 *  Copyright 2020 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.odf.schedulable;

import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.time.ZonedDateTime;
import java.time.format.DateTimeFormatter;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.quartz.JobDataMap;
import org.quartz.JobExecutionContext;

import org.ametys.cms.data.Binary;
import org.ametys.cms.repository.Content;
import org.ametys.cms.repository.ModifiableContent;
import org.ametys.cms.repository.WorkflowAwareContent;
import org.ametys.core.schedule.Schedulable;
import org.ametys.core.util.DateUtils;
import org.ametys.odf.ProgramItem;
import org.ametys.plugins.core.schedule.Scheduler;
import org.ametys.plugins.repository.data.holder.group.ModifiableModelAwareRepeater;
import org.ametys.plugins.repository.data.holder.group.ModifiableModelAwareRepeaterEntry;
import org.ametys.plugins.workflow.AbstractWorkflowComponent;
import org.ametys.plugins.workflow.component.CheckRightsCondition;
import org.ametys.runtime.i18n.I18nizableText;

/**
 * {@link Schedulable} for archive educational booklet.
 */
public class ArchiveEducationalBookletSchedulable extends EducationalBookletSchedulable
{
    /** Scheduler parameter name of archive date */
    public static final String PARAM_ARCHIVE_DATE = "archiveDate";
    /** Scheduler parameter name of edit workflow action id */
    public static final String PARAM_WORKFLOW_ACTION_ID = "workflowActionId";
    
    @Override
    protected void _generateProgramItemsEducationalBooklet(JobExecutionContext context, File bookletDirectory, Map<String, Object> pdfParameters)
    {
        JobDataMap jobDataMap = context.getJobDetail().getJobDataMap();
        
        ZonedDateTime archiveDate = (ZonedDateTime) jobDataMap.get(Scheduler.PARAM_VALUES_PREFIX + PARAM_ARCHIVE_DATE);
        long workflowActionId = (long) jobDataMap.get(Scheduler.PARAM_VALUES_PREFIX + PARAM_WORKFLOW_ACTION_ID);
        
        pdfParameters.put(PARAM_ARCHIVE_DATE, DateUtils.zonedDateTimeToString(archiveDate));
        pdfParameters.put(PARAM_INCLUDE_SUBPROGRAMS, jobDataMap.get(Scheduler.PARAM_VALUES_PREFIX + PARAM_INCLUDE_SUBPROGRAMS));
        
        super._generateProgramItemsEducationalBooklet(context, bookletDirectory, pdfParameters);
        
        EducationalBookletReport report = (EducationalBookletReport) context.get(_EDUCATIONAL_BOOKLET_REPORT);
        
        List<Content> exportedProgramItems = new ArrayList<>();
        List<Content> programItemsInError = report.getProgramItemsInError();
        for (Content content : report.getExportedProgramItems())
        {
            ModifiableModelAwareRepeater repeater = ((ModifiableContent) content).getRepeater(ProgramItem.EDUCATIONAL_BOOKLETS, true);
            ModifiableModelAwareRepeaterEntry entry = repeater.addEntry();
            
            entry.setValue("date", archiveDate);

            File educationalBookletPDF = new File(report.getBookletDirectory(), content.getName() + "/" + content.getLanguage() + "/educational-booklet.pdf");
            
            try
            {
                Binary binary = _getEducationalBookletAsBinary(educationalBookletPDF, content, archiveDate);
                entry.setValue("pdf", binary);
                
                Map<String, Object> paramsEdit = new HashMap<>();
                paramsEdit.put(CheckRightsCondition.FORCE, true); // Ignore the right condition
                paramsEdit.put(AbstractWorkflowComponent.CONTEXT_PARAMETERS_KEY, new HashMap<>());
                
                _contentWorkflowHelper.doAction((WorkflowAwareContent) content, (int) workflowActionId, paramsEdit);
                
                exportedProgramItems.add(content);
            }
            catch (Exception e)
            {
                programItemsInError.add(content);
                getLogger().error("An error occurred during saving the educational booklet into content '{} ({})'", content.getTitle(), content.getId(), e);
            }
            
        }
        
        report.setExportedProgramItems(exportedProgramItems);
        
        context.put(_EDUCATIONAL_BOOKLET_REPORT, report);
    }
    
    /**
     * Build a binary object from the educational booklet file
     * @param educationalBookletPDF the educational booklet file
     * @param content the content
     * @param votedDate the voted date for the educational booklet 
     * @return the binary
     * @throws IOException if an error occurred
     */
    protected Binary _getEducationalBookletAsBinary(File educationalBookletPDF, Content content, ZonedDateTime votedDate) throws IOException
    {
        Binary binary = new Binary();
        try (InputStream is = new FileInputStream(educationalBookletPDF))
        {
            DateTimeFormatter formatter = DateTimeFormatter.ofPattern("yyyy-MM-dd");
            String dateAsString = votedDate.format(formatter);
            
            String prefix = _i18nUtils.translate(new I18nizableText("plugin.odf", "PLUGINS_ODF_EDUCATIONAL_BOOKLET_PROGRAMITEM_PDF_NAME_PREFIX"), content.getLanguage());
            String fileName = prefix + " " + content.getTitle() + " - " + dateAsString + ".pdf";

            binary.setEncoding("UTF-8");
            binary.setMimeType("application/pdf");
            binary.setLastModificationDate(ZonedDateTime.now());
            binary.setFilename(fileName);
            binary.setInputStream(is);
        }
        return binary;
    }
    
    @Override
    protected String _getMailSubjectBaseKey()
    {
        return "PLUGINS_ODF_EDUCATIONAL_BOOKLET_PROGRAMITEM_MAIL_SUBJECT_ARCHIVE_";
    }

    @Override
    protected String _getMailBodyBaseKey()
    {
        return "PLUGINS_ODF_EDUCATIONAL_BOOKLET_PROGRAMITEM_MAIL_BODY_ARCHIVE_";
    }
}
