/*
 *  Copyright 2025 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.odf.skill.imports.csv;

import java.io.IOException;
import java.util.List;
import java.util.Map;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.commons.lang3.StringUtils;

import org.ametys.core.right.RightManager;
import org.ametys.core.right.RightManager.RightResult;
import org.ametys.core.ui.Callable;
import org.ametys.odf.program.Program;
import org.ametys.plugins.contentio.csv.ImportCSVFileHelper;
import org.ametys.runtime.authentication.AccessDeniedException;

/**
 * Import skills from an uploaded CSV file.
 */
public class ImportSkillsCSVFileHelper extends ImportCSVFileHelper
{
    /** The macro skill catalog attribute path */
    public static final String SKILL_CATALOG_ATTRIBUTE_PATH = "catalog";
    /** The skill parent program attribute path */
    public static final String SKILL_PARENT_PROGRAM_ATTRIBUTE_PATH = "parentProgram";
    /** The micro skill catalog attribute path */
    public static final String MICROSKILL_CATALOG_ATTRIBUTE_PATH = "microSkills.catalog";
    
    private RightManager _rightManager;
    
    @Override
    public void service(ServiceManager serviceManager) throws ServiceException
    {
        super.service(serviceManager);
        _csvImporter = (SkillCSVImporter) serviceManager.lookup(SkillCSVImporter.ROLE);
        _rightManager = (RightManager) serviceManager.lookup(RightManager.ROLE);
    }
    
    /**
     * Import contents from path, content type and language.
     * @implNote Override to add checks on data before import of skills
     */
    @Override
    @Callable (rights = Callable.CHECKED_BY_IMPLEMENTATION)
    public Map<String, Object> importContents(Map<String, Object> config, Map<String, Object> formValues, List<Map<String, Object>> mappingValues) throws IOException
    {
        // Check if we have a program parent to use
        String programId = (String) config.get(CONFIG_PARENT_ID);
        if (StringUtils.isNotEmpty(programId))
        {
            Program program = (Program) _resolver.resolveById(programId);
            if (program != null)
            {
                // Check that the user has the right to import skills under this program
                if (_rightManager.currentUserHasRight("ODF_Rights_Skills_Import", program) == RightResult.RIGHT_ALLOW)
                {
                    // Ignore catalog field, the catalog of the parent program will be used
                    _removeMappingValuesForAttributePath(SKILL_CATALOG_ATTRIBUTE_PATH, mappingValues);
                }
                else
                {
                    throw new AccessDeniedException("User tried to import skills under program '" + programId + "' without sufficient rights.");
                }
            }
            else
            {
                throw new IllegalArgumentException("The import of skills under a program could not be done because no program was found for given id");
            }
        }
        // If we don't have a parent program, it is a transversal import, check that the user has the right to import transversal skills
        else if (_rightManager.currentUserHasRight("ODF_Rights_Skills_Import", "/${WorkspaceName}") == RightResult.RIGHT_ALLOW)
        {
            // Check that the catalog field is present and set as an Id
            _checkMacroCatalogFieldPresenceAndSetAsId(mappingValues);
        }
        else
        {
            throw new AccessDeniedException("User tried to import transversal skills without sufficient rights.");
        }
        
        // Remove unneeded values, these fields will be forced at skill creation
        // Ignore microskills catalog field, the catalog of the parent macro skill will be used
        _removeMappingValuesForAttributePath(MICROSKILL_CATALOG_ATTRIBUTE_PATH, mappingValues);
        
        // Ignore parent program fields, we should not try to set parent program or parent program's attributes
        _checkNoProgramFields(mappingValues);
        
        return super.importContents(config, formValues, mappingValues);
    }
    
    private void _checkMacroCatalogFieldPresenceAndSetAsId(List<Map<String, Object>> mappingValues)
    {
        // Check that we have a catalog in which to import the skills, if not, don't do the import
        Map<String, Object> catalogField = mappingValues.stream()
                .filter(values -> SKILL_CATALOG_ATTRIBUTE_PATH.equals(values.get(MAPPING_COLUMN_ATTRIBUTE_PATH)))
                .findFirst()
                .orElseThrow(() -> new IllegalArgumentException("The CSV file cannot be imported because the catalog column is missing"));
        
        // Check that the catalog is used as an Id, if not, force it to be an Id
        if (!(boolean) catalogField.getOrDefault(MAPPING_COLUMN_IS_ID, false))
        {
            getLogger().warn("The catalog field was not an identifier in a transversal skill import, it should have been one. The catalog field will be forced as an identifier.");
            catalogField.put(MAPPING_COLUMN_IS_ID, true);
        }
    }
    
    private void _checkNoProgramFields(List<Map<String, Object>> mappingValues) throws IllegalArgumentException
    {
        // Check that we don't have program attributes set in the CSV
        if (mappingValues.stream().anyMatch(values -> values.get(MAPPING_COLUMN_ATTRIBUTE_PATH) instanceof String attributePath && attributePath.contains(SKILL_PARENT_PROGRAM_ATTRIBUTE_PATH)))
        {
            throw new IllegalArgumentException("The CSV could not be imported because it is trying to import skills with program attributes. The program parent will be set automatically, it should not be set.");
        }
    }
    
    private void _removeMappingValuesForAttributePath(String attributePathToRemove, List<Map<String, Object>> mappingValues)
    {
        boolean hasRemovedPath = false;
        
        // For each of these values
        for (Map<String, Object> mappingValue : mappingValues)
        {
            String attributePath = (String) mappingValue.get(MAPPING_COLUMN_ATTRIBUTE_PATH);
            // If the mapping value has the attribute path to remove, remove it
            if (attributePath != null && attributePath.replaceAll("/", ".").equals(attributePathToRemove))
            {
                // Remove the mapping in the map
                mappingValue.put(MAPPING_COLUMN_ATTRIBUTE_PATH, null);
                hasRemovedPath = true;
            }
        }
       
        // If at least one entry was removed, log it
        if (hasRemovedPath)
        {
            getLogger().warn("The skill import was attempted with a '{}' column, it will be ignored", attributePathToRemove);
        }
    }
}
