/*
 *  Copyright 2015 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.odf;

import java.io.File;
import java.io.FileFilter;
import java.util.HashMap;
import java.util.Map;

import org.apache.avalon.framework.configuration.Configuration;
import org.apache.avalon.framework.configuration.ConfigurationException;
import org.apache.avalon.framework.configuration.DefaultConfigurationBuilder;
import org.apache.avalon.framework.context.Context;
import org.apache.avalon.framework.context.ContextException;
import org.apache.avalon.framework.context.Contextualizable;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;
import org.apache.cocoon.Constants;
import org.apache.cocoon.util.log.SLF4JLoggerAdapter;
import org.apache.commons.lang3.StringUtils;

import org.ametys.cms.content.indexing.solr.observation.ObserverHelper;
import org.ametys.cms.contenttype.ContentType;
import org.ametys.cms.contenttype.ContentTypeExtensionPoint;
import org.ametys.cms.repository.ContentQueryHelper;
import org.ametys.cms.repository.ContentTypeExpression;
import org.ametys.cms.repository.ModifiableContent;
import org.ametys.cms.repository.WorkflowAwareContent;
import org.ametys.cms.workflow.AbstractContentWorkflowComponent;
import org.ametys.cms.workflow.ContentWorkflowHelper;
import org.ametys.cms.workflow.CreateContentFunction;
import org.ametys.core.engine.BackgroundEngineHelper;
import org.ametys.core.util.I18nUtils;
import org.ametys.odf.enumeration.OdfReferenceTableEntry;
import org.ametys.plugins.repository.AmetysObject;
import org.ametys.plugins.repository.AmetysObjectIterable;
import org.ametys.plugins.repository.AmetysObjectIterator;
import org.ametys.plugins.repository.AmetysObjectResolver;
import org.ametys.plugins.repository.query.expression.AndExpression;
import org.ametys.plugins.repository.query.expression.Expression.Operator;
import org.ametys.plugins.repository.query.expression.StringExpression;
import org.ametys.runtime.config.Config;
import org.ametys.runtime.i18n.I18nizableText;
import org.ametys.runtime.plugin.Init;
import org.ametys.runtime.plugin.component.AbstractLogEnabled;

import com.opensymphony.workflow.WorkflowException;

/**
 * This init class populates reference tables contents base on WEB-INF/params/odf files
 *
 */
public class PopulateOdfTableRef extends AbstractLogEnabled implements Init, Serviceable, Contextualizable
{
    private AmetysObjectResolver _resolver;
    private org.apache.cocoon.environment.Context _cocoonContext;
    private I18nUtils _i18nUtils;
    private ContentTypeExtensionPoint _cTypeExtensionPoint;
    private ServiceManager _manager;
    private ContentWorkflowHelper _workflowHelper;

    @Override
    public void service(ServiceManager manager) throws ServiceException
    {
        _manager = manager;
        _resolver = (AmetysObjectResolver) manager.lookup(AmetysObjectResolver.ROLE);
        _i18nUtils = (I18nUtils) manager.lookup(I18nUtils.ROLE);
        _cTypeExtensionPoint = (ContentTypeExtensionPoint) manager.lookup(ContentTypeExtensionPoint.ROLE);
        _workflowHelper = (ContentWorkflowHelper) manager.lookup(ContentWorkflowHelper.ROLE);
    }
    
    public void contextualize(Context context) throws ContextException
    {
        _cocoonContext = (org.apache.cocoon.environment.Context) context.get(Constants.CONTEXT_ENVIRONMENT_CONTEXT);
    }
    
    public void init() throws Exception
    {
        File file = new File(_cocoonContext.getRealPath("/WEB-INF/param/odf"));
        
        if (file.exists() && file.isDirectory())
        {
            Map<String, Object> environmentInformation = null;
            try
            {
                // Enter in a background environment as i18nUtils can not be used outside a request
                // because of XMLBundleResource uses SourceUtil.toSAX(..) to parse its values.
                environmentInformation = BackgroundEngineHelper.createAndEnterEngineEnvironment(_manager, _cocoonContext, new SLF4JLoggerAdapter(getLogger()));
                
                ObserverHelper.suspendObservationForIndexation();
                
                File[] xmlFiles = file.listFiles(new FileFilter()
                {
                    public boolean accept(File child)
                    {
                        return child.isFile() && child.getName().endsWith(".xml");
                    }
                });
                
                for (File xmlFile : xmlFiles)
                {
                    int count = 0;
                    String fileName = xmlFile.getName();
                    
                    String cTypeId = _transform2ContentTypeId (fileName);
                    
                    // Check that the simple content type exists
                    if (_cTypeExtensionPoint.hasExtension(cTypeId))
                    {
                        Configuration configuration = new DefaultConfigurationBuilder().buildFromFile(xmlFile);
                        for (Configuration itemConfig : configuration.getChildren("item"))
                        {
                            boolean hasSubEntries = itemConfig.getChild("subitem", false) != null;
                            count += _createEntryIfNotExists(cTypeId + (hasSubEntries ? "Category" : ""), itemConfig, null);
                        }
                        
                        getLogger().info("Create {} entries for content type '{}'", count, cTypeId);
                    }
                    else
                    {
                        getLogger().info("Content type '{}' is not a valid content type.", cTypeId);
                    }
                }
            }
            finally
            {
                // Leave the environment.
                if (environmentInformation != null)
                {
                    BackgroundEngineHelper.leaveEngineEnvironment(environmentInformation);
                }
                
                ObserverHelper.restartObservationForIndexation();
            }
        }
    }
    
    private String _transform2ContentTypeId (String filename)
    {
        int index = filename.lastIndexOf(".");
        String id = filename.substring(0, index);
        
        StringBuilder sb = new StringBuilder();
        sb.append("odf-enumeration.");
        
        for (String segment : id.split("_"))
        {
            sb.append(StringUtils.capitalize(segment));
        }
        
        return sb.toString();
    }
    
    private int _createEntryIfNotExists (String cTypeId, Configuration itemConfig, String parentCategoryId) throws WorkflowException, ConfigurationException
    {
        int count = 0;
        
        String code = itemConfig.getAttribute("code");
        String i18nKey = itemConfig.getAttribute("i18n-key");
        String cdmValue = itemConfig.getAttribute("cdmValue", null);
        
        ContentTypeExpression cTypeExpr = new ContentTypeExpression(Operator.EQ, cTypeId);
        StringExpression codeExpr = new StringExpression(OdfReferenceTableEntry.CODE, Operator.EQ, code);
        
        String xpathQuery = ContentQueryHelper.getContentXPathQuery(new AndExpression(cTypeExpr, codeExpr));
        AmetysObjectIterable<AmetysObject> contents = _resolver.query(xpathQuery);
        AmetysObjectIterator<AmetysObject> it = contents.iterator();

        ContentType contentType = _cTypeExtensionPoint.getExtension(cTypeId);
        
        ModifiableContent entry = null;
        if (!it.hasNext())
        {
            String titleFR = _i18nUtils.translate(new I18nizableText("application", i18nKey), "fr");
            
            if (StringUtils.isEmpty(titleFR))
            {
                getLogger().error("The reference table item with the code '{}' of type '{}' can't be migrated because it's I18N key '{}' is not defined for french.", code, cTypeId, i18nKey);
            }
            else
            {
                String titleEN = _i18nUtils.translate(new I18nizableText("application", i18nKey), "en");
                
                Map<String, String> titleVariants = new HashMap<>();
                titleVariants.put("fr", titleFR);
                if (titleEN != null)
                {
                    titleVariants.put("en", titleEN);
                }
                
                Map<String, Object> inputs = new HashMap<>();
                if (!contentType.isMultilingual())
                {
                    inputs.put(CreateContentFunction.CONTENT_LANGUAGE_KEY, Config.getInstance().getValue("odf.programs.lang")); 
                }
                Map<String, Object> result = _workflowHelper.createContent("reference-table", 1, titleFR, titleVariants, new String[] {cTypeId}, new String[0], inputs);
                
                entry = (ModifiableContent) result.get(AbstractContentWorkflowComponent.CONTENT_KEY);
                if (entry.hasDefinition(OdfReferenceTableEntry.CODE))
                {
                    entry.setValue(OdfReferenceTableEntry.CODE, code);
                }
                
                if (StringUtils.isNotEmpty(cdmValue) && entry.hasDefinition(OdfReferenceTableEntry.CDM_VALUE))
                {
                    entry.setValue(OdfReferenceTableEntry.CDM_VALUE, cdmValue);
                }
                
                if (parentCategoryId != null && entry.hasDefinition("category"))
                {
                    entry.setValue("category", parentCategoryId);
                }
                entry.saveChanges();
                _workflowHelper.doAction((WorkflowAwareContent) entry, 22);
                
                count++;
            }
        }
        else
        {
            entry = (ModifiableContent) it.next();
        }
        
        if (entry != null)
        {
            for (Configuration childConfig : itemConfig.getChildren("subitem"))
            {
                String subCType = cTypeId.endsWith("Category") ? StringUtils.substringBefore(cTypeId, "Category") : cTypeId;
                if (_cTypeExtensionPoint.hasExtension(subCType))
                {
                    count += _createEntryIfNotExists(subCType, childConfig, entry.getId());
                }
                else
                {
                    getLogger().error("Unknow content type of id {}", subCType);
                }
            }
        }
        
        return count;
        
    }

}
