/*
 *  Copyright 2012 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.odf.oai;

import java.io.IOException;
import java.util.Arrays;
import java.util.Collection;
import java.util.Set;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;
import org.apache.cocoon.ProcessingException;
import org.apache.cocoon.environment.ObjectModelHelper;
import org.apache.cocoon.environment.Request;
import org.apache.cocoon.xml.XMLUtils;
import org.apache.commons.lang.StringUtils;
import org.xml.sax.SAXException;

import org.ametys.runtime.i18n.I18nizableText;

/**
 * <code>ListSets</code> generator.
 */
public class ListSetsGenerator extends AbstractOAIVerbGenerator implements Serviceable
{
    private OaiSetExtensionPoint _oaiSetEP;

    public void service(ServiceManager serviceManager) throws ServiceException
    {
        _oaiSetEP = (OaiSetExtensionPoint) serviceManager.lookup(OaiSetExtensionPoint.ROLE);
    }
    
    @Override
    protected Collection<String> getRequiredParameters()
    {
        return Arrays.asList("verb");
    }

    @Override
    protected Collection<String> getAllowedParameters()
    {
        return Arrays.asList("verb", "resumptionToken");
    }

    @Override
    protected void generateVerb() throws IOException, SAXException, ProcessingException
    {
        Request request = ObjectModelHelper.getRequest(objectModel);

        String token = request.getParameter("resumptionToken");
        
        if (StringUtils.isNotEmpty(token))
        {
            generateError("badResumptionToken ", "This repository does not support resumption tokens");
            return;
        }
        
        Set<String> extensionsIds = _oaiSetEP.getExtensionsIds();
        
        XMLUtils.startElement(contentHandler, "ListSets");
        
        for (String extensionId : extensionsIds)
        {
            OaiSet oaiSet = _oaiSetEP.getExtension(extensionId);
            
            XMLUtils.startElement(contentHandler, "set");
            
            XMLUtils.createElement(contentHandler, "setSpec", extensionId);
            oaiSet.getName().toSAX(contentHandler, "setName");
            
            I18nizableText description = oaiSet.getDescription();
            if (description != null)
            {
                description.toSAX(contentHandler, "setDescription");
            }
            
            XMLUtils.endElement(contentHandler, "set");
        }
        
        XMLUtils.endElement(contentHandler, "ListSets");
    }

}
