/*
 *  Copyright 2019 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.odf;

import java.util.ArrayList;
import java.util.List;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;

import org.ametys.cms.content.RootContentHelper;
import org.ametys.cms.test.framework.WithCmsFeaturesTestCase;
import org.ametys.core.right.Profile;
import org.ametys.core.right.ProfileAssignmentStorageExtensionPoint;
import org.ametys.core.right.RightManager;
import org.ametys.core.right.RightProfilesDAO;
import org.ametys.core.user.CurrentUserProvider;
import org.ametys.plugins.repository.collection.AmetysObjectCollection;
import org.ametys.runtime.test.SimpleUserProvider;
import org.ametys.runtime.test.framework.AbstractPopulationAwareTestCase;
import org.ametys.runtime.test.framework.CocoonTestCase.NeedFeatures;

/**
 * Abstract test case for all ODF test cases.
 */
@NeedFeatures({
    "cms/root.content.helper",
    "core/runtime.rights.profiles.dao",
    "test/test.user.provider",
    "cms/workflow.reference.table",
    "odf/workflow.program",
    "odf/workflow.subprogram",
    "odf/workflow.container",
    "odf/workflow.courselist",
    "odf/workflow.course",
    "odf/workflow.course.part",
    "odf/workflow.orgunit",
    "odf/workflow.person"
})
public abstract class AbstractODFTestCase extends AbstractPopulationAwareTestCase implements WithCmsFeaturesTestCase
{
    /** The profile identifier */
    protected static final String PROFILE_ID = "profile-odf-test";
    
    /** The current user provider */
    protected static SimpleUserProvider _currentUserProvider;
    /** The extension point for profile assignment storages */
    protected static ProfileAssignmentStorageExtensionPoint _profileAssignmentStorageEP;
    /** The profiles DAO */
    protected static RightProfilesDAO _profilesDAO;
    /** The root content helper */
    protected static RootContentHelper _rootContentHelper;

    @Override
    public void lookupComponents(ServiceManager manager) throws ServiceException
    {
        super.lookupComponents(manager);
        
        _currentUserProvider = (SimpleUserProvider) manager.lookup(CurrentUserProvider.ROLE);
        _profileAssignmentStorageEP = (ProfileAssignmentStorageExtensionPoint) manager.lookup(ProfileAssignmentStorageExtensionPoint.ROLE);
        _profilesDAO = (RightProfilesDAO) manager.lookup(RightProfilesDAO.ROLE);
        _rootContentHelper = (RootContentHelper) manager.lookup(RootContentHelper.ROLE);
    }
    
    @Override
    public void prepareComponents() throws Exception
    {
        super.prepareComponents();
        
        _currentUserProvider.setUser(USER1_TEST_IDENTITY);
        
        AmetysObjectCollection rootOfContents = _rootContentHelper.getRootContent();
        _profileAssignmentStorageEP.allowProfileToAnyConnectedUser(RightManager.READER_PROFILE_ID, rootOfContents);
        
        Profile profile = _profilesDAO.addProfile(PROFILE_ID);

        List<String> rightsNeeded = new ArrayList<>();
        rightsNeeded.add("Workflow_Rights_Restore");
        rightsNeeded.add("Workflow_Rights_Archive");
        rightsNeeded.add("Workflow_Rights_Unarchive");
        rightsNeeded.add("Workflow_Rights_Edition_Online");
        rightsNeeded.add("Workflow_Rights_Validate");
        rightsNeeded.add("Workflow_Rights_Propose");
        rightsNeeded.add("Workflow_Rights_Refuse");

        rightsNeeded.add("ODF_Rights_OrgUnit_Edit");
        rightsNeeded.add("ODF_Rights_OrgUnit_Synchro");
        rightsNeeded.add("ODF_Rights_OrgUnit_Validate");
        rightsNeeded.add("ODF_Rights_OrgUnit_Propose");
        rightsNeeded.add("ODF_Rights_OrgUnit_Refuse");

        rightsNeeded.add("ODF_Rights_Program_Edit");
        rightsNeeded.add("ODF_Rights_Program_Synchro");
        rightsNeeded.add("ODF_Rights_Program_Validate");
        rightsNeeded.add("ODF_Rights_Program_Validate_Global");
        rightsNeeded.add("ODF_Rights_Program_Propose");
        rightsNeeded.add("ODF_Rights_Program_Refuse");

        rightsNeeded.add("ODF_Rights_SubProgram_Edit");
        rightsNeeded.add("ODF_Rights_SubProgram_Synchro");

        rightsNeeded.add("ODF_Rights_Container_Edit");
        rightsNeeded.add("ODF_Rights_Container_Synchro");

        rightsNeeded.add("ODF_Rights_CourseList_Edit");
        rightsNeeded.add("ODF_Rights_CourseList_Synchro");

        rightsNeeded.add("ODF_Rights_Course_Edit");
        rightsNeeded.add("ODF_Rights_Course_Synchro");
        rightsNeeded.add("ODF_Rights_Course_Validate");
        rightsNeeded.add("ODF_Rights_Course_Propose");
        rightsNeeded.add("ODF_Rights_Course_Refuse");

        rightsNeeded.add("ODF_Rights_Person_Edit");
        rightsNeeded.add("ODF_Rights_Person_Synchro");
        rightsNeeded.add("ODF_Rights_Person_Validate");

        rightsNeeded.add("ODF_Rights_Strategics_Fields");
        _profilesDAO.addRights(profile, rightsNeeded);

        _profileAssignmentStorageEP.allowProfileToUser(USER1_TEST_IDENTITY, profile.getId(), "/test");
        _profileAssignmentStorageEP.allowProfileToUser(USER1_TEST_IDENTITY, profile.getId(), "/cms");
        _profileAssignmentStorageEP.allowProfileToUser(USER1_TEST_IDENTITY, profile.getId(), "/odf");
    }
}
