<?xml version="1.0" encoding="UTF-8"?>
<!--
   Copyright 2014 Anyware Services

   Licensed under the Apache License, Version 2.0 (the "License");
   you may not use this file except in compliance with the License.
   You may obtain a copy of the License at

       http://www.apache.org/licenses/LICENSE-2.0

   Unless required by applicable law or agreed to in writing, software
   distributed under the License is distributed on an "AS IS" BASIS,
   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
   See the License for the specific language governing permissions and
   limitations under the License.
-->
<xsl:stylesheet version="1.0"
                xmlns:xsl="http://www.w3.org/1999/XSL/Transform"
                xmlns:docbook="http://docbook.org/ns/docbook"
                xmlns:xalan="http://xml.apache.org/xalan"
                xmlns:i18n="http://apache.org/cocoon/i18n/2.1"
                xmlns:fo="http://www.w3.org/1999/XSL/Format"
                xmlns:cms="http://ametys.org/cms/3.0"
                xmlns:date="http://exslt.org/dates-and-times"
                exclude-result-prefixes="docbook">
    
    <xsl:template match="/">
        <fo:root>
            <fo:layout-master-set>
                <xsl:call-template name="layout-master-set"/>
            </fo:layout-master-set>
            
            <!--  The bookmark tree -->
            <xsl:call-template name="bookmark-tree" />
            
            <!-- Front page -->
            <xsl:call-template name="front-page"/>
            
            <!-- Table of contents -->
            <xsl:call-template name="toc"/>
            
            <!-- Contents -->
            <xsl:call-template name="contents"/>
        </fo:root>
    </xsl:template>
    
    <xsl:template name="layout-master-set">
        <!-- Master single columns -->
        <fo:simple-page-master master-name="page" page-width="210mm" page-height="297mm" margin-left="1cm" margin-right="1cm" margin-top="2cm" margin-bottom="2cm">
            <fo:region-body margin-top="0cm" margin-bottom="1cm">
                <xsl:call-template name="region-body-definition"/>
            </fo:region-body>
            <fo:region-before extent="1cm">
                <xsl:call-template name="region-before-definition"/>
            </fo:region-before>
            <fo:region-after extent="0.5cm">
                <xsl:call-template name="region-after-definition"/>
            </fo:region-after>
        </fo:simple-page-master>

        <!-- Master 2 columns -->
        <fo:simple-page-master master-name="page-2-columns" page-width="210mm" page-height="297mm" margin-left="1cm" margin-right="1cm" margin-top="2cm" margin-bottom="2cm">
            <fo:region-body margin-top="0cm" margin-bottom="1cm" column-count="2" column-gap="20">
                <xsl:call-template name="region-body-definition"/>
            </fo:region-body>
            <fo:region-before>
                <xsl:call-template name="region-before-definition"/>
            </fo:region-before>
            <fo:region-after>
                <xsl:call-template name="region-after-definition"/>
            </fo:region-after>
        </fo:simple-page-master>
    </xsl:template>
    
    <xsl:variable name="programTree"><xsl:call-template name="program-tree"/></xsl:variable>
    
    <!-- + 
         | Get programs tree as follow:
         | <programs>
         |     <degree value="content://..." label="Licence">
         |        <program id="program://..." title="Licence Histoire">
         |            <fo>
         |               <fo:root>...</fo:root>
         |            </fo>
         |            <subprogram id="subprogram://..." title="Parcours Histoire et Géographie">
         |               <fo>
         |                   <fo:root>...</fo:root>
         |               </fo>
         |            </subprogram>
         |            <subprogram id="subprogram://..." title="Parcours Histoire - Option Journalisme">
         |               <fo>
         |                   <fo:root>...</fo:root>
         |               </fo>
         |            </subprogram>
         |        </program>
         |     </degree>
         |     <degree value="content://..." label="Master">
         |        <program id="program://..." title="Master Droits">
         |            <fo>
         |               <fo:root>...</fo:root>
         |            </fo>
         |        </program>
         |     </degree>
         | </programs>
         | Degrees are sort by 'order' property then title.
         | Programs are sort by alphabetical order.
         -->
    <xsl:template name="program-tree">   
        <programs>
            <xsl:for-each select="/programs/enumerated-metadata/items[@metadataName = 'degree']/item">
                <xsl:sort select="@order" data-type="number"/>
                <xsl:sort select="."/>
                
                <xsl:variable name="degree" select="@id"/>
                <xsl:if test="/programs/program[metadata/degree/@id = $degree]">
                    <degree value="{@id}" label="{.}">
                        <xsl:for-each select="/programs/program[metadata/degree/@id = $degree]">
                            <xsl:sort select="@title" />
                            <xsl:copy-of select="."/>
                        </xsl:for-each>
                    </degree>
                </xsl:if>
            </xsl:for-each>
        </programs>
    </xsl:template>
    
    <!-- // Bookmark -->
    <xsl:template name="bookmark-tree">
        <fo:bookmark-tree>
            <fo:bookmark internal-destination="chapter-refid-toc">
                <fo:bookmark-title><i18n:text i18n:key="PLUGINS_ODF_PDF_CATALOG_TABLE_OF_CONTENTS" i18n:catalogue="plugin.odf"/></fo:bookmark-title>
            </fo:bookmark>
            <xsl:apply-templates select="xalan:nodeset($programTree)/programs/degree" mode="bookmark"/>
        </fo:bookmark-tree>
    </xsl:template>
    
    <xsl:template match="degree" mode="bookmark">
        <xsl:choose>
            <xsl:when test="count(/programs/degree) > 1">
                <xsl:variable name="ref"><xsl:apply-templates select="program[1]" mode="chapter-reference"/></xsl:variable>
                <fo:bookmark internal-destination="chapter-refid-{$ref}">
                    <fo:bookmark-title><xsl:value-of select="@label"/></fo:bookmark-title>
                    
                    <xsl:apply-templates select="program" mode="bookmark"/>
                </fo:bookmark>
            </xsl:when>
            <xsl:otherwise>
                <xsl:apply-templates select="program" mode="bookmark"/>
            </xsl:otherwise>
        </xsl:choose>
    </xsl:template>
    
    <xsl:template match="program" mode="bookmark">
        <xsl:variable name="ref"><xsl:apply-templates select="." mode="chapter-reference"/></xsl:variable>
        <fo:bookmark internal-destination="chapter-refid-{$ref}">
            <fo:bookmark-title><xsl:value-of select="@title"/></fo:bookmark-title>
            
            <xsl:apply-templates select="subprogram" mode="bookmark"/>
        </fo:bookmark>
    </xsl:template>
    
    <xsl:template match="subprogram" mode="bookmark">
        <xsl:variable name="ref"><xsl:apply-templates select="." mode="chapter-reference"/></xsl:variable>
        <fo:bookmark internal-destination="chapter-refid-{$ref}">
            <fo:bookmark-title><xsl:value-of select="@title"/></fo:bookmark-title>
        </fo:bookmark>
    </xsl:template>
    
      
    <!-- // Front page -->
    <xsl:template name="front-page">
        <fo:page-sequence master-reference="page" force-page-count="no-force">
            <xsl:call-template name="xsl-region-before"/>
            <xsl:call-template name="xsl-region-after"/>
            <fo:flow flow-name="xsl-region-body">
                <xsl:call-template name="front-page-header"/>
                <xsl:call-template name="front-page-image"/>
                <xsl:call-template name="front-page-footer"/>
            </fo:flow>
        </fo:page-sequence>
    </xsl:template>
    
    <xsl:template name="front-page-header">
        <fo:block xsl:use-attribute-sets="front-page-styles">
            <fo:block xsl:use-attribute-sets="front-page-catalog-title">
                <i18n:text i18n:key="PLUGINS_ODF_PDF_CATALOG_TITLE" i18n:catalogue="plugin.odf"/>
            </fo:block>
            <fo:block xsl:use-attribute-sets="front-page-catalog-subtitle">
                <xsl:call-template name="catalog-subtitle"/>
                <xsl:call-template name="catalog-orgunits"/>
                <xsl:call-template name="catalog-degree"/>
            </fo:block>
        </fo:block>
    </xsl:template>
    
    <xsl:template name="front-page-image"/>
    <xsl:template name="front-page-footer"/>
    
    <xsl:template name="catalog-subtitle">
        <!-- Catalog's name -->
        <fo:block><xsl:value-of select="/programs/catalog"/></fo:block>
    </xsl:template>
    
    <xsl:template name="catalog-orgunits">
        <!-- Orgunits' title when catalog is restricted to one or more orgunits -->
        <xsl:if test="/programs/orgunit">
            <fo:block xsl:use-attribute-sets="front-page-catalog-orgunit">
                <xsl:for-each select="/programs/orgunit">
                    <fo:block><xsl:value-of select="."/></fo:block>
                    <xsl:if test="position() != last()"><fo:block/></xsl:if>
                </xsl:for-each>
            </fo:block>
        </xsl:if>
    </xsl:template>
    
    <xsl:template name="catalog-degree">
        <!-- Degree's title when catalog is restricted to one degree -->
        <xsl:if test="count(/programs/degree) = 1">
            <fo:block xsl:use-attribute-sets="front-page-catalog-degree">
                <xsl:value-of select="/programs/degree"/>
            </fo:block>
        </xsl:if>
    </xsl:template>
    
    <!-- // Front page styles -->
    <xsl:attribute-set name="front-page-styles">
        <xsl:attribute name="font-size">25px</xsl:attribute>
        <xsl:attribute name="text-align">center</xsl:attribute>
    </xsl:attribute-set>
    
    <xsl:attribute-set name="front-page-catalog-title">
        <xsl:attribute name="margin-top">0.5cm</xsl:attribute>
    </xsl:attribute-set>
    
    <xsl:attribute-set name="front-page-catalog-subtitle">
        <xsl:attribute name="font-size">20px</xsl:attribute>
        <xsl:attribute name="margin-top">0.5cm</xsl:attribute>
    </xsl:attribute-set>
    
    <xsl:attribute-set name="front-page-catalog-orgunit">
        <xsl:attribute name="font-size">18px</xsl:attribute>
        <xsl:attribute name="margin-top">0.5cm</xsl:attribute>
    </xsl:attribute-set>
    
    <xsl:attribute-set name="front-page-catalog-degree">
        <xsl:attribute name="font-size">18px</xsl:attribute>
        <xsl:attribute name="margin-top">0.5cm</xsl:attribute>
    </xsl:attribute-set>
    <!-- // END Front page styles -->
    
    <!-- // END Front page -->
       
    

    <!-- // Table of contents -->
    <xsl:template name="toc">   
        <!-- +
             | Generates TOC such as:
             | 1. Licence
             |   1.1 Licence Histoire
             |     1.1.1 Parcours Histoire et Géographie
             |     1.1.2 Parcours Histoire - Option Journalisme
             | 2. Master
             |   2.1 Master Droits
             |   2.2 Master Mathématiques
             |     2.2.1 Parcours Mathématiques pour l’assurance et la finance
             |     2.2.2 Parcours Mathématiques générale
             + -->
        <fo:page-sequence master-reference="page" force-page-count="no-force">
            <xsl:call-template name="xsl-region-before"/>
            <xsl:call-template name="xsl-region-after"/>
            <fo:flow flow-name="xsl-region-body">
                <fo:block id="chapter-refid-toc"></fo:block>
                <fo:block xsl:use-attribute-sets="catalog-toc-title">
                    <i18n:text i18n:key="PLUGINS_ODF_PDF_CATALOG_TABLE_OF_CONTENTS" i18n:catalogue="plugin.odf"/>
                </fo:block>
                
                <xsl:apply-templates select="xalan:nodeset($programTree)/programs/degree" mode="toc"/>
            </fo:flow>
        </fo:page-sequence>
    </xsl:template>
    
    <xsl:template match="degree" mode="toc">
        <xsl:if test="count(/programs/degree) > 1">
            <!-- Skip degree in TOC if catalog is restricted to one degree only -->
            <fo:block xsl:use-attribute-sets="catalog-toc-degree" keep-with-next.within-page="always">
                <xsl:apply-templates select="." mode="chapter-number"/>
                <xsl:value-of select="@label"/>
            </fo:block>
        </xsl:if>
        
         <xsl:apply-templates select="program" mode="toc"/>
    </xsl:template>
    
    <xsl:template match="program" mode="toc">
        <fo:block xsl:use-attribute-sets="catalog-toc-program" text-align-last="justify">
            <xsl:variable name="ref"><xsl:apply-templates select="." mode="chapter-reference"/></xsl:variable>
            
            <fo:basic-link xsl:use-attribute-sets="catalog-toc-program-link" internal-destination="chapter-refid-{$ref}">
                
                <xsl:apply-templates select="." mode="chapter-number"/>
                <xsl:value-of select="@title"/>&#160;<fo:leader leader-pattern='dots' rule-thickness='.2pt' leader-alignment='reference-area' font-size="10pt"/>&#160;
                
                <fo:page-number-citation ref-id="chapter-refid-{$ref}"/>
            </fo:basic-link>
        </fo:block>
        
        <!-- Generate subprograms TOC when including subprograms -->
        <xsl:apply-templates select="subprogram" mode="toc"/>
        
        <!-- +
             | OR generate subprograms TOC when subprograms are included in PDF of programs.
             | Be careful ! DO NOT check the case "include subprograms" in subprograms are already included in PDF of programs.
             + -->
        <xsl:apply-templates select="fo/fo:root/fo:page-sequence//fo:block[@role = 'heading']" mode="toc"/>
    </xsl:template>
    
    <xsl:template match="subprogram" mode="toc">
        <fo:block xsl:use-attribute-sets="catalog-toc-subprogram" text-align-last="justify">
            <xsl:variable name="ref"><xsl:apply-templates select="." mode="chapter-reference"/></xsl:variable>
            
            <fo:basic-link xsl:use-attribute-sets="catalog-toc-subprogram-link" internal-destination="chapter-refid-{$ref}">
                
                <xsl:apply-templates select="." mode="chapter-number"/>
                <xsl:value-of select="@title"/>&#160;<fo:leader leader-pattern='dots' rule-thickness='.2pt' leader-alignment='reference-area' font-size="10pt"/>&#160;
                
                <fo:page-number-citation ref-id="chapter-refid-{$ref}"/>
            </fo:basic-link>
        </fo:block>
    </xsl:template>
    
    <!-- // Template to generate TOC with  -->
    <xsl:template match="fo:block[@role = 'heading']" mode="toc">
        <fo:block xsl:use-attribute-sets="catalog-toc-subprogram" text-align-last="justify">
            <xsl:variable name="ref"><xsl:call-template name="ahead-reference"/></xsl:variable>
            
            <fo:basic-link xsl:use-attribute-sets="catalog-toc-subprogram-link" internal-destination="ahead-refid-{$ref}">
                
                <xsl:call-template name="ahead-number"/>
                <xsl:value-of select="."/>&#160;<fo:leader leader-pattern='dots' rule-thickness='.2pt' leader-alignment='reference-area' font-size="10pt"/>&#160;
                
                <fo:page-number-citation ref-id="ahead-refid-{$ref}"/>
            </fo:basic-link>
        </fo:block>
    </xsl:template>
    
    <!-- TOC number format -->
    <xsl:template match="degree" mode="chapter-number">
        <xsl:number level="multiple" count="degree" format="1. "/>
    </xsl:template>
    
    <xsl:template match="program" mode="chapter-number">
        <xsl:choose>
            <xsl:when test="count(/programs/degree) = 1">
                <!-- Skip degree in toc -->
                <xsl:number level="multiple" count="program" format="1. "/>
            </xsl:when>
            <xsl:otherwise>
                <xsl:number level="multiple" count="degree|program" format="1.1. "/>
            </xsl:otherwise>
        </xsl:choose>
    </xsl:template>
    
    <xsl:template match="subprogram" mode="chapter-number">
        <xsl:choose>
            <xsl:when test="count(/programs/degree) = 1">
                <!-- Skip degree in toc -->
                <xsl:number level="multiple" count="program|subprogram" format="1.1. "/>
            </xsl:when>
            <xsl:otherwise>
                <xsl:number level="multiple" count="degree|program|subprogram" format="1.1.1. "/>
            </xsl:otherwise>
        </xsl:choose>
    </xsl:template>
    
    <!-- Used when subprograms are already included in PDF programs -->
    <xsl:template name="ahead-number">
        <xsl:choose>
            <xsl:when test="count(/programs/degree) = 1">
                <!-- Skip degree in toc -->
                <xsl:number level="multiple" count="program|fo:block[@role = 'heading']" format="1.1. "/>
            </xsl:when>
            <xsl:otherwise>
                <xsl:number level="multiple" count="degree|program|fo:block[@role = 'heading']" format="1.1.1. "/>
            </xsl:otherwise>
        </xsl:choose>
    </xsl:template>
    <!-- //END TOC number format -->
    
    <!-- // TOC styles -->
    <xsl:attribute-set name="catalog-toc-title">
        <xsl:attribute name="font-size">20px</xsl:attribute>
        <xsl:attribute name="text-align">center</xsl:attribute>
    </xsl:attribute-set>
    
    <xsl:attribute-set name="catalog-toc-degree">
        <xsl:attribute name="font-size">14px</xsl:attribute>
        <xsl:attribute name="margin-top">0.5cm</xsl:attribute>
        <xsl:attribute name="margin-left">0.5cm</xsl:attribute>
        <xsl:attribute name="margin-right">1.2cm</xsl:attribute>
    </xsl:attribute-set>
    
    <xsl:attribute-set name="catalog-toc-program">
        <xsl:attribute name="text-align">justify</xsl:attribute>
        <xsl:attribute name="margin-left">1cm</xsl:attribute>
        <xsl:attribute name="margin-right">1.2cm</xsl:attribute>
        <xsl:attribute name="margin-top">0.3cm</xsl:attribute>
        <xsl:attribute name="space-after">3pt</xsl:attribute>
    </xsl:attribute-set>
    
    <xsl:attribute-set name="catalog-toc-program-link">
        <xsl:attribute name="font-size">14px</xsl:attribute>
        <xsl:attribute name="margin-top">0.7cm</xsl:attribute>
    </xsl:attribute-set>
    
    <xsl:attribute-set name="catalog-toc-subprogram">
        <xsl:attribute name="text-align">justify</xsl:attribute>
        <xsl:attribute name="margin-left">2cm</xsl:attribute>
        <xsl:attribute name="margin-right">1.2cm</xsl:attribute>
        <xsl:attribute name="space-after">3pt</xsl:attribute>
    </xsl:attribute-set>
    
    <xsl:attribute-set name="catalog-toc-subprogram-link">
        <xsl:attribute name="font-size">12px</xsl:attribute>
        <xsl:attribute name="margin-top">0.7cm</xsl:attribute>
    </xsl:attribute-set>
    <!-- //END TOC styles -->
    
    <!-- // Chapter reference -->
    <xsl:template match="program" mode="chapter-reference">
        <xsl:number level="multiple" count="degree|program" format="1.1"/>
    </xsl:template>
    
    <xsl:template match="subprogram" mode="chapter-reference">
        <xsl:number level="multiple" count="degree|program|subprogram" format="1.1.1"/>
    </xsl:template>
    
    <!-- Used when subprograms are already included in PDF programs -->
    <xsl:template name="ahead-reference">
        <xsl:number level="multiple" count="degree|program|fo:block[@role = 'heading']" format="1.1.1"/>
    </xsl:template>   
    <!-- // END Chapter reference -->
    
    <!-- // END Table of contents -->
    
    <xsl:template name="contents">   
        <xsl:apply-templates select="xalan:nodeset($programTree)/programs//program"/>
    </xsl:template>
    
   <xsl:template match="program|subprogram">
        <xsl:variable name="self" select="."/>
        <xsl:variable name="isLast" select="position() = last()"/>

        <xsl:for-each select="fo/fo:root/fo:page-sequence">
            <fo:page-sequence master-reference="{@master-reference}">
                <xsl:call-template name="xsl-region-before"/>
                <xsl:call-template name="xsl-region-after"/>

                <fo:flow flow-name="xsl-region-body">
                    <xsl:if test="position() = 1">
                        <!-- Block for bookmark and anchors -->
                        <xsl:variable name="ref"><xsl:apply-templates select="$self" mode="chapter-reference"/></xsl:variable>
                        <fo:block id="chapter-refid-{$ref}"></fo:block>
                    </xsl:if>

                    <xsl:apply-templates select="fo:flow" />

                    <xsl:if test="$isLast and position() = last()">
                        <xsl:apply-templates select="$self" mode="theEnd"/>
                    </xsl:if>
                </fo:flow>
            </fo:page-sequence>
        </xsl:for-each>
        
        <xsl:apply-templates select="subprogram"/>
    </xsl:template>
    
    <xsl:template match="program" mode="theEnd">
        <xsl:if test="not(subprogram) and not(ancestor::degree/following-sibling::degree)">
            <fo:block id="theEnd"/>
        </xsl:if>
    </xsl:template>
    
    <xsl:template match="subprogram" mode="theEnd">
        <xsl:variable name="parentProgram" select="ancestor::program"/>
        
        <xsl:if test="not(ancestor::program/following-sibling::program) and not(ancestor::degree/following-sibling::degree)">
            <fo:block id="theEnd"/>
        </xsl:if>
    </xsl:template>
    
    <xsl:template match="fo:flow">
        <xsl:apply-templates select="fo:block[not(@id = 'theEnd')]" mode="content"/>
    </xsl:template>
    
    <xsl:template match="@*|node()" mode="content">
        <xsl:copy>
            <xsl:apply-templates select="@*|node()" mode="content"/>
        </xsl:copy>
    </xsl:template>
    
    <xsl:template match="fo:block[@role = 'internal-chapter']" mode="content">
        <xsl:variable name="flowId" select="generate-id(ancestor::fo:flow[1])"/>
        <fo:block id="{concat($flowId, '-', @id)}"/>
    </xsl:template>
    
    <xsl:template match="fo:basic-link[@internal-destination]" mode="content">
        <xsl:copy>
            <xsl:copy-of select="@*[local-name() != 'internal-destination']"/>
            
            <xsl:variable name="flowId" select="generate-id(ancestor::fo:flow[1])"/>
            <xsl:attribute name='internal-destination'><xsl:value-of select="concat($flowId, '-', @internal-destination)"/></xsl:attribute>
            
            <!-- Remove page numbering and dots -->
            <xsl:apply-templates select="node()[local-name() != 'page-number-citation' and local-name() != 'leader']" mode="content"/>
        </xsl:copy>
    </xsl:template>
    
    <xsl:template match="fo:block[@role = 'heading']" mode="content">
        <xsl:variable name="ref"><xsl:call-template name="ahead-reference"/></xsl:variable>
        <fo:block id="ahead-refid-{$ref}"></fo:block>
        
        <xsl:copy>
            <xsl:apply-templates select="@*|node()" mode="content"/>
        </xsl:copy>
    </xsl:template>
    
    <!-- // Footer -->
    <xsl:template name="xsl-region-after">
        <fo:static-content flow-name="xsl-region-after">
            <fo:block margin-left="1cm" margin-right="1cm">
                <fo:table table-layout="fixed" margin-top="0pt" padding-top="0pt">
                    <fo:table-column column-width="4cm"/>
                    <fo:table-column/>
                    <fo:table-column column-width="4cm"/>
                    <fo:table-body>
                        <fo:table-row>
                            <xsl:call-template name="footer-page-number"/>
                            <xsl:call-template name="footer-center"/>
                            <xsl:call-template name="footer-logo"/>
                        </fo:table-row>
                    </fo:table-body>
                </fo:table>
            </fo:block>
        </fo:static-content> 
    </xsl:template>
    
    <xsl:template name="footer-center">
        <fo:table-cell margin-left="0cm" padding="0" text-align="center" vertical-align="middle">
            <fo:block>
                <xsl:call-template name="footer-text-style"/>
                
                <xsl:call-template name="footer-text"/>
                <xsl:call-template name="footer-date"/>
            </fo:block>
        </fo:table-cell>
    </xsl:template>
    
    <xsl:template name="footer-date">
        <fo:block>
            <xsl:variable name="now" select="date:date-time()"/>
            <xsl:number format="01" value="date:day-in-month($now)"/>
            <xsl:text>/</xsl:text>
            <xsl:number format="01" value="date:month-in-year($now)"/>
            <xsl:text>/</xsl:text>
            <xsl:value-of select="date:year($now)"/>
        </fo:block>
    </xsl:template>
    
    <xsl:template name="footer-text">
        <fo:block><i18n:text i18n:key="PLUGINS_ODF_PDF_CATALOG_FOOTER_TEXT" i18n:catalogue="plugin.odf"/></fo:block>
    </xsl:template>
    
    <xsl:template name="footer-page-number">
        <fo:table-cell vertical-align="middle" text-align="left">
            <fo:block>
                <xsl:call-template name="footer-pages-style"/>
                <fo:page-number /> / <fo:page-number-citation ref-id="theEnd" />
            </fo:block>
        </fo:table-cell>
    </xsl:template>
    
    <xsl:template name="footer-logo"/>
    
    <!-- // Footer styles -->
    <xsl:template name="footer-pages-style"/>

    <xsl:template name="footer-text-style"/>
    
    <xsl:template name="footer-date-style"/>
    <!-- // END Footer styles -->
    
    <!-- // END Footer -->

    <!-- // Header -->    
    <xsl:template name="xsl-region-before">
        <fo:static-content flow-name="xsl-region-before">
              <fo:block-container absolute-position="fixed" left="0in" top="0in" width="21cm" height="29.7cm">
                  <fo:block />
              </fo:block-container>
        </fo:static-content>
    </xsl:template>
    <!-- // END Header -->
    

    <xsl:template name="region-body-definition"/>
    
    <xsl:template name="region-before-definition"/>
    
    <xsl:template name="region-after-definition"/>

</xsl:stylesheet>