/*
 *  Copyright 2019 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.odf.skill;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.LinkedHashMap;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.function.Predicate;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import org.apache.avalon.framework.component.Component;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;

import org.ametys.cms.ObservationConstants;
import org.ametys.cms.data.ContentValue;
import org.ametys.cms.indexing.solr.SolrIndexHelper;
import org.ametys.cms.repository.Content;
import org.ametys.cms.repository.ContentQueryHelper;
import org.ametys.cms.repository.ContentTypeExpression;
import org.ametys.core.observation.Event;
import org.ametys.core.observation.ObservationManager;
import org.ametys.core.right.RightManager;
import org.ametys.core.right.RightManager.RightResult;
import org.ametys.core.ui.Callable;
import org.ametys.core.user.CurrentUserProvider;
import org.ametys.odf.ODFHelper;
import org.ametys.odf.ProgramItem;
import org.ametys.odf.course.Course;
import org.ametys.odf.program.AbstractProgram;
import org.ametys.odf.program.Container;
import org.ametys.odf.program.Program;
import org.ametys.odf.program.SubProgram;
import org.ametys.odf.skill.workflow.SkillEditionFunction;
import org.ametys.plugins.repository.AmetysObjectIterable;
import org.ametys.plugins.repository.AmetysObjectResolver;
import org.ametys.plugins.repository.ModifiableAmetysObject;
import org.ametys.plugins.repository.data.holder.group.ModifiableModelAwareRepeater;
import org.ametys.plugins.repository.data.holder.group.ModifiableModelAwareRepeaterEntry;
import org.ametys.plugins.repository.query.expression.AndExpression;
import org.ametys.plugins.repository.query.expression.Expression;
import org.ametys.plugins.repository.query.expression.Expression.Operator;
import org.ametys.plugins.repository.query.expression.StringExpression;
import org.ametys.runtime.config.Config;
import org.ametys.runtime.plugin.component.AbstractLogEnabled;

/**
 * ODF skills helper
 */
public class ODFSkillsHelper extends AbstractLogEnabled implements Serviceable, Component
{
    /** The avalon role. */
    public static final String ROLE = ODFSkillsHelper.class.getName();
    
    /** The internal attribute name to excluded from skills */
    public static final String SKILLS_EXCLUDED_INTERNAL_ATTRIBUTE_NAME = "excluded";
    
    /** The ametys object resolver */
    protected AmetysObjectResolver _resolver;

    /** The ODF helper */
    protected ODFHelper _odfHelper;
    
    /** The observation manager */
    protected ObservationManager _observationManager;

    /** The Solr index helper */
    protected SolrIndexHelper _solrIndexHelper;
    
    /** The current user provider */
    protected CurrentUserProvider _currentUserProvider;

    /** The right manager */
    protected RightManager _rightManager;
    
    public void service(ServiceManager manager) throws ServiceException
    {
        _resolver = (AmetysObjectResolver) manager.lookup(AmetysObjectResolver.ROLE);
        _odfHelper = (ODFHelper) manager.lookup(ODFHelper.ROLE);
        _observationManager = (ObservationManager) manager.lookup(ObservationManager.ROLE);
        _currentUserProvider = (CurrentUserProvider) manager.lookup(CurrentUserProvider.ROLE);
        _rightManager = (RightManager) manager.lookup(RightManager.ROLE);
    }
    
    /**
     * Determines if rules are enabled
     * @return <code>true</code> if rules are enabled
     */
    public static boolean isSkillsEnabled()
    {
        return Config.getInstance().getValue("odf.skills.enabled", false, false);
    }
    
    /**
     * Get the name of the catalog of a skill content
     * @param contentId The id of content
     * @return The catalog's name or null if the content does not have a catalog
     */
    @Callable (rights = Callable.NO_CHECK_REQUIRED)
    public String getCatalog(String contentId)
    {
        Content content = _resolver.resolveById(contentId);
        
        if (content.hasValue("catalog"))
        {
            return content.getValue("catalog");
        }
        
        return null;
    }
    
    /**
     * Get the path of the ODF root content
     * @return The path of the ODF root content
     */
    @Callable (rights = Callable.NO_CHECK_REQUIRED)
    public String getOdfRootContentPath()
    {
        return _odfHelper.getRootContent(false).getPath();
    }

    /**
     * Exclude or include the program items from skills display
     * @param programItemIds the list of program item ids
     * @param excluded <code>true</code> if the program items need to be excluded.
     * @return the map of changed program items properties
     */
    @Callable (rights = Callable.CHECKED_BY_IMPLEMENTATION)
    public Map<String, Object> setProgramItemsExclusion(List<String> programItemIds, boolean excluded)
    {
        Map<String, Object> results = new HashMap<>();
        results.put("allright-program-items", new ArrayList<>());
        results.put("noright-program-items", new ArrayList<>());
        
        for (String programItemId : programItemIds)
        {
            ProgramItem programItem = _resolver.resolveById(programItemId);
            if (programItem instanceof AbstractProgram || programItem instanceof Container)
            {
                Map<String, Object> programItem2Json = new HashMap<>();
                programItem2Json.put("id", programItem.getId());
                programItem2Json.put("title", ((Content) programItem).getTitle());
                
                if (_rightManager.hasRight(_currentUserProvider.getUser(), "ODF_Right_Skills_Excluded", programItem) == RightResult.RIGHT_ALLOW)
                {
                    ((Content) programItem).getInternalDataHolder().setValue(SKILLS_EXCLUDED_INTERNAL_ATTRIBUTE_NAME, excluded);
                    ((ModifiableAmetysObject) programItem).saveChanges();
                    
                    @SuppressWarnings("unchecked")
                    List<Map<String, Object>> allRightProgramItems = (List<Map<String, Object>>) results.get("allright-program-items");
                    allRightProgramItems.add(programItem2Json);
                    
                    Map<String, Object> eventParams = new HashMap<>();
                    eventParams.put(ObservationConstants.ARGS_CONTENT, programItem);
                    eventParams.put(ObservationConstants.ARGS_CONTENT_ID, programItem.getId());
                    eventParams.put(org.ametys.odf.observation.OdfObservationConstants.ODF_CONTENT_SKILLS_EXCLUSION_ARG, excluded);
                    _observationManager.notify(new Event(org.ametys.odf.observation.OdfObservationConstants.ODF_CONTENT_SKILLS_EXCLUSION_CHANGED, _currentUserProvider.getUser(), eventParams));
                }
                else
                {
                    @SuppressWarnings("unchecked")
                    List<Map<String, Object>> noRightProgramItems = (List<Map<String, Object>>) results.get("noright-program-items");
                    noRightProgramItems.add(programItem2Json);
                }
            }
        }
        
        return results;
        
    }
    
    /**
     * <code>true</code> if the program item is excluded from skills display
     * @param programItem the program item
     * @return <code>true</code> if the program item is excluded from skills display
     */
    public boolean isExcluded(ProgramItem programItem)
    {
        // If the skills are not enabled, every item is excluded
        if (!isSkillsEnabled())
        {
            return true;
        }
        
        if (programItem instanceof AbstractProgram || programItem instanceof Container)
        {
            return ((Content) programItem).getInternalDataHolder().getValue(SKILLS_EXCLUDED_INTERNAL_ATTRIBUTE_NAME, false);
        }
        
        return false;
    }
    
    /**
     * Get the skills distribution by courses over a {@link AbstractProgram}
     * Distribution is computed over the course of first level only
     * @param abstractProgram The program or subProgram for which to get the skills distribution
     * @return the skills distribution or null if the content is not a program or a compatible subProgram
     */
    public Map<Content, Map<Content, Set<Content>>> getSkillsDistribution(AbstractProgram abstractProgram)
    {
        return getSkillsDistribution(abstractProgram, 1);
    }
    
    /**
     * Get the skills distribution by courses over a {@link AbstractProgram}
     * Distribution is computed over the course of first level only
     * @param abstractProgram The program or subProgram for which to get the skills distribution
     * @param maxDepth the max depth of courses. For example, set to 1 to compute distribution over UE only, set 2 to compute distribution over UE and EC, ...
     * @return the skills distribution or null if the content is not a program or a compatible subProgram
     */
    public Map<Content, Map<Content, Set<Content>>> getSkillsDistribution(AbstractProgram abstractProgram, int maxDepth)
    {
        // If it is a program, the parentProgram that contains the skills is itself
        if (abstractProgram instanceof Program program)
        {
            return getSkillsDistribution(program, program, maxDepth);
        }
        // If it is a subProgram not shared, retrieve the parent program that contains the skills
        else if (abstractProgram instanceof SubProgram subProgram && !_odfHelper.isShared(subProgram))
        {
            // Since it is not shared, we can get the parent program
            Program parentProgram = (Program) _odfHelper.getParentPrograms(subProgram).toArray()[0];
            
            return getSkillsDistribution(parentProgram, subProgram, maxDepth);
        }
        
        return null;
    }
    
    /**
     * Get the skills distribution by courses over a {@link ProgramItem}
     * @param parentProgram The parent program that contains the skills
     * @param program the program
     * @param maxDepth the max depth of courses. For example, set to 1 to compute distribution over UE only, set 2 to compute distribution over UE and EC, ...
     * @return the skills distribution as Map&lt;MacroSkill, Map&lt;MicroSkill, Set&lt;Course&gt;&gt;&gt;
     */
    public Map<Content, Map<Content, Set<Content>>> getSkillsDistribution(Program parentProgram, AbstractProgram program, int maxDepth)
    {
        // Map<MacroSkill, Map<MicroSkill, Set<Course>>>
        Map<Content, Map<Content, Set<Content>>> skillsDistribution = new LinkedHashMap<>();
        
        if (!isExcluded(program))
        {
            // Get all macro skills. First skills, then transversal skills
            List<Content> macroSkills = parentProgram.getSkills();
            macroSkills.addAll(parentProgram.getTransversalSkills());
            
            // First initialize macro and micro skills to :
            // 1. Keep the macro skills order defined in the program
            // 2. Keep the micro skills order defined in the macro skill
            for (Content skill: macroSkills)
            {
                LinkedHashMap<Content, Set<Content>> microSkills = new LinkedHashMap<>();
                for (ContentValue contentValue : skill.getValue("microSkills", false, new ContentValue[0]))
                {
                    microSkills.put(contentValue.getContent(), new HashSet<>());
                }
                skillsDistribution.put(skill, microSkills);
            }
            
            _buildSkillsDistribution(parentProgram, program, skillsDistribution, maxDepth);
        }
        
        return skillsDistribution;
    }
    
    
    private void _buildSkillsDistribution(Program parentProgram, ProgramItem programItem, Map<Content, Map<Content, Set<Content>>> skillsDistribution, int maxDepth)
    {
        if (programItem instanceof Course course)
        {
            // If it is a course, get its skills for the program
            _buildSkillsDistribution(parentProgram, course, course, skillsDistribution, 1, maxDepth);
        }
        else
        {
            // If it is not a course, go through its course children
            List<ProgramItem> children = _odfHelper.getChildProgramItems(programItem)
                .stream()
                .filter(Predicate.not(this::isExcluded))
                .collect(Collectors.toList());
            for (ProgramItem childProgramItem : children)
            {
                _buildSkillsDistribution(parentProgram, childProgramItem, skillsDistribution, maxDepth);
            }
        }
    }
    
    private void _buildSkillsDistribution(Program parentProgram, Course course, Course parentCourse, Map<Content, Map<Content, Set<Content>>> skillsDistribution, int depth, int maxDepth)
    {
        // Get the micro skills of the course by program
        List<? extends ModifiableModelAwareRepeaterEntry> microSkillsByProgramEntries = Optional.of(course)
                .map(e -> e.getRepeater(Course.ACQUIRED_MICRO_SKILLS))
                .map(ModifiableModelAwareRepeater::getEntries)
                .orElse(List.of());

        // Get the micro skills of the course for the program
        ModifiableModelAwareRepeaterEntry microSkillsForProgram = microSkillsByProgramEntries.stream()
                                   .filter(entry -> ((ContentValue) entry.getValue("program")).getContentId().equals(parentProgram.getId()))
                                   .findFirst()
                                   .orElse(null);
        
        if (microSkillsForProgram != null)
        {
            // Get the micro skills
            ContentValue[] microSkills = microSkillsForProgram.getValue(Course.ACQUIRED_MICRO_SKILLS_SKILLS);
            
            if (microSkills != null)
            {
                for (ContentValue microSkillContentValue : microSkills)
                {
                    Content microSkill = microSkillContentValue.getContent();
                    ContentValue macroSkill = microSkill.getValue("parentMacroSkill");
                    
                    // Add the microSkill under the macro skill if it is not already
                    // Map<MicroSkills, Set<Course>>
                    Map<Content, Set<Content>> coursesByMicroSkills = skillsDistribution.computeIfAbsent(macroSkill.getContent(), __ -> new LinkedHashMap<>());
                    
                    // Add the course under the micro skill if it is not already
                    Set<Content> coursesForMicroSkill = coursesByMicroSkills.computeIfAbsent(microSkill, __ -> new LinkedHashSet<>());
                    coursesForMicroSkill.add(parentCourse);
                }
            }
        }

        if (depth < maxDepth)
        {
            // Get skills distribution over child courses
            course.getCourseLists()
                .stream()
                .forEach(cl ->
                {
                    cl.getCourses()
                        .stream().forEach(c ->
                        {
                            _buildSkillsDistribution(parentProgram, c, parentCourse, skillsDistribution, depth + 1, maxDepth);
                        });
                });
        }
    }

    /**
     * Get all micro skills of a requested catalog
     * @param catalog The catalog
     * @return The micro skills
     */
    public AmetysObjectIterable<Content> getMicroSkills(String catalog)
    {
        List<Expression> exprs = new ArrayList<>();
        exprs.add(new ContentTypeExpression(Operator.EQ, SkillEditionFunction.MICRO_SKILL_TYPE));
        exprs.add(new StringExpression("catalog", Operator.EQ, catalog));
        Expression expression = new AndExpression(exprs.toArray(Expression[]::new));
        
        String query = ContentQueryHelper.getContentXPathQuery(expression);
        return _resolver.<Content>query(query);
    }
    
    /**
     * Get the micro skills of a program
     * @param program The program
     * @return The microskills attached to the program
     */
    public Stream<String> getProgramMicroSkills(Program program)
    {
        Set<Content> macroSkills = new HashSet<>();
        macroSkills.addAll(program.getSkills());
        macroSkills.addAll(program.getTransversalSkills());
        
        return macroSkills
            .stream()
            .map(macroSkill -> macroSkill.<ContentValue[]>getValue("microSkills"))
            .filter(Objects::nonNull)
            .flatMap(Stream::of)
            .map(ContentValue::getContentId);
    }
}
