/*
 *  Copyright 2024 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.odf.cdmfr;

import java.io.IOException;
import java.io.InputStream;
import java.util.Collection;
import java.util.Set;
import java.util.stream.Stream;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.cocoon.util.log.SLF4JLoggerAdapter;
import org.apache.commons.lang.StringUtils;

import org.ametys.odf.CallWSHelper;
import org.ametys.odf.catalog.CatalogsManager;
import org.ametys.odf.program.Program;
import org.ametys.runtime.config.Config;

/**
 * Processor to send the CDM-fr file
 */
public class SendCDMfrProcessor extends AbstractCDMfrProcessor
{
    /** The component role. */
    public static final String ROLE = SendCDMfrProcessor.class.getName();
    
    /** The catalogs manager */
    protected CatalogsManager _catalogsManager;

    private String _catalogMode;
    private Collection<String> _selectedCatalogs;
    
    @Override
    public void service(ServiceManager manager) throws ServiceException
    {
        super.service(manager);
        _catalogsManager = (CatalogsManager) manager.lookup(CatalogsManager.ROLE);
    }
    
    @Override
    public void initialize() throws Exception
    {
        super.initialize();
        _catalogMode = Config.getInstance().getValue("odf.publish.catalog.mode", true, "ALL");
        if (_catalogMode.equals("SELECT"))
        {
            _selectedCatalogs = org.ametys.core.util.StringUtils.stringToCollection(Config.getInstance().getValue("odf.publish.catalogs"));
        }
    }
    
    @Override
    protected String getConfigActiveParam()
    {
        return "odf.publish.server";
    }
    
    @Override
    protected Stream<Program> filterPrograms(Stream<Program> programs)
    {
        switch (_catalogMode)
        {
            case "ALL":
                // Do not filter programs
                return programs;
            case "DEFAULT":
                // Filter programs on default catalog
                String defaultCatalogName = _catalogsManager.getDefaultCatalogName();
                return programs.filter(p -> p.getCatalog().equals(defaultCatalogName));
            case "SELECT":
                // Filter programs on selected catalogs
                return programs.filter(p -> _selectedCatalogs.contains(p.getCatalog()));
            default:
                // Should not happen
                throw new RuntimeException("The parameter 'odf.publish.catalog.mode' has not a valid value.");
        }
    }
    
    @Override
    protected boolean isCDMfrForAmetys()
    {
        return true;
    }
    
    @Override
    protected void processProgram(Program program, InputStream cdmfrContent) throws IOException
    {
        // Send CDM-fr to portails for the given program
        Set<String> failedUrl = CallWSHelper.callWS("_odf-sync/upload-cdm", cdmfrContent, new SLF4JLoggerAdapter(getLogger()));
        if (failedUrl == null || failedUrl.size() > 0)
        {
            getLogger().error("The program {} can't be synchronized with portals {}", program.getId(), StringUtils.defaultString(StringUtils.join(failedUrl, ", ")));
        }
    }
    
    @Override
    protected String getErrorMessage()
    {
        return "Unable to publish CDM-fr on distant server";
    }
}
