/*
 *  Copyright 2020 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.odf.export.pdf;

import java.io.IOException;
import java.net.MalformedURLException;
import java.util.HashMap;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;

import org.apache.avalon.framework.context.Context;
import org.apache.avalon.framework.context.ContextException;
import org.apache.avalon.framework.context.Contextualizable;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.cocoon.ProcessingException;
import org.apache.cocoon.components.ContextHelper;
import org.apache.cocoon.components.source.impl.SitemapSource;
import org.apache.cocoon.environment.ObjectModelHelper;
import org.apache.cocoon.environment.Request;
import org.apache.cocoon.generation.ServiceableGenerator;
import org.apache.cocoon.xml.AttributesImpl;
import org.apache.cocoon.xml.XMLUtils;
import org.apache.commons.lang.StringUtils;
import org.xml.sax.SAXException;

import org.ametys.cms.CmsConstants;
import org.ametys.cms.repository.Content;
import org.ametys.cms.repository.DefaultContent;
import org.ametys.core.util.IgnoreRootHandler;
import org.ametys.odf.EducationalPathHelper;
import org.ametys.odf.NoLiveVersionException;
import org.ametys.odf.ODFHelper;
import org.ametys.odf.ProgramItem;
import org.ametys.odf.catalog.Catalog;
import org.ametys.odf.catalog.CatalogsManager;
import org.ametys.odf.course.Course;
import org.ametys.odf.program.SubProgram;
import org.ametys.odf.schedulable.ArchiveEducationalBookletSchedulable;
import org.ametys.odf.schedulable.EducationalBookletSchedulable;
import org.ametys.plugins.repository.AmetysObjectResolver;
import org.ametys.plugins.repository.UnknownAmetysObjectException;
import org.ametys.plugins.repository.jcr.DefaultAmetysObject;


/**
 * Generator producing the SAX of subprogram and its courses for the educational booklet 
 */ 
public class EducationalBookletGenerator extends ServiceableGenerator implements Contextualizable
{
    private static final String __EXPORT_MODE = "educational-booklet";
    
    /** The Ametys object resolver */
    protected AmetysObjectResolver _resolver;
    
    /** The ODF helper */
    protected ODFHelper _odfHelper;

    /** The catalog manager */
    protected CatalogsManager _catalogManager;

    /** The avalon context */
    protected Context _context;
    
    @Override
    public void service(ServiceManager sManager) throws ServiceException
    {
        super.service(sManager);
        _resolver = (AmetysObjectResolver) sManager.lookup(AmetysObjectResolver.ROLE);
        _odfHelper = (ODFHelper) sManager.lookup(ODFHelper.ROLE);
        _catalogManager = (CatalogsManager) sManager.lookup(CatalogsManager.ROLE);
    }
    
    public void contextualize(Context context) throws ContextException
    {
        _context = context;
    }
    
    public void generate() throws IOException, SAXException, ProcessingException
    {
        Request request = ObjectModelHelper.getRequest(objectModel);
        
        contentHandler.startDocument();

        AttributesImpl attrs = new AttributesImpl();
        
        String archiveDateAsString = (String) request.getAttribute(ArchiveEducationalBookletSchedulable.PARAM_ARCHIVE_DATE);
        if (StringUtils.isNotBlank(archiveDateAsString))
        {
            attrs.addCDATAAttribute("isValidated", "true");
            attrs.addCDATAAttribute("archiveDate", archiveDateAsString);
        }
        else
        {
            attrs.addCDATAAttribute("isValidated", "false");
        }

        XMLUtils.startElement(contentHandler, "booklet", attrs);
        
        String programItemId = (String) request.getAttribute(EducationalBookletSchedulable.PARAM_PROGRAM_ITEM_ID);
        ProgramItem programItem = _resolver.resolveById(programItemId);
        
        _saxCatalog(programItem);
        
        try
        {
            Content content = (Content) programItem;
            _odfHelper.switchToLiveVersion((DefaultContent) content);
            _saxProgramItem(programItem, "programItem", List.of(programItem), null);
            
            boolean includeSubPrograms = (boolean) request.getAttribute(EducationalBookletSchedulable.PARAM_INCLUDE_SUBPROGRAMS);
            
            if (includeSubPrograms)
            {
                for (SubProgram subProgram : _getChildSubPrograms(programItem))
                {
                    _saxProgramItem(subProgram, "subprogram", List.of(programItem, subProgram), programItem);
                }
            }
            
            for (Course course : _getChildCourses(programItem))
            {
                _saxProgramItem(course, "course", List.of(course), programItem);
            }
        }
        catch (NoLiveVersionException e) 
        {
            throw new IllegalArgumentException("The program item with id " + programItemId + " has no live version.");
        }
        
        XMLUtils.endElement(contentHandler, "booklet");
        contentHandler.endDocument();
    }
    
    /**
     * Sax the catalog of program item
     * @param programItem the program item
     * @throws SAXException if an error occurred
     */
    protected void _saxCatalog(ProgramItem programItem) throws SAXException
    {
        String catalogName = programItem.getCatalog();
        Catalog catalog = _catalogManager.getCatalog(catalogName);
        
        if (catalog != null)
        {
            AttributesImpl attrs = new AttributesImpl();
            attrs.addCDATAAttribute("name", catalogName);
            XMLUtils.createElement(contentHandler, "catalog", attrs, catalog.getTitle());
        }
    }
    
    /**
     * Get all child courses from the program item
     * @param programItem the program item
     * @return the set of child courses
     * @throws MalformedURLException if an error occurred
     * @throws IOException if an error occurred
     * @throws SAXException if an error occurred
     */
    protected Set<Course> _getChildCourses(ProgramItem programItem) throws MalformedURLException, IOException, SAXException
    {
        Set<Course> courses = new LinkedHashSet<>();
        for (ProgramItem childProgramItem : _odfHelper.getChildProgramItems(programItem))
        {
            try
            {
                _odfHelper.switchToLiveVersion((DefaultAmetysObject) childProgramItem);
                if (childProgramItem instanceof Course)
                {
                    courses.add((Course) childProgramItem);
                }
                courses.addAll(_getChildCourses(childProgramItem));
            }
            catch (NoLiveVersionException e) 
            {
                getLogger().warn("Cannot add the course to the educational booklet : Live label is required but there is no Live version for content " + childProgramItem.getId());
            }
        }
        
        return courses;
    }
    
    /**
     * Get the direct child {@link SubProgram}s of a {@link ProgramItem}
     * @param programItem the program item
     * @return the subprograms
     */
    protected Set<SubProgram> _getChildSubPrograms(ProgramItem programItem)
    {
        return _odfHelper.getChildProgramItems(programItem)
            .stream()
            .filter(SubProgram.class::isInstance)
            .map(SubProgram.class::cast)
            .collect(Collectors.toSet());
    }
    
    /**
     * Sax a {@link ProgramItem} as fo
     * @param programItem the program item
     * @param tagName the xml tag name
     * @param ancestorPath The path of this program item (computed from the initial program item). Can be a partial path.
     * @param rootProgramItem The root program item ancestor for this booklet
     * @throws MalformedURLException if an error occurred
     * @throws IOException if an error occurred
     * @throws SAXException if an error occurred
     */
    protected void _saxProgramItem(ProgramItem programItem, String tagName, List<ProgramItem> ancestorPath, ProgramItem rootProgramItem) throws MalformedURLException, IOException, SAXException
    {
        AttributesImpl attrs = new AttributesImpl();
        attrs.addCDATAAttribute("id", programItem.getId());
        attrs.addCDATAAttribute("name", programItem.getName());
        attrs.addCDATAAttribute("title", ((Content) programItem).getTitle());
        attrs.addCDATAAttribute("code", programItem.getCode());
        
        XMLUtils.startElement(contentHandler, tagName, attrs);
        _saxContentAsFo((Content) programItem, ancestorPath, rootProgramItem);
        XMLUtils.endElement(contentHandler, tagName);
    }
    
    /**
     * Sax content as fo
     * @param content the content
     * @param ancestorPath The path of this program item (computed from the initial program item). Can be a partial path.
     * @param rootAncestor The root program item ancestor for this booklet
     * @throws MalformedURLException if an error occurred
     * @throws IOException if an error occurred
     * @throws SAXException if an error occurred
     */
    protected void _saxContentAsFo(Content content, List<ProgramItem> ancestorPath, ProgramItem rootAncestor) throws MalformedURLException, IOException, SAXException
    {
        XMLUtils.startElement(contentHandler, "fo");
        SitemapSource src = null;      
        
        Request request = ContextHelper.getRequest(_context);
        
        try
        {
            request.setAttribute(EducationalPathHelper.PROGRAM_ITEM_ANCESTOR_PATH_REQUEST_ATTR, ancestorPath);
            if (rootAncestor != null)
            {
                request.setAttribute(EducationalPathHelper.ROOT_PROGRAM_ITEM_REQUEST_ATTR, rootAncestor);
            }
            
            Map<String, Object> pdfParameters = new HashMap<>();
            pdfParameters.put("versionLabel", CmsConstants.LIVE_LABEL);
            pdfParameters.put("exportMode", __EXPORT_MODE);
            
            String uri = "cocoon://_plugins/odf/_content/" + content.getName() + ".fo";
            src = (SitemapSource) resolver.resolveURI(uri, null, pdfParameters);
            src.toSAX(new IgnoreRootHandler(contentHandler));
        }
        catch (UnknownAmetysObjectException e)
        {
            // The content may be archived
        }
        finally
        {
            resolver.release(src);
            
            request.removeAttribute(EducationalPathHelper.PROGRAM_ITEM_ANCESTOR_PATH_REQUEST_ATTR);
            request.removeAttribute(EducationalPathHelper.ROOT_PROGRAM_ITEM_REQUEST_ATTR);
        }
        
        XMLUtils.endElement(contentHandler, "fo");
    }
}
