/*
 *  Copyright 2019 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.odf.migration;

import java.util.HashSet;
import java.util.Map;
import java.util.Set;

import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.tuple.Pair;

import org.ametys.cms.data.ContentDataHelper;
import org.ametys.cms.repository.Content;
import org.ametys.cms.repository.ModifiableContent;
import org.ametys.odf.enumeration.OdfReferenceTableEntry;
import org.ametys.odf.enumeration.OdfReferenceTableHelper;
import org.ametys.plugins.repository.AmetysRepositoryException;

import com.opensymphony.workflow.WorkflowException;

/**
 * Component class to migrate nature enseignement
 */
public class MigrateNatureEnseignementComponent extends MigrateCoursePartComponent
{
    /** The component role. */
    @SuppressWarnings("hiding")
    public static final String ROLE = MigrateNatureEnseignementComponent.class.getName();
    
    /**
     * Migrate the nature enseignement
     * @throws AmetysRepositoryException if an error occurs
     * @throws WorkflowException if an error occurs
     */
    public void migrateNaturesEnseignement() throws AmetysRepositoryException, WorkflowException
    {
        migrateNaturesEnseignement(null);
    }
    
    /**
     * Migrate the nature enseignement
     * @param natureEnseignementCategories the map of nature enseignement categories
     * @throws AmetysRepositoryException if an error occurs
     * @throws WorkflowException if an error occurs
     */
    public void migrateNaturesEnseignement(Map<String, Pair<String, Long>> natureEnseignementCategories) throws AmetysRepositoryException, WorkflowException
    {
        // Migrate categories and get the minimum order to set
        Pair<Long, Set<String>> orderAndCategories = _getOrderAndCategories(natureEnseignementCategories);
        Long order = orderAndCategories.getLeft();
        Set<String> categories = orderAndCategories.getRight();
        
        // Check for each nature if the category has been set with an old value and create a category if needed
        for (OdfReferenceTableEntry nature : _odfRefTableHelper.getItems(OdfReferenceTableHelper.ENSEIGNEMENT_NATURE))
        {
            String category = ContentDataHelper.getContentIdFromContentData(nature.getContent(), "category");
            if (StringUtils.isNotEmpty(category) && !categories.contains(category))
            {
                // Order always incremented, the perfect way would be following numbers but it complexify the source code
                order++;
                OdfReferenceTableEntry categoryEntry = _getOrCreateNatureEnseignement(category, category, order);
                
                // Set attribute category with category entry value
                Content natureContent = nature.getContent();
                if (natureContent instanceof ModifiableContent)
                {
                    ModifiableContent modifNatureContent = (ModifiableContent) natureContent;
                    modifNatureContent.setValue("category", categoryEntry.getId());
                    modifNatureContent.saveChanges();
                }
            }
        }
    }
    
    private Pair<Long, Set<String>> _getOrderAndCategories(Map<String, Pair<String, Long>> natureEnseignementCategories) throws AmetysRepositoryException, WorkflowException
    {
     // Create CM, TD, TP categories if it doesn't exist
        _createNatureEnseignementCategories(natureEnseignementCategories);
        
        Long order = 1L;
        Set<String> categories = new HashSet<>();
        
        for (OdfReferenceTableEntry category : _odfRefTableHelper.getItems(OdfReferenceTableHelper.ENSEIGNEMENT_NATURE_CATEGORY))
        {
            Long categoryOrder = category.getContent().getValue("order", false, 0L);
            if (categoryOrder > order)
            {
                order = categoryOrder;
            }
            categories.add(category.getId());
        }
        return Pair.of(order, categories);
    }
}
