/*
 *  Copyright 2010 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.odf.program.generators;

import java.io.IOException;
import java.util.HashMap;
import java.util.Map;

import org.apache.cocoon.components.source.impl.SitemapSource;
import org.apache.cocoon.environment.ObjectModelHelper;
import org.apache.cocoon.environment.Request;
import org.apache.cocoon.generation.Generator;
import org.apache.cocoon.xml.AttributesImpl;
import org.apache.cocoon.xml.XMLUtils;
import org.xml.sax.ContentHandler;
import org.xml.sax.SAXException;

import org.ametys.cms.CmsConstants;
import org.ametys.cms.repository.Content;
import org.ametys.core.util.DateUtils;
import org.ametys.core.util.IgnoreRootHandler;
import org.ametys.odf.ODFHelper;
import org.ametys.odf.program.Program;
import org.ametys.odf.program.SubProgram;
import org.ametys.plugins.repository.UnknownAmetysObjectException;

/**
 * {@link Generator} for rendering raw content data for a {@link Program}, to 
 */
public class ProgramPdfContentGenerator extends ProgramContentGenerator
{
    /**
     * SAX a sub program
     * @param subProgram the sub program to SAX
     * @param parentPath the parent path
     * @throws SAXException if an error occurs
     */
    @Override
    protected void saxSubProgram (SubProgram subProgram, String parentPath) throws SAXException
    {
        try
        {
            AttributesImpl attrs = new AttributesImpl();
            attrs.addCDATAAttribute("title", subProgram.getTitle());
            attrs.addCDATAAttribute("code", subProgram.getCode());
            attrs.addCDATAAttribute("id", subProgram.getId());
            if (parentPath != null)
            {
                attrs.addCDATAAttribute("path", parentPath + "/" + subProgram.getName());
            }
            XMLUtils.startElement(contentHandler, "subprogram", attrs);
            saxContent(subProgram, contentHandler, "abstract");  
            XMLUtils.endElement(contentHandler, "subprogram");
        }
        catch (IOException e)
        {
            throw new SAXException(e);
        }
        catch (UnknownAmetysObjectException e)
        {
            throw new SAXException(e);
        }
    }
    
    
    @Override
    protected void saxContent (Content content, ContentHandler handler, String viewName) throws SAXException, IOException
    {
        Request request = ObjectModelHelper.getRequest(objectModel);
        
        AttributesImpl attrs = new AttributesImpl();
        attrs.addCDATAAttribute("id", content.getId());
        attrs.addCDATAAttribute("name", content.getName());
        attrs.addCDATAAttribute("title", content.getTitle());
        attrs.addCDATAAttribute("lastModifiedAt", DateUtils.zonedDateTimeToString(content.getLastModified()));
        
        XMLUtils.startElement(handler, "content", attrs);
        
        Map<String, String> params = new HashMap<>();
        if (request.getAttribute(ODFHelper.REQUEST_ATTRIBUTE_VALID_LABEL) != null)
        {
            params.put("versionLabel", CmsConstants.LIVE_LABEL);
        }
        
        String uri = _contentHelper.getContentViewUrl(content, viewName, "xml", params);
        SitemapSource src = null;
        
        try
        {
            src = (SitemapSource) _srcResolver.resolveURI(uri);
            src.toSAX(new IgnoreRootHandler(handler));
        }
        finally
        {
            _srcResolver.release(src);
        }
        
        XMLUtils.endElement(handler, "content");
    }
}
