/*
 *  Copyright 2025 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.odf.restriction;

import org.apache.avalon.framework.configuration.Configuration;
import org.apache.avalon.framework.configuration.ConfigurationException;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;

import org.ametys.cms.model.restrictions.DefaultRestriction;
import org.ametys.cms.model.restrictions.RestrictedModelItem;
import org.ametys.cms.repository.Content;
import org.ametys.odf.ODFHelper;
import org.ametys.odf.ProgramItem;

/**
 * Restrictions for data that should be available only when the element is shared.
 */
public class SharedOnlyRestriction extends DefaultRestriction
{
    private boolean _sharedOnly;
    private ODFHelper _odfHelper;

    @Override
    public void configure(Configuration configuration) throws ConfigurationException
    {
        Configuration classConfig = configuration.getChild("custom-restriction", true);
        _sharedOnly = classConfig.getChild("shared-program-item-only").getValueAsBoolean(false);
        super.configure(classConfig);
    }
    
    @Override
    public void service(ServiceManager smanager) throws ServiceException
    {
        super.service(smanager);
        _odfHelper = (ODFHelper) smanager.lookup(ODFHelper.ROLE);
    }
    
    @Override
    public RestrictionResult canRead(Content content, RestrictedModelItem modelItem)
    {
        RestrictionResult canRead = super.canRead(content, modelItem);
        
        if (canRead == RestrictionResult.FALSE || _sharedOnly && content instanceof ProgramItem programItem && !_odfHelper.isShared(programItem))
        {
            return RestrictionResult.FALSE;
        }
        
        return canRead;
    }
}
