/*
 *  Copyright 2017 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.odf.clientsideelement;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;

import javax.jcr.RepositoryException;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;

import org.ametys.cms.repository.Content;
import org.ametys.cms.repository.WorkflowAwareContent;
import org.ametys.core.observation.Event;
import org.ametys.core.observation.ObservationManager;
import org.ametys.core.ui.Callable;
import org.ametys.odf.EducationalPathHelper;
import org.ametys.odf.EducationalPathHelper.EducationalPathReference;
import org.ametys.odf.ODFHelper;
import org.ametys.odf.ProgramItem;
import org.ametys.odf.observation.OdfObservationConstants;
import org.ametys.runtime.i18n.I18nizableText;
import org.ametys.runtime.model.ModelHelper;

/**
 * Set the attribute of type 'content' of a content, with another content.
 * If content is a {@link ProgramItem}, additional check will be done on catalog and content language
 */
public class SetContentAttributeClientSideElement extends org.ametys.cms.clientsideelement.relations.SetContentAttributeClientSideElement
{
    /** The ODF helper */
    protected ODFHelper _odfHelper;
    /** The educational path helper */
    protected EducationalPathHelper _educationalHelper;
    /** The observation manager */
    protected ObservationManager _observationManager;
    
    @Override
    public void service(ServiceManager manager) throws ServiceException
    {
        super.service(manager);
        _odfHelper = (ODFHelper) manager.lookup(ODFHelper.ROLE);
        _educationalHelper = (EducationalPathHelper) manager.lookup(EducationalPathHelper.ROLE);
        _observationManager = (ObservationManager) manager.lookup(ObservationManager.ROLE);
    }
    
    @Override
    @Callable (rights = Callable.CHECKED_BY_IMPLEMENTATION) // Rights checking is handled by the workflow action and any write restrictions on the attribute
    public Map<String, Object> setContentAttribute(List<String> contentIdsToReference, Map<String, Integer> contentIdsToEdit, List<Map<String, String>> contentsToEditToRemove,
            String attributePath, List<String> workflowActionIds, Map<String, Object> additionalParams)
    {
        Map<String, Object> result = super.setContentAttribute(contentIdsToReference, contentIdsToEdit, contentsToEditToRemove, attributePath, workflowActionIds, additionalParams);
        
        if (Boolean.TRUE.equals(result.get("success")))
        {
            // Notify observers for educational paths that were changed or removed
            @SuppressWarnings("unchecked")
            List<String> contentsInError = result.containsKey("errorIds") ? (List<String>) result.get("errorIds") : List.of();
            
            _notifyObserverIfHierarchyChanged(contentIdsToReference, contentIdsToEdit, contentsToEditToRemove, contentsInError);
        }
        
        return result;
    }
    
    /**
     * Notify observers if the hierarchy of program item has been changed
     * @param contentIdsToReference The list of content identifiers that will be added as values in the content field
     * @param contentIdsToEdit The map {key: content identifiers to edit and that will have an attribute of type content modified; value: the new position if attribute is multiple and it is a reorder of values. May be null or equals to -1 if it is not a reorder}
     * @param contentsToEditToRemove  The list of content to edit to remove currently referenced content. Keys are "contentId" and "valueToRemove"
     * @param contentsInError the ids of contents in error
     */
    protected void _notifyObserverIfHierarchyChanged(List<String> contentIdsToReference, Map<String, Integer> contentIdsToEdit, List<Map<String, String>> contentsToEditToRemove, List<String> contentsInError)
    {
        if (!contentsToEditToRemove.isEmpty())
        {
            // 'move' or 'remove' mode
            for (Map<String, String> removeObject : contentsToEditToRemove)
            {
                String contentIdToEdit = removeObject.get("contentId");
                String valueToRemove = removeObject.get("valueToRemove");
                
                if (!contentsInError.contains(contentIdToEdit))
                {
                    Content removedContent = (Content) _resolver.resolveById(valueToRemove);
                    Content contentToEdit = (Content) _resolver.resolveById(contentIdToEdit);
                    
                    if (removedContent instanceof ProgramItem programItem && contentToEdit instanceof ProgramItem oldParentProgramItem)
                    {
                        // A program item has been detached from its parent
                        Map<String, Object> eventParams = new HashMap<>();
                        eventParams.put(OdfObservationConstants.ARGS_PROGRAM_ITEM, programItem);
                        eventParams.put(OdfObservationConstants.ARGS_PROGRAM_ITEM_ID, programItem.getId());
                        eventParams.put(OdfObservationConstants.ARGS_OLD_PARENT_PROGRAM_ITEM_ID, oldParentProgramItem.getId());
                        
                        if (contentIdsToReference.contains(valueToRemove))
                        {
                            // move case: the program item has been detached and appended to new parent(s)
                            List<String> newParentProgramItems = contentIdsToEdit.entrySet().stream()
                                .filter(e -> e.getValue() == null || e.getValue() == -1) // ignore reorder
                                .map(e -> e.getKey())
                                .filter(id -> !contentsInError.contains(id))
                                .map(id -> _resolver.resolveById(id))
                                .filter(ProgramItem.class::isInstance)
                                .map(ProgramItem.class::cast)
                                .map(ProgramItem::getId)
                                .toList();
                            
                            if (!newParentProgramItems.isEmpty())
                            {
                                eventParams.put(OdfObservationConstants.ARGS_NEW_PARENT_PROGRAM_ITEM_IDS, newParentProgramItems);
                            }
                        }
                        
                        _observationManager.notify(new Event(OdfObservationConstants.EVENT_PROGRAM_ITEM_HIERARCHY_CHANGED, _currentUserProvider.getUser(), eventParams));
                    }
                }
            }
        }
        else if (!contentIdsToReference.isEmpty())
        {
            // 'append' mode
            for (String contentIdToReference : contentIdsToReference)
            {
                Content appendedContent = (Content) _resolver.resolveById(contentIdToReference);
                
                if (appendedContent instanceof ProgramItem programItem)
                {
                    List<String> newParentProgramItems = contentIdsToEdit.entrySet().stream()
                            .filter(e -> e.getValue() == null || e.getValue() == -1) // ignore reorder
                            .map(e -> e.getKey())
                            .filter(id -> !contentsInError.contains(id))
                            .map(id -> _resolver.resolveById(id))
                            .filter(ProgramItem.class::isInstance)
                            .map(ProgramItem.class::cast)
                            .map(ProgramItem::getId)
                            .toList();
                    
                    if (!newParentProgramItems.isEmpty())
                    {
                        // The program item has been appended to new parent(s)
                        Map<String, Object> eventParams = new HashMap<>();
                        eventParams.put(OdfObservationConstants.ARGS_PROGRAM_ITEM, programItem);
                        eventParams.put(OdfObservationConstants.ARGS_NEW_PARENT_PROGRAM_ITEM_IDS, newParentProgramItems);
                        
                        _observationManager.notify(new Event(OdfObservationConstants.EVENT_PROGRAM_ITEM_HIERARCHY_CHANGED, _currentUserProvider.getUser(), eventParams));
                    }
                }
            }
        }
    }
    
    /**
     * Get the educational paths that reference the program items to removed
     * @param contentIdsToReference The list of content identifiers that will be added as values in the content field
     * @param contentsToEditToRemove  The list of content to edit to remove currently referenced content. Keys are "contentId" and "valueToRemove"
     * @return a Map with program items referencing the given program items in a educational path attribute as key and the list of educational path references as value
     */
    @Callable (rights = Callable.CHECKED_BY_IMPLEMENTATION)
    public Map<String, Object> getEducationalPathReferences(List<String> contentIdsToReference, List<Map<String, String>> contentsToEditToRemove)
    {
        Map<String, Object> result = new HashMap<>();
        
        for (Map<String, String> removeObject : contentsToEditToRemove)
        {
            String contentIdToEdit = removeObject.get("contentId");
            String valueToRemove = removeObject.get("valueToRemove");
            
            Content contentToEdit = (Content) _resolver.resolveById(contentIdToEdit);
            Content contentToRemove = (Content) _resolver.resolveById(valueToRemove);
            
            if (contentToEdit instanceof ProgramItem oldParentProgramItem && contentToRemove instanceof ProgramItem programItemToRemove && _rightManager.currentUserHasReadAccess(programItemToRemove))
            {
                result.put("contentToRemove", Map.of("id", contentToRemove.getId(), "title", contentToRemove.getTitle()));
                
                List<Map<String, Object>> references2json = new ArrayList<>();
                
                try
                {
                    // When a program item is detached from its parent, all educational paths refering it AND its old parent become obsolete
                    Map<ProgramItem, List<EducationalPathReference>> educationalPathReferences = _educationalHelper.getEducationalPathReferences(programItemToRemove, oldParentProgramItem);
                    for (ProgramItem programItem : educationalPathReferences.keySet())
                    {
                        Map<String, Object> json = new HashMap<>();
                        json.put("id", programItem.getId());
                        json.put("code", programItem.getDisplayCode());
                        json.put("title", ((Content) programItem).getTitle());
                        
                        List<Map<String, Object>> educationalPathsToJson = new ArrayList<>();
                        List<EducationalPathReference> references = educationalPathReferences.get(programItem);
                        for (EducationalPathReference reference : references)
                        {
                            String repeaterDefinitionPath = ModelHelper.getDefinitionPathFromDataPath(reference.repeaterEntryPath());
                            I18nizableText repeaterLabel = ((Content) programItem).getDataHolder().getDefinition(repeaterDefinitionPath).getLabel();
                            educationalPathsToJson.add(Map.of("value", reference.value().toString(), "repeaterLabel", repeaterLabel));
                        }
                        
                        json.put("educationalPaths", educationalPathsToJson);
                     
                        references2json.add(json);
                    }
                    
                    result.put("educationalPathReferences", references2json);
                }
                catch (RepositoryException e)
                {
                    throw new RuntimeException("Unable to get the educational path attributes that refer the moved contents", e);
                }
            }
        }
        
        return result;
    }
    
    @Override
    protected Map<WorkflowAwareContent, Integer> _filterContentsToEdit(Map<WorkflowAwareContent, Integer> contentsToEdit, List<String> contentIdsToReference, List<I18nizableText> errorMessages, List<String> errorIds, Map<String, Object> additionalParams)
    {
        Map<WorkflowAwareContent, Integer> validContents = new LinkedHashMap<>();
        Map<WorkflowAwareContent, Integer> filteredContents = super._filterContentsToEdit(contentsToEdit, contentIdsToReference, errorMessages, errorIds, additionalParams);
        
        @SuppressWarnings("unchecked")
        List<Content> refContents = (List<Content>) _resolve(contentIdsToReference);
        
        for (Map.Entry<WorkflowAwareContent, Integer> contentEntry : filteredContents.entrySet())
        {
            WorkflowAwareContent content = contentEntry.getKey();
            
            for (Content refContent : refContents)
            {
                if (!_odfHelper.isRelationCompatible(refContent, content, errorMessages, additionalParams))
                {
                    errorIds.add(content.getId());
                }
                else
                {
                    validContents.put(content, contentEntry.getValue());
                }
            }
        }
        
        return validContents;
    }
    
}
