/*
 *  Copyright 2018 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.odf.observation;

import java.util.Arrays;
import java.util.HashMap;
import java.util.Map;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;

import org.ametys.cms.ObservationConstants;
import org.ametys.cms.data.ContentValue;
import org.ametys.cms.repository.Content;
import org.ametys.cms.workflow.ContentWorkflowHelper;
import org.ametys.core.observation.Event;
import org.ametys.core.observation.ObservationManager;
import org.ametys.core.observation.Observer;
import org.ametys.core.user.CurrentUserProvider;
import org.ametys.odf.course.Course;
import org.ametys.odf.coursepart.CoursePart;
import org.ametys.runtime.plugin.component.AbstractLogEnabled;

/**
 * Observer to unlink a {@link CoursePart} on {@link Course} deletion.
 */
public class CourseDeletedObserver extends AbstractLogEnabled implements Observer, Serviceable
{
    private ObservationManager _observationManager;
    private CurrentUserProvider _userProvider;
    private ContentWorkflowHelper _contentWorkflowHelper;

    @Override
    public void service(ServiceManager manager) throws ServiceException
    {
        _observationManager = (ObservationManager) manager.lookup(ObservationManager.ROLE);
        _userProvider = (CurrentUserProvider) manager.lookup(CurrentUserProvider.ROLE);
        _contentWorkflowHelper = (ContentWorkflowHelper) manager.lookup(ContentWorkflowHelper.ROLE);
    }
    
    @Override
    public boolean supports(Event event)
    {
        if (event.getId().equals(ObservationConstants.EVENT_CONTENT_DELETING))
        {
            Content content = (Content) event.getArguments().get(ObservationConstants.ARGS_CONTENT);
            return content instanceof Course;
        }
        
        return false;
    }

    @Override
    public int getPriority()
    {
        return 0;
    }

    @Override
    public void observe(Event event, Map<String, Object> transientVars) throws Exception
    {
        Course course = (Course) event.getArguments().get(ObservationConstants.ARGS_CONTENT);
        for (CoursePart coursePart : course.getCourseParts())
        {
            ContentValue[] newCourses = Arrays.stream(coursePart.getValue(CoursePart.PARENT_COURSES, false, new ContentValue[0]))
                                        .filter(contentValue -> !contentValue.getContentId().equals(course.getId()))
                                        .toArray(ContentValue[]::new);
            coursePart.setValue(CoursePart.PARENT_COURSES, newCourses);
            coursePart.saveChanges();
            
            // Notify listeners
            Map<String, Object> eventParams = new HashMap<>();
            eventParams.put(ObservationConstants.ARGS_CONTENT, coursePart);
            eventParams.put(ObservationConstants.ARGS_CONTENT_ID, coursePart.getId());
            _observationManager.notify(new Event(ObservationConstants.EVENT_CONTENT_MODIFIED, _userProvider.getUser(), eventParams));
           
            _contentWorkflowHelper.doAction(coursePart, 22);
        }
    }

}
