/*
 *  Copyright 2024 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.odf.tree;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;

import org.ametys.core.ui.ClientSideElementHelper;
import org.ametys.core.util.JSONUtils;
import org.ametys.plugins.contentstree.ui.TreeToolClientSideElement;

/**
 * Client side element for ODF tree tool that supports indicators
 *
 */
public class ODFTreeToolClientSideElement extends TreeToolClientSideElement
{
    /** Extension point for indicators */
    protected ODFTreeIndicatorExtensionPoint _odfTreeIndicatorEP;
    /** JSON utils */
    protected JSONUtils _jsonUtils;
    
    @Override
    public void service(ServiceManager smanager) throws ServiceException
    {
        super.service(smanager);
        _odfTreeIndicatorEP = (ODFTreeIndicatorExtensionPoint) smanager.lookup(ODFTreeIndicatorExtensionPoint.ROLE);
        _jsonUtils = (JSONUtils) smanager.lookup(JSONUtils.ROLE);
    }
    
    @Override
    public List<Script> getScripts(boolean ignoreRights, Map<String, Object> contextParameters)
    {
        List<Script> scripts = super.getScripts(ignoreRights, contextParameters);
        
        if (scripts.size() > 0)
        {
            Script script = ClientSideElementHelper.cloneScript(scripts.get(0));
            
            scripts = new ArrayList<>();
            
            List<Map<String, Object>> indicators = new ArrayList<>();
            
            for (String indicatorId : _odfTreeIndicatorEP.getExtensionsIds())
            {
                ODFTreeIndicator indicator = _odfTreeIndicatorEP.getExtension(indicatorId);
                
                indicators.add(_indicator2json(indicator));
                
                List<Script> indicatorScripts = indicator.getScripts(contextParameters);
                if (indicatorScripts.size() > 0)
                {
                    // Add required scripts and/or css files for indicator
                    Script indicatorScript = ClientSideElementHelper.cloneScript(indicatorScripts.get(0));
                    script.getScriptFiles().addAll(indicatorScript.getScriptFiles());
                    script.getCSSFiles().addAll(indicatorScript.getCSSFiles());
                }
            }
            
            script.getParameters().put("indicators", _jsonUtils.convertObjectToJson(indicators));
            
            scripts = new ArrayList<>();
            scripts.add(script);
        }
        
        return scripts;
    }
    
    private Map<String, Object> _indicator2json(ODFTreeIndicator indicator)
    {
        Map<String, Object> indicatorInfo = new HashMap<>();
        indicatorInfo.put("id", indicator.getId());
        indicatorInfo.put("label", indicator.getLabel());
        indicatorInfo.put("description", indicator.getDescription());
        indicatorInfo.put("iconGlyph", indicator.getIconGlyph());
        indicatorInfo.put("matchFn", indicator.getMatchJSFunction());
        indicatorInfo.put("applyFn", indicator.getApplyJSFunction());
        return indicatorInfo;
    }
}
