/*
 *  Copyright 2019 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */

/**
 * Button controller that launches the global validation report on the server.
 * @private
 */
Ext.define('Ametys.plugins.odf.validation.GlobalValidationReportController', {
	extend: 'Ametys.ribbon.element.ui.ButtonController',
	
    statics : {
        
        /**
         * Start a global validation report on all programs
         * @param {Ametys.ribbon.element.ui.ButtonController} controller The controller calling this function
         */
        startReport: function(controller)
	    {
	        var me = this;
	        Ametys.Msg.confirm("{{i18n PLUGINS_ODF_GLOBAL_VALIDATION_START_CONFIRM_LABEL}}",
	                "{{i18n PLUGINS_ODF_GLOBAL_VALIDATION_START_CONFIRM_DESCRIPTION}}",
	                function(answer)
	                {
	                    if (answer == 'yes')
	                    {
	                        Ametys.data.ServerComm.callMethod({
	                            role: "org.ametys.odf.schedulable.GlobalValidationReport",
	                            methodName: "startReport",
	                            parameters: [],
	                            callback: {
	                                handler: me._startReportCb,
	                                scope: me
	                            },
	                            waitMessage: false,
	                            errorMessage: "{{i18n PLUGINS_ODF_UITOOL_GLOBAL_VALIDATION_CHECK_START_ERROR}}"
	                        });
	                    }
	                }
	            );
	    },
        
        /**
         * @private
         * Callback function invoked after lauching the report
         * @param {Object} response the server response
         * @param {Object/Object[]} args the callback arguments
         */
        _startReportCb: function(response, args)
        {
	        if (response.success)
	        {
	            Ext.create("Ametys.message.Message", { type: Ametys.message.Message.GLOBAL_VALIDATION_CHECK_STARTED });
	        }
	        else if (response.error == "already-running")
	        {
	            Ametys.Msg.show({
	                title: "{{i18n PLUGINS_ODF_UITOOL_GLOBAL_VALIDATION_REPORT_LABEL}}",
	                msg: "{{i18n PLUGINS_ODF_UITOOL_GLOBAL_VALIDATION_CHECK_ALREADY_RUNNING}}",
	                buttons: Ext.Msg.OK,
	                icon: Ext.Msg.ERROR
	            });
	        }
	        else
	        {
	            Ametys.Msg.show({
                    title: "{{i18n PLUGINS_ODF_UITOOL_GLOBAL_VALIDATION_REPORT_LABEL}}",
                    msg: "{{i18n PLUGINS_ODF_UITOOL_GLOBAL_VALIDATION_CHECK_START_ERROR}}",
                    buttons: Ext.Msg.OK,
                    icon: Ext.Msg.ERROR
                });
	        }
        }
    },
    
    
	constructor: function(config)
	{
		this.callParent(arguments);
		
		Ametys.message.MessageBus.on(Ametys.message.Message.GLOBAL_VALIDATION_CHECK_STARTED, this._onCheckMsg, this);
        Ametys.message.MessageBus.on(Ametys.message.Message.GLOBAL_VALIDATION_CHECK_ENDED, this._onCheckMsg, this);
	},

	/**
	 * Listener when a validation check message has been fired on the bus.
	 * Will refresh the button state.
	 * @param {Ametys.message.Message} message The message
	 * @protected
	 */
	_onCheckMsg: function (message)
	{
		this.refresh();
	},
	
	/**
	 * @inheritdoc
	 */
	updateState: function()
	{
		this.disable();
		this.setAdditionalDescription('');
        this.refreshing();
        
        Ametys.data.ServerComm.callMethod({
            role: "org.ametys.odf.schedulable.GlobalValidationReport",
            methodName: "isRunning",
            callback: {
                handler: this._updateStateCb,
                scope: this
            },
            errorMessage: false
        });
	},
	
	/**
	 * The callback for the server call made in the #updateState methid.
	 * @param {Object} params The returned value of the java call
	 * @param {Boolean} params.engine-running Indicate if the process is running on the server.
	 */
	_updateStateCb: function(running)
	{
        this.stopRefreshing();
        
		if (running)
		{
			this.disable();
			this.toggle(true);
			this.setAdditionalDescription(this.getInitialConfig('disabled-description'));
		}
		else
		{
			this.enable();
			this.toggle(false);
			this.setAdditionalDescription('');
		}
	}
});
